#!/bin/bash

# シェルスクリプトの if 文の判定に合わせて 0 を true にする
clpcloudutil_true=0
clpcloudutil_false=1

# 処理を中断
clpcloudutil_break=2


clpcloudutil_set_aws_env()
{
	local logflag=$1
	local r
	local line
	local ids
	local id
	local name
	local value

	ids=(`clpcfget -e "/root/cluster/cloud/aws/envs/env"`)
	r=$?

	if [ "$r" -eq 11 ]
	then
		if [ "$logflag" -eq "$clpcloudutil_true" ]
		then
			# 戻り値 11 の not exit xmlpath エラーは
			# 設定によっては常時発生するためログ出力フラグの制御対象にする
			echo "cmd: clpcfget -e /root/cluster/cloud/aws/envs/env = $r" 1>&2
			echo "msg: ${ids[@]}" 1>&2
		fi
		return $clpcloudutil_false
	elif [ "$r" -ne 0 ]
	then
		echo "cmd: clpcfget -e /root/cluster/cloud/aws/envs/env = $r" 1>&2
		echo "msg: ${ids[@]}" 1>&2
		return $clpcloudutil_break
	fi

	for id in "${ids[@]}"
	do
		name=`clpcfget -g "/root/cluster/cloud/aws/envs/env@$id/name"`
		r=$?

		if [ "$r" -ne 0 ]
		then
			echo "env$id: cmd: clpcfget -e /root/cluster/cloud/aws/envs/env@$id/name = $r" 1>&2
			echo "env$id: msg: $name" 1>&2
			return $clpcloudutil_break
		fi

		value=`clpcfget -g "/root/cluster/cloud/aws/envs/env@$id/value"`
		r=$?

		if [ "$r" -ne 0 ]
		then
			echo "env$id: cmd: clpcfget -e /root/cluster/cloud/aws/envs/env@$id/value = $r" 1>&2
			echo "env$id: msg: $value" 1>&2
			return $clpcloudutil_break
		fi

		if [ "$logflag" -eq "$clpcloudutil_true" ]
		then
			echo "env$id: set: '$name' = '$value'" 1>&2
		fi

		if [ "$name" = "" ]
		then
			echo "env$id: empty name" 1>&2
			continue
		fi

		export "$name"="$value"
	done

	return $clpcloudutil_true
}

clpcloudutil_line_clpaws_setting()
{
	local logflag=$1
	local line=$2

	# コメント除去
	line="`echo "${line%%#*}" | sed -e "s/^[ \t]*\|[ \t]*$//"`"

	# 設定情報読み込み
	name="`echo "${line%%=*}" | sed -e "s/[ \t]*$//"`"
	value="`echo "${line#*=}" | sed -e "s/^[ \t]*//"`"

	if [ "$line" = "$name" -o "$line" = "$value" ]
	then
		# = で分割できなかった場合
		return
	fi

	if [ "$logflag" -eq "$clpcloudutil_true" ]
	then
		echo "clpaws_setting: set: '$name' = '$value'" 1>&2
	fi

	if [ "$name" = "" ]
	then
		echo "clpaws_setting: empty name" 1>&2
		return
	fi

	export "$name"="$value"
}

clpcloudutil_load_clpaws_setting()
{
	local logflag=$1
	local filepath=$2
	local r

	if [ ! -f "$filepath" ]
	then
		if [ "$logflag" -eq "$clpcloudutil_true" ]
		then
			# 設定によっては常時発生するためログ出力フラグの制御対象にする
			echo "not found: $filepath" 1>&2
		fi
		return $clpcloudutil_false
	fi

	if [ "$logflag" -eq "$clpcloudutil_true" ]
	then
		echo "load: $filepath" 1>&2
	fi

	while read line
	do
		clpcloudutil_line_clpaws_setting "$logflag" "$line"
	done < "$filepath"

	if [ -n "$line" ]
	then
		# ファイル末尾が改行以外の場合
		clpcloudutil_line_clpaws_setting "$logflag" "$line"
	fi

	return $clpcloudutil_true
}

clpcloudutil_env_init()
{
	local logflag=$1
	local filepath=$2

	clpcloudutil_set_aws_env "$logflag"
	r=$?
	if [ "$r" -eq "$clpcloudutil_break" ]
	then
		echo "failure: clpcloudutil_set_aws_env()" 1>&2
		return $clpcloudutil_false
	elif [ "$r" -eq "$clpcloudutil_false" ]
	then
		echo "failure: clpcloudutil_set_aws_env()" 1>&2
	else
		return $clpcloudutil_true
	fi

	if ! clpcloudutil_load_clpaws_setting "$logflag" "$filepath"
	then
		echo "failure: clpcloudutil_load_clpaws_setting()" 1>&2
		return $clpcloudutil_false
	fi

	return $clpcloudutil_true
}

clpcloudutil_awscli_cmdopt()
{
	local logflag=$1
	local awssvc=$2
	local r
	local awscmdopt

	awscmdopt=`clpcfget -g "/root/cluster/cloud/aws/cmdopt/${awssvc}" -p cluster`
	r=$?

	if [ "$r" -ne 0 ]
	then
		echo "awscli_cmdopt: cmd: clpcfget -g /root/cluster/cloud/aws/cmdopt/${awssvc} -p cluster = $r" 1>&2
		echo "awscli_cmdopt: msg: $awscmdopt" 1>&2
		return $clpcloudutil_false
	fi

	if [ -z "${awscmdopt}" ]
	then
		return $clpcloudutil_true
	fi

	if [[ "$awscmdopt" =~ ";" ]] || [[ "$awscmdopt" =~ "&&" ]] || [[ "$awscmdopt" =~ "||" ]] || [[ "$awscmdopt" =~ "\`" ]]
	then
		echo "awscli_cmdopt: contains the special characters" 1>&2
		echo "awscli_cmdopt: svcname=${awssvc} cmdopt=\"${awscmdopt}\"" 1>&2
		return $clpcloudutil_true
	fi

	if [[ "$awscmdopt" =~ "--output" ]]
	then
		echo "awscli_cmdopt: contains the special output option" 1>&2
		echo "awscli_cmdopt: svcname=${awssvc} cmdopt=\"${awscmdopt}\"" 1>&2
		return $clpcloudutil_true
	fi

	if [ "$logflag" -eq "$clpcloudutil_true" ]
	then
		echo "awscli_cmdopt: svcname=${awssvc} cmdopt=\"${awscmdopt}\"" 1>&2
	fi

	echo $awscmdopt
	return $clpcloudutil_true
}


#
# 先頭、末尾の不要な空白・改行を除去する
#
function TrimWhiteSpace {
	cat | sed -e '1{/^$/d;}' -e '${/^$/d;}' \
		| sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//'
}


#
# 指定したパターンに一致する文字列を抽出する
#
function ExtractPtn {
	local message=$1
	local regExp=$2

	if [[ $message =~ $regExp ]]; then
		cause=$(echo "${BASH_REMATCH[1]}" | TrimWhiteSpace)
	else
		echo "Couldn't extract the cause of the CLI error." >&2
		return 1
	fi
	echo "$cause"
}


#
# 先頭から指定数文字列を抽出する
#
function Trancate {
	local message=$1
	local maxTrancateSize=$2

	if [[ ${#message} -gt $maxTrancateSize ]]; then
		message=${message:0:$maxTrancateSize}
	fi
	echo "$message"
}


#
# AWS CLIエラー原因を抽出する
#
function ExtractAWSErrorCause {
	local message=$1
	local maxTrancateSize=$2
	local regExp="A.* error .* operation: (.+)"
	local removeStr="Encoded authorization failure message:*"
	cause=$(ExtractPtn "$message" "$regExp") || return 1
	cause=$(echo "${cause/$removeStr/}" | TrimWhiteSpace)
	Trancate "$cause" "$maxTrancateSize"
}


#
# Azure CLIエラー原因を抽出する
#
function ExtractAzureErrorCause {
	local message=$1
	local maxTrancateSize=$2
	local regExp="ERROR: \((.+?)\)"
	cause=$(ExtractPtn "$message" "$regExp") || return 1
	Trancate "$cause" "$maxTrancateSize"
}


#
# GCP CLIエラー原因を抽出する
#
function ExtractGCPErrorCause {
	local message=$1
	local maxTrancateSize=$2
	local regExp="ERROR: \(.+?\)(.+)$"
	cause=$(ExtractPtn "$message" "$regExp") || return 1
	Trancate "$cause" "$maxTrancateSize" | head -n 1
}


#
# OCI CLIエラー原因を抽出する
#
function ExtractOCIErrorCause {
	local message=$1
	local maxTrancateSize=$2
	local regExp="\"code\": \"([^\"]+)"
	cause=$(ExtractPtn "$message" "$regExp") || return 1
	Trancate "$cause" "$maxTrancateSize"
}
