#!/bin/sh
# NAME
#  clporacle.sh
#
# DESCRIPTOIN
#  Get current status/information of oracle DB and oracle monitor
#
# MODIFIED (YYYY/MM/DD)
#  hemu@necsthz / jianxiaoyun@necsthz 2009/12/24
#  jianxiaoyun@necsthz 2010/03/09
#  jianxiaoyun@necsthz 2010/04/14

# USAGE:
# clporacle.sh --listener --error <ERRNO> --path <DIR>
# clporacle.sh --listener --sid <SID> --error <ERRNO> --path <DIR>
# clporacle.sh --listener --sid <SID> --kill <PID> --path <DIR>

# set the user and password to connect to oracle server
# password needs not to be set if oracle is in local server.
DBAUSER=sys
#PASSWORD=

# set oracle user in the OS. default user is oracle
OSORAUSER=oracle

# it is recommended to set LISTENER name if have multiple listeners
# and the target db does not use the default one. 
#LISTENER=

# if ERR_IGNORE is set, it will do nothing when the error(s) occured
# more than one error can be set by separating them with space
#ERR_IGNORE="01034 12541"

# ORACONF/ORALOG: set the files to be collected
ORACONF="listener.ora tnsnames.ora sqlnet.ora"
ORALOG="listener.log sqlnet.log"

# Collect times
NCOLLECT=3
# Collect interval in seconds
INTERVAL=180

# dump systemstate level (not essential). 
#SYSSTATLVL=

# enable to attach client process to excute oradebug. 
# this may induce some obstacles to client's business.
# 9i or former versions do not have MMON process.
# so enable it with your own risk. 
#ENABLE_ATTACH_CLI=0

#####!!! user needs not modify the text below !!!#####

HelpMsg()
{
	echo "Invalid parameters." >> $INFOLOG
	echo "USAGE:"
	echo "    clporacle.sh --listener --error <ERRNO> --path <DIR>"
	echo "    clporacle.sh --listener --sid <SID> --error <ERRNO> --path <DIR>"
	echo "    clporacle.sh --listener --sid <SID> --kill <PID> --path <DIR>"
	echo "COMMENT:"
	echo "    --listener: collect listener infomation"
	echo "    --error   : the error number detected by oraclew"
	echo "    --path    : the path to save collected files"
	echo "    --sid     : oracle System IDentifier"
	echo "    --kill    : kill sessions of the specified PID"
}

# init variables
CHECKLSNR=0
PARAERR=0
MONTYPE=oraclew
ERRSQL=${MONTYPE}_err.sql # execute when error occured
HANGSQL=${MONTYPE}_hang.sql # execute when timeout occured

# set path for solaris
if [ -d /usr/xpg4/bin ]
then
	PATH=/usr/xpg4/bin:$PATH
fi

OPTS=$*
# check param
while [ $# != 0 ]
do
	case "$1" in
	"--mon") # ignore
		if echo "${2};" |egrep -v "^(-|;)"
		then
			shift
		fi
		;;  
	"--listener")
		if echo "${2};" |egrep -v "^-"
		then
			PARAERR=1
		fi
		CHECKLSNR=1
		;;  
	"--sid")
		if echo "${2};" |egrep "^(-|;)"
		then
			SID=
			PARAERR=1
		else
			SID=$2
			shift
		fi
		;;  
	"--kill")
		if echo "${2};" |egrep "^(-|;)"
		then
			PID=
			PARAERR=1
		else
			PID=$2
			shift
		fi
		;;  
	"--error")
		if echo "${2};" |egrep "^(-|;)"
		then
			ERRNO=
			PARAERR=1
		else
			ERRNO=$2
			shift
		fi
		;;  
	"--path")
		if echo "${2};" |egrep "^(-|;)"
		then
			ERRNO=
			PARAERR=1
		else
			LOG_DIR=$2
			shift
		fi
		;;  
	"--prefix")
		# prefix is only used for test.
		if echo "${2};" |egrep "^(-|;)"
		then
			PREFIX=
		else
			PREFIX=${2}_
			shift
		fi
		;;  
	*)  
		PARAERR=1
		;;  
	esac
	shift
done

# set log directory
if echo $LOG_DIR |egrep -v '^/' >/dev/null 2>&1
then
	LOG_DIR=`pwd`/$LOG_DIR
fi
if [ ! -d "${LOG_DIR}" ]
then
	LOG_DIR=`pwd`
fi

cd "`dirname $0`"

# clear log file
INFOLOG="${LOG_DIR}/${PREFIX}${MONTYPE}_info.log"
> "$INFOLOG"
echo "Execute [clporacle.sh $OPTS] at `date +'%Y%m%d %H:%M:%S'`" |tee -a "$INFOLOG"
echo "" |tee -a "$INFOLOG"

DIR1=`dirname $LOG_DIR`
DIR2=`dirname $DIR1`
chmod 755 $DIR2 $DIR1 $LOG_DIR

if [ $PARAERR = 1 ]
then
	HelpMsg
	exit 1
fi

# check user
if groups $OSORAUSER 2>/dev/null |egrep -v '[:space:]*dba[:space:]*' >/dev/null 2>&1; 
then
	if groups |egrep -v '[:space:]*dba[:space:]*' >/dev/null 2>&1;
	then
		echo "WARNING: Neither \"$OSORAUSER\" nor \"`whoami`\" belongs to dba group." |tee -a "$INFOLOG"
	fi
fi
id $OSORAUSER >/dev/null 2>&1 || OSORAUSER=`whoami`

# ignore error
for err in $ERR_IGNORE
do
	if [ x"$err" = x"$ERRNO" -a x"$ERRNO" != x ]
	then
		echo "The error \"$err\" will be ignored." |tee -a "$INFOLOG"
		exit 0
	fi
done


# get ORACLE_HOME
BASESID=`basename $SID` # when SID=<IP>/<BASESID>
if [ "$BASESID" = "$SID" ]
then # can not get remote ORACLE_HOME from local host
	if [ x"$ORACLE_HOME" = x ]
	then
		if [ -f /etc/oratab ]; then
			ORATAB="/etc/oratab";
		elif [ -f /var/opt/oracle/oratab ]; then
			ORATAB="/var/opt/oracle/oratab" ;
		fi
		if [ -f "$ORATAB" ]; then
			ORACLE_HOME=`awk -F':' '/^[[:space:]]*'"$SID"':/,/$/{print $2}' $ORATAB| tail -1`
		else
			echo "WARNING: Unable to locate oratab file" |tee -a "$INFOLOG"
		fi
	fi
	if [ x"$ORACLE_HOME" = x ]
	then
		ORACLE_HOME=`su - $OSORAUSER -c 'echo $ORACLE_HOME'`
	fi
fi
echo "ORACLE_HOME=$ORACLE_HOME" |tee -a "$INFOLOG"

# get oracle version and set systemstate level
[ x"$SYSSTATLVL" = x ] && SYSSTATLVL=10
if [ -d "$ORACLE_HOME" -a -f "$ORACLE_HOME/inventory/ContentsXML/comps.xml" ]
then
	ORAVERSTR=`egrep '\<(COMP|PATCHSET)[[:space:]]+NAME=\"oracle.(patchset.db|server)\"' "$ORACLE_HOME/inventory/ContentsXML/comps.xml"`
	TOKEN='\<PATCHSET.*NAME=\"oracle.patchset.db\"'
	echo $ORAVERSTR |egrep $TOKEN || TOKEN='\<PATCHSET.*NAME=\"oracle.server\"'
	echo $ORAVERSTR |egrep $TOKEN || TOKEN='\<COMP.*NAME=\"oracle.server\"'
	ORAVER=`awk '/'"$TOKEN"'/{i=1;while(match($i,"VER=")!=1)i++;sub(/VER="/,"",$i);sub(/"/,"",$i);print$i}' \
		"$ORACLE_HOME/inventory/ContentsXML/comps.xml" \
		|awk -F'.' '{print $1*1000000+$2*10000+$3*100+$4}'`
	if [ x$ORAVER != x ]
	then
		SYSSTATLVL=266
		if [ $ORAVER -lt 9020007 ]
		then # < 9.2.0.7
			SYSSTATLVL=10
		elif [ $ORAVER -lt 10010004 -a $ORAVER -gt 10000000 ]
		then # 10g and < 10.1.0.4
			SYSSTATLVL=10
		fi
	fi
fi
echo "ORAVER=$ORAVER" |tee -a "$INFOLOG"

# collect listener info 
if [ $CHECKLSNR = 1 ]
then
	echo "" |tee -a "$INFOLOG"
	echo "Get listener status" |tee -a "$INFOLOG"
	echo "----------------------------------" |tee -a "$INFOLOG"
	su - $OSORAUSER -c "lsnrctl status $LISTENER"  |tee -a "$INFOLOG"
	echo "----------------------------------" |tee -a "$INFOLOG"
	echo "" |tee -a "$INFOLOG"
	if [ -d "$ORACLE_HOME" ]
	then
		echo "Copy files" |tee -a "$INFOLOG"
		echo "----------------------------------" |tee -a "$INFOLOG"
		for FILE in $ORACONF
		do
			cp "$ORACLE_HOME/network/admin/$FILE" "$LOG_DIR/${PREFIX}${FILE}" |tee -a "$INFOLOG" 2>&1
		done
		if [ ! -f "$ORACLE_HOME/bin/adrci" ]
		then
			for FILE in $ORALOG
			do
				echo "copy \"$FILE\" (only the latest 1000L) to $LOG_DIR" |tee -a "$INFOLOG"
				tail -1000 "$ORACLE_HOME/network/log/$FILE" > "$LOG_DIR/${PREFIX}`basename ${FILE}`" 2>&1
			done
		else
			for ADRHOME in `su - $OSORAUSER -c "adrci exec=\"show home\" " | egrep -v '(\/clients\/|:)'`
			do
				echo "command adrci: get alert log (only the latest 1000L) to $LOG_DIR" |tee -a "$INFOLOG"
				su - $OSORAUSER -c "adrci exec=\"set home $ADRHOME; show alert -tail 1000\" " \
					> "$LOG_DIR/${PREFIX}alert_`basename ${ADRHOME}`.log" 2>&1
			done
		fi
		echo "----------------------------------" |tee -a "$INFOLOG"
		echo "" |tee -a "$INFOLOG"
	fi
	echo "" |tee -a "$INFOLOG"
fi

if [ -n "$SID" ]
then # current work directory is <CLP_INSTALLPATH>/bin
	ORACLE_SID=$SID
	[ x"$PID" = x ] && PID=:

	# attach oracle process to take dump
	if [ "$BASESID" = "$SID" ]
	then # oracle is in local server
		OSPID=`ps -eopid,args|egrep "ora_mmon_$SID$"|awk '{print $1}'`
		if [ x$OSPID = x ] # mmon process do not exist (9i or former version)
		then
			if [ x$ENABLE_ATTACH_CLI != x1 ] # attach to the lastet client process.
			then
				OSPID=`ps -eopid,args|egrep "ora.*$SID.*[\(]LOCAL="|awk '{print $1}'|tail -1`
			else
				echo "Can not find MMON process." |tee -a "$INFOLOG"
				exit 1
			fi
		fi
	fi
	i=1
	while [ $i -le $NCOLLECT ]
	do
		echo "" |tee -a "$INFOLOG"
		echo "Get instance infomation [$i]" |tee -a "$INFOLOG"
		echo "----------------------------------" |tee -a "$INFOLOG"
		if [ "$PID" != ":" -a -f "$HANGSQL" -a x"$OSPID" != x ] # timeout
		then
			SQLLOG=${PREFIX}${MONTYPE}_oradebug.$i.log
			touch "$LOG_DIR/$SQLLOG"
			chmod 666 "$LOG_DIR/$SQLLOG"
			echo "sqlplus /nolog @$HANGSQL $SQLLOG $OSPID $SYSSTATLVL" >> "$INFOLOG"
			su - $OSORAUSER -c "cd $LOG_DIR; sqlplus -L /nolog @\"`pwd`/$HANGSQL\" $SQLLOG $OSPID $SYSSTATLVL" |tee -a "$INFOLOG" 2>&1
			[ x"$TRACEFILE" = x ] && TRACEFILE=`sed -n '/oradebug tracefile_name/{n;p;}' "$LOG_DIR/$SQLLOG" |grep '^/'`
		fi
		if [ ! -f "$ERRSQL" ]
		then
			echo "Can not locate $ERRSQL" |tee -a "$INFOLOG"
			exit 1
		fi
		SQLLOG=${PREFIX}${MONTYPE}_sql.$i.xls
		touch "$LOG_DIR/$SQLLOG"
		chmod 666 "$LOG_DIR/$SQLLOG"
		j=`expr $i - 1`
		[ $i = $NCOLLECT ] && j=-1 # -1: end flag
		#  [select p.spid,process from v$process p,v$session s where s.paddr=p.addr;] to find PID for test
		if [ x"$PASSWORD" = x ]
		then
			echo "sqlplus \"/ as sysdba\" @$ERRSQL $j $SQLLOG $PID" >> "$INFOLOG"
			su - $OSORAUSER -c "cd $LOG_DIR; sqlplus -L \"/ as sysdba\" @\"`pwd`/$ERRSQL\" $j $SQLLOG $PID" |tee -a "$INFOLOG" 2>&1
		else
			echo "sqlplus \"$DBAUSER/<PASSWORD>@$SID as sysdba\" @$ERRSQL $SQLLOG $PID" >> "$INFOLOG"
			su - $OSORAUSER -c "cd $LOG_DIR; sqlplus -L \"$DBAUSER/$PASSWORD@$SID as sysdba\" @\"`pwd`/$ERRSQL\" $j $SQLLOG $PID" |tee -a "$INFOLOG" 2>&1
		fi
		if [ x"$OSPID" = x ]
		then # if SID name is different from dbname, it is possiable that fails to get OSPID by SID.
			DBNAME=`grep @@udump= "$LOG_DIR/${PREFIX}${MONTYPE}_sql.1.xls" |awk -F'/' '{print $(NF-1)}'`
			if [ x$DBNAME != x ]
			then
				OSPID=`ps -eopid,args|egrep "ora.*$DBNAME.*[\(]LOCAL="|awk 'END{print $1}'`
			fi
		fi
		i=`expr $i + 1`
		echo "----------------------------------" |tee -a "$INFOLOG"
		if [ $i -lt $NCOLLECT ]
		then
			echo "SLEEP for ${INTERVAL}s..." |tee -a "$INFOLOG"
			sleep $INTERVAL
		fi
	done

	echo "" |tee -a "$INFOLOG"
	echo "Collect trace files" |tee -a "$INFOLOG"
	echo "----------------------------------" |tee -a "$INFOLOG"
	SPID=`grep @@spid= "$LOG_DIR/${PREFIX}${MONTYPE}_sql.1.xls" |awk -F'=' '{print $2}'`
	UDUMP=`grep @@udump= "$LOG_DIR/${PREFIX}${MONTYPE}_sql.1.xls" |awk -F'=' '{print $2}'`
	BDUMP=`grep @@bdump= "$LOG_DIR/${PREFIX}${MONTYPE}_sql.1.xls" |awk -F'=' '{print $2}'`
	echo "SPID:$SPID" |tee -a "$INFOLOG"
	echo "UDUMP:$UDUMP" |tee -a "$INFOLOG"
	echo "BDUMP:$BDUMP" |tee -a "$INFOLOG"
	echo "OSPID:$OSPID" |tee -a "$INFOLOG"
	echo "TRACEFILE:$TRACEFILE" |tee -a "$INFOLOG"
	echo "" |tee -a "$INFOLOG"
	if [ ! -d "$ORACLE_HOME" ]
	then
		echo "Please collect the files manually." |tee -a "$INFOLOG"
		echo "----------------------------------" |tee -a "$INFOLOG"
		exit 0
	fi
	if [ -f "$TRACEFILE" ]
	then
		echo "Copy \"$TRACEFILE\" to $LOG_DIR" |tee -a "$INFOLOG"
		cp "$TRACEFILE" "$LOG_DIR/$PREFIX`basename $TRACEFILE`" |tee -a "$INFOLOG" 2>&1
	fi
	if [ x"$SPID" = x -o ! -d "$UDUMP" -o ! -d "$BDUMP" ];
	then
		echo "Can not locate trace files." |tee -a "$INFOLOG"
		echo "----------------------------------" |tee -a "$INFOLOG"
		exit 1
	fi
	echo "Copy trace files to $LOG_DIR" |tee -a "$INFOLOG"
	for FILE in "$UDUMP"/*${SPID}.* "$BDUMP"/*${SPID}.*
	do
		[ -f $FILE -a "$FILE" != "$TRACEFILE" ] && \
		cp "$FILE" "$LOG_DIR/$PREFIX`basename $FILE`" |tee -a "$INFOLOG" 2>&1
	done
	echo "----------------------------------" |tee -a "$INFOLOG"
fi

echo "" |tee -a "$INFOLOG"
echo "Execute clporacle.sh end at `date +'%Y%m%d %H:%M:%S'`" |tee -a "$INFOLOG"

