exports.init = log_init;

const fs = require('fs');
const path = require('path');
const util = require('util');
const debuglog = util.debuglog('clplogger');

const lvnum = {
	emerg: 0,
	alrt: 1,
	crit: 2,
	error: 3,
	warn: 4,
	notice: 5,
	info: 6,
	debug: 7
};

function log_init(logname, logdir, loglevel, logsize)
{
	const handle = {
		name: logname,
		dir: logdir,
		level: loglevel,
		size: logsize,
		promise: Promise.resolve(),
		termed: false
	};

	if (typeof handle.name !== 'string') {
		debuglog('init: invalid log name: %s', handle.name);
		handle.name = 'logger';
	}
	if (typeof handle.dir !== 'string') {
		debuglog('init: invalid log directory: %s', handle.dir);
		handle.dir = '.';
	}
	if (typeof handle.level !== 'string' || !lvnum[handle.level]) {
		debuglog('init: invalid log level: %s', handle.level);
		handle.level = 'debug';
	}
	if (typeof handle.size !== 'number' || handle.size !== handle.size /* NaN */ || handle.size <= 0) {
		debuglog('init: invalid log size: %s', handle.size);
		handle.size = 1000000;
	}
	debuglog('init: name=%s, dir=%s, level=%s, size=%s',
			handle.name, handle.dir, handle.level, handle.size);

	handle.write = (level, obj) => {
		if (!lvnum[level]) {
			debuglog('write: undefined log level: %s', level);
		} else if (lvnum[handle.level] < lvnum[level]) {
			return handle.promise;
		}

		let message;
		const time = log_time();
		const fileline = log_fileline();
		if (typeof obj == 'object' || typeof obj == 'function') {
			message = util.format('%s <%s> [%s]\n%O\n', time, level, fileline, obj);
		} else {
			message = util.format('%s <%s> %s [%s]\n', time, level, obj, fileline);
		}

		return log_write(handle, message);
	}

	handle.write.term = () => {
		handle.promise = handle.write('debug', '**** term ****').then(() => {
			handle.termed = true;
		});

		return handle.promise;
	}

	handle.write('debug', '**** init ****');
	return handle.write;
}

function log_write(handle, message)
{
	const curpath = handle.dir + '/' + handle.name + '.log';
	const prepath = handle.dir + '/' + handle.name + '.pre.log';

	handle.promise = handle.promise.then(() => {
		return new Promise((resolve, reject) => {
			debuglog('write: lstat(%s)', curpath);
			fs.lstat(curpath, (err, stats) => {
				if (err)
					reject(err);
				else if (stats.size < handle.size)
					reject();
				else
					resolve();
			});
		});
	}).then(() => {
		return new Promise((resolve, reject) => {
			debuglog('write: unlink(%s)', prepath);
			fs.unlink(prepath, (err) => { err ? reject(err) : resolve(); });
		}).catch((err) => {
			if (err) debuglog('write: <error>\n%O', err);
		});
	}).then(() => {
		return new Promise((resolve, reject) => {
			debuglog('write: rename(%s => %s)', curpath, prepath);
			fs.rename(curpath, prepath, (err) => { err ? reject(err) : resolve(); });
		});
	}).catch((err) => {
		if (err) debuglog('write: <error>\n%O', err);
	}).then(() => {
		if (handle.termed) {
			debuglog('write: logger was terminated');
			return;
		}
		return new Promise((resolve, reject) => {
			debuglog('write: appendFile(%s)', curpath);
			fs.appendFile(curpath, message, (err) => { err ? reject(err) : resolve(); });
		});
	}).catch((err) => {
		debuglog('write: <error>\n%O', err);
	});

	return handle.promise;
}

function log_fileline() {
	const limit = Error.stackTraceLimit;
	const prepare = Error.prepareStackTrace;
	Error.stackTraceLimit = 2;
	Error.prepareStackTrace = (_, stack) => {
		return path.basename('' + stack[1].getFileName())
			+ ':' + stack[1].getLineNumber() + ':' + stack[1].getColumnNumber();
	};

	const error = {};
	Error.captureStackTrace(error, log_fileline);
	const fileline = error.stack;

	Error.prepareStackTrace = prepare;
	Error.stackTraceLimit = limit;

	return fileline;
}

function log_time()
{
	const date = new Date();
	return '' + date.getFullYear() + '-'
		+ ('0' + (date.getMonth() + 1)).slice(-2) + '-'
		+ ('0' + date.getDate()).slice(-2) + ' '
		+ ('0' + date.getHours()).slice(-2) + ':'
		+ ('0' + date.getMinutes()).slice(-2) + ':'
		+ ('0' + date.getSeconds()).slice(-2) + '.'
		+ ('00' + date.getMilliseconds()).slice(-3);
}
