#!/bin/bash

# 戻り値
readonly CMD_SUCCESS=0          # 成功
readonly CMD_FAILED=1           # 失敗
readonly CMD_ERR_GET_VERSION=2  # カーネルバージョンの取得に失敗
readonly CMD_ERR_GET_PREFIX=3   # prefixの取得に失敗
readonly CMD_ERR_GET_SUFFIX=4   # suffixの取得に失敗
readonly CMD_ERR_NOTFOUND=5     # コピーするドライバが存在しない
readonly CMD_ERR_OS=6           # 対応OSでない
readonly CMD_ERR_ARCHITECTURE=7 # 対応アーキテクチャでない
readonly CMD_ERR_INTERNAL=8     # その他エラー
readonly CMD_ERR_COPY=9         # コピー失敗
readonly CMD_ERR_LOG_INIT=10    # logの初期化に失敗
readonly CMD_LOAD_FAILED=11     # ロード失敗

readonly VERSION_LESS=20        # バージョンが古い
readonly VERSION_GREATER=21     # バージョンが新しい
readonly VERSION_EQUAL=22       # バージョンが同じ

# static variables
readonly FILE_SIZE=1048576 # 1MB
readonly CMD_NAME="drvcp"
readonly SCR_NAME="clp${CMD_NAME}.sh"

# xmlpath
readonly XML_PATH_KHBMAJOR="/root/lankhb/khb/major"
readonly XML_PATH_KHBMINOR="/root/lankhb/khb/minor"
readonly XML_PATH_KAMAJOR="/root/cluster/driver/ka/major"
readonly XML_PATH_KAMINOR="/root/cluster/driver/ka/minor"
readonly XML_PATH_LISCALMAJOR="/root/mddriver/major"
readonly XML_PATH_EDITION="/root/all/edition"

# 環境変数
PATH=/sbin:/bin:/usr/sbin:/usr/bin:/opt/nec/clusterpro/bin

# functions
# ログ関数
log_start() {
    local logpath=""
    local ret_logcf=0

    #------- search log dir -------
    logpath=$(clplogcf -p)
    ret_logcf=$?
    if [[ ${ret_logcf} -ne 0 ]]; then
        logpath="/opt/nec/clusterpro/log/"
    fi
    if [[ -d "${logpath}" ]]; then
        log_cur="${logpath}${CMD_NAME}.log.cur"
        log_pre="${logpath}${CMD_NAME}.log.pre"
    else
        echo "${SCR_NAME} error : Log directory is not found." 1>&2
        return ${CMD_ERR_LOG_INIT}
    fi

    #------- save previous log file -------
    if [[ -e "${log_cur}" && $(stat -c%s "${log_cur}") -ge FILE_SIZE ]]; then
        mv -f "${log_cur}" "${log_pre}" > /dev/null 2>&1
    fi

    echo "$(date '+%Y/%m/%d %H:%M:%S') ----------${SCR_NAME}  START.----------" >> "${log_cur}"
}

log_write() {
    local msg="$1"
    echo "$(date '+%Y/%m/%d %H:%M:%S') ${msg}" >> "${log_cur}"
}

log_end() {
    local ret_logend="$1"
    echo "$(date '+%Y/%m/%d %H:%M:%S') ----------${SCR_NAME}  END.------------" >> "${log_cur}"
    return "${ret_logend}"
}

# バージョン番号を"."と"-"で分割する関数
split_version() {
    IFS='.-' read -ra parts <<< "$1"
    for part in "${parts[@]}"; do
        echo "${part}"
    done
}

# バージョンを比較する関数
compare_version() {
    version1="$1"
    version2="$2"

    readarray -t version1_parts < <(split_version "${version1}")
    readarray -t version2_parts < <(split_version "${version2}")

    # バージョン番号の各部分を数値として比較
    len1=${#version1_parts[@]}
    len2=${#version2_parts[@]}
    max_len=$((len1>len2 ? len1 : len2))
    for ((i=0; i<max_len; i++)); do
        part1=${version1_parts[$i]:-0}
        part2=${version2_parts[$i]:-0}
        if (( part1 < part2 )); then
            return ${VERSION_LESS}
        elif (( part1 > part2 )); then
            return ${VERSION_GREATER}
        fi
    done
    return ${VERSION_EQUAL}
}

# コピーするドライバを探す関数
search_driver(){
    local kernel_version="$1"
    local pre="$2"
    local suf="$3"
    drvfile=''
    tmpfile=''

    for file in "${drv_dir}"/*.ko; do
        filename=$(basename "${file}")
        filename_version=${filename#*-}
        filename_version=${filename_version%.*.*.ko}

        # prefix, suffixで絞る
        if [[ ${filename} == *-"${pre}"* ]] && [[ ${filename} == *"${suf}".ko ]]; then
            # バージョン番号を取得
            compare_version "${filename_version}" "${kernel_version}"
            local ret=$?
            if [[ ${ret} -ne ${VERSION_GREATER} ]]; then
                log_write "found:${filename}"
                # すでに見つけているものよりも新しい
                if [[ -n ${drvfile} ]]; then
                    drvfile_version=${drvfile#*-}
                    drvfile_version=${drvfile_version%"${suf}".ko}
                    compare_version "${filename_version}" "${drvfile_version}"
                    ret=$?
                    if [[ ${ret} -eq ${VERSION_GREATER} ]]; then
                        drvfile=${filename}
                        log_write "get:${filename}"
                    else
                        log_write "skip:${filename}"
                    fi
                else
                    drvfile=${filename}
                    log_write "get:${filename}"
                fi
            else
                # ベースのドライバを覚えておく
                dot_count=$(awk -F'.' '{print NF-1}' <<< "${filename}")
                if [[ ${dot_count} -lt ${ndot} ]]; then
                    log_write "found base:${filename}"
                    # ベースのドライバは最新になる
                    if [[ -n "${tmpfile}" ]]; then
                        tmpfile_version=${tmpfile#*-}
                        tmpfile_version=${tmpfile_version%.*.*.ko}
                        compare_version "${filename_version}" "${tmpfile_version}"
                        ret=$?
                        if [[ ${ret} -eq ${VERSION_GREATER} ]]; then
                            tmpfile=${filename}
                            log_write "get base:${filename}"
                        else
                            log_write "skip base:${filename}"
                        fi
                    else
                        tmpfile=${filename}
                        log_write "get base:${filename}"
                    fi
                fi
            fi
        fi
    done
    # 条件に一致するファイル(drvfile)がない場合は最新のベースカーネルドライバを適用
    if [[ -z "${drvfile}" ]] && [[ -n "${tmpfile}" ]]; then
        drvfile="${tmpfile}"
    elif [[ -z "${drvfile}" ]] && [[ -z "${tmpfile}" ]]; then
        log_write "not found driver."
        return ${CMD_ERR_NOTFOUND}
    fi
    if [[ -n "${drvfile}" ]]; then
        log_write "found driver:${drvfile}"
        return ${CMD_SUCCESS}
    fi
}

# ドライバをコピーする関数
drvcp_exec_cp () {
    local drvpath="$1"

    # prefixとsuffixを取得
    # RHEL
    if [[ ${kernel} =~ \.el[0-9] ]]; then
        log_write "OS is RHEL."
        local OS="RHEL"
        # ドット3つ目までをprefix 
        pre=$(echo "${kernel}" | cut -d'.' -f1-3)
        if [[ -z "${pre}" ]]; then
            log_write "invalid version name.(${kernel})"
            return ${CMD_ERR_GET_PREFIX}
        fi
        log_write "prefix is ${pre}"
        # 最後からドット2つ目までをsuffix
        suf=$(echo "${kernel}" | rev | cut -d"." -f1-2 | rev)
        if [[ -z "${suf}" ]]; then
            log_write "invalid version name.(${kernel})"
            return ${CMD_ERR_GET_SUFFIX}
        fi
        log_write "suffix is ${suf}"
    # Amazon Linux 2023
    elif [[ ${kernel} =~ \.amzn2023 ]]; then
        log_write "OS is Amazon Linux 2023."
        local OS="Amazon Linux 2023"
        # ドット2つ目までをprefix
        pre=$(echo "${kernel}" | cut -d'.' -f1-2)
        if [[ -z "${pre}" ]]; then
            log_write "invalid version name.(${kernel})"
            return ${CMD_ERR_GET_PREFIX}
        fi
        log_write "prefix is ${pre}"
        # 最後からドット2つ目までをsuffix
        suf=$(echo "${kernel}" | rev | cut -d"." -f1-2 | rev)
        if [[ -z "${suf}" ]]; then
            log_write "invalid version name.(${kernel})"
            return ${CMD_ERR_GET_SUFFIX}
        fi
        log_write "suffix is ${suf}"
    # RHELでもAmazon Linux 2023でもない場合
    else
        log_write "unsupported OS.(${kernel})"
        return ${CMD_ERR_OS}
    fi

    # suffixのアーキテクチャを確認
    if [[ ${suf#*.} == x86_64 || ${suf#*.} == aarch64 ]]; then
        log_write "architecture is ok"
    else
        log_write "unsupported architecture.(suffix=${suf})"
        return ${CMD_ERR_ARCHITECTURE}
    fi

    # ドライバのディレクトリを取得
    drv_dir=${drvpath%/*}
    if [[ ! -d "${drv_dir}" ]]; then
        log_write "opendir(${drv_dir}) failed."
        return ${CMD_ERR_INTERNAL}
    fi

    # ドット数のカウント
    kernelmodule=$(basename "${drvpath}")
    if [[ -z "${kernelmodule}" ]]; then
        log_write "invalid driver name.(path:${drvpath})"
        return ${CMD_ERR_INTERNAL}
    fi
    ndot=$(awk -F'.' '{print NF-1}' <<< "${kernelmodule}")
    if [[ -z "${ndot}" ]]; then
        log_write "invalid kernel name.(${kernelmodule})"
        return ${CMD_ERR_INTERNAL}
    fi

    # kernelのバージョン番号を取得
    kernel_version=${kernelmodule#*-}
    kernel_version=${kernel_version%."${suf}".ko}
    if [[ -z "${kernel_version}" ]]; then
        log_write "invalid kernel version.(${kernel_version})"
        return ${CMD_ERR_GET_VERSION}
    fi

    # ディレクトリからコピーするドライバを探す
    search_driver "${kernel_version}" "${pre}" "${suf}"
    local ret=$?

    if [[ ${ret} -eq ${CMD_ERR_NOTFOUND} && ${OS} == "RHEL" ]]; then
        # suffix 文字列内の _X 部分を削除 (ex. "el8_2.x86_64")
        if [[ ${suf} =~ ([^_]*)(_[^.]*)(\..*) ]]; then
            suftmp=${BASH_REMATCH[1]}${BASH_REMATCH[3]}
            # (el8.x86_64)
            suf=${suftmp}
            log_write "suffix is ${suf}"
            log_write "search driver again."
            search_driver "${kernel_version}" "${pre}" "${suf}"      # 再度探索
            ret=$?
        fi
    fi
    if [[ ${ret} -eq ${CMD_SUCCESS} ]]; then
        # コピーを実行
        log_write "copy from ${drv_dir}/${drvfile} to ${drvpath}"
        /bin/cp -a "${drv_dir}/${drvfile}" "${drvpath}"
        ret=$?
        if [[ ${ret} -eq 0 ]]; then
            log_write "copy success."
            return ${CMD_SUCCESS}
        else
            log_write "copy failed."
            return ${CMD_ERR_COPY}
        fi
    else
        log_write "search driver failed."
        return ${CMD_ERR_NOTFOUND}
    fi
}

# ドライバをロードする関数
load_exec() {
    local drvtype="$1"
    local drvpath="$2"
    local driver_name
    local local err_cmd=""

    driver_name=$(basename "${drvpath%%-*}")
    
    case ${drvtype} in
        "khb")
            err_cmd=$(insmod "${drvpath}" g_major="${khbmajor}" g_minor="${khbminor}" 2>&1)
            ret=$?
            if [[ ${ret} -ne 0 ]]; then
                log_write "${err_cmd}"
            fi
            if [[ ${ret} -eq 0 ]]; then
                log_write "insmod clpkhb success."
                unload_khb_flag=1
                return ${CMD_SUCCESS}
            else
                log_write "insmod clpkhb failed.(drvpath=${drvpath})"
                err_cmd=$(rmmod "${driver_name}" 2>&1)
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                log_write "insmod clpkhb -f again"
                err_cmd=$(insmod "${drvpath}" g_major="${khbmajor}" g_minor="${khbminor}" -f 2>&1)
                ret=$?
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                if [[ ${ret} -eq 0 ]]; then
                    log_write "insmod clpkhb -f success."
                    unload_khb_flag=1
                    return ${CMD_SUCCESS}
                else
                    log_write "insmod clpkhb -f failed."
                    err_cmd=$(rmmod "${driver_name}" 2>&1)
                    if [[ ${ret} -ne 0 ]]; then
                        log_write "${err_cmd}"
                    fi
                    log_write "remove ${drvpath}"
                    rm -f "${drvpath}" > /dev/null 2>&1
                    return ${CMD_LOAD_FAILED}
                fi
            fi
            ;;
        "ka")
            err_cmd=$(insmod "${drvpath}" g_major="${kamajor}" g_minor="${kaminor}" 2>&1)
            ret=$?
            if [[ ${ret} -ne 0 ]]; then
                log_write "${err_cmd}"
            fi
            if [[ ${ret} -eq 0 ]]; then
                log_write "insmod clpka success."
                unload_ka_flag=1
                return ${CMD_SUCCESS}
            else
                log_write "insmod clpka failed.(drvpath=${drvpath})"
                err_cmd=$(rmmod "${driver_name}" 2>&1)
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                log_write "insmod clpka -f again"
                err_cmd=$(insmod "${drvpath}" g_major="${kamajor}" g_minor="${kaminor}" -f 2>&1)
                ret=$?
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                if [[ ${ret} -eq 0 ]]; then
                    log_write "insmod clpka -f success."
                    unload_ka_flag=1
                    return ${CMD_SUCCESS}
                else
                    log_write "insmod clpka -f failed."
                    err_cmd=$(rmmod "${driver_name}" 2>&1)
                    if [[ ${ret} -ne 0 ]]; then
                        log_write "${err_cmd}"
                    fi
                    log_write "remove ${drvpath}"
                    rm -f "${drvpath}" > /dev/null 2>&1
                    return ${CMD_LOAD_FAILED}
                fi
            fi
            ;;
        "md")
            err_cmd=$(insmod "${drvpath}" g_liscal_major="${liscalmajor}" nmp_minor="${liscalminor}" 2>&1)
            ret=$?
            if [[ ${ret} -ne 0 ]]; then
                log_write "${err_cmd}"
            fi
            if [[ ${ret} -eq 0 ]]; then
                log_write "insmod liscal success."
                unload_liscal_flag=1
                return ${CMD_SUCCESS}
            else
                log_write "insmod liscal failed.(drvpath=${drvpath})"
                err_cmd=$(rmmod "${driver_name}" 2>&1)
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                log_write "insmod liscal -f again"
                err_cmd=$(insmod "${drvpath}" g_liscal_major="${liscalmajor}" nmp_minor="${liscalminor}" -f 2>&1)
                ret=$?
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                if [[ ${ret} -eq 0 ]]; then
                    log_write "insmod liscal -f success."
                    unload_liscal_flag=1
                    return ${CMD_SUCCESS}
                else
                    log_write "insmod liscal -f failed."
                    err_cmd=$(rmmod "${driver_name}" 2>&1)
                if [[ ${ret} -ne 0 ]]; then
                    log_write "${err_cmd}"
                fi
                    log_write "remove ${drvpath}"
                    rm -f "${drvpath}" > /dev/null 2>&1
                    return ${CMD_LOAD_FAILED}
                fi
            fi
            ;;
        *)
            log_write "unsupported driver type.(${drvtype})"
            return ${CMD_LOAD_FAILED}
            ;;
    esac
}

# main function to copy and load driver
drvcp_main_cp() {
    local drvtype="$1"
    local drvname="$2"
    local inspath="/opt/nec/clusterpro"
    local drvpath="${inspath}/drivers/${drvtype}/distribution/current/${drvname}-${kernel}.ko"
    log_write "drvpath=${drvpath}"
    if [[ -e "${drvpath}" ]]; then
        log_write "not need to copy. (drv=${drvpath})"
        return ${CMD_SUCCESS}
    else
        drvcp_exec_cp "${drvpath}"  # copy
        ret=$?
        if [[ ${ret} -ne ${CMD_SUCCESS} ]]; then
            log_write "drvcp_exec_cp(${drvtype}) failed. (ret=${ret})"
            return ${ret}
        else
            log_write "${drvtype} driver copy success."
            log_write "${drvtype} driver load test start."
            load_exec "${drvtype}" "${drvpath}"        # exec load test
            ret=$?
            if [[ ${ret} -eq ${CMD_SUCCESS} ]]; then
                log_write "${drvtype} driver load test success."
                return ${ret}
            else
                log_write "${drvtype} driver load test failed. (ret=${ret})"
                return ${ret}
            fi
        fi
    fi
}

#------------------------------------------------
# Main
#------------------------------------------------
# Log start
log_start || exit ${CMD_ERR_LOG_INIT}

# Initialize variables
unload_khb_flag=0
unload_ka_flag=0
unload_liscal_flag=0
khb_ret=1
ka_ret=1
liscal_ret=1
err_cmd=""

# Get version number for linsmod
khbmajor=$(clpcfget -g ${XML_PATH_KHBMAJOR} -p lankhb)           # khb    メジャー番号
khbminor=$(clpcfget -g ${XML_PATH_KHBMINOR} -p lankhb)           # khb    マイナー番号
kamajor=$(clpcfget -g ${XML_PATH_KAMAJOR} -p cluster)            # ka     メジャー番号
kaminor=$(clpcfget -g ${XML_PATH_KAMINOR} -p cluster)            # ka     マイナー番号
liscalmajor=$(clpcfget -g ${XML_PATH_LISCALMAJOR} -p mddriver)   # liscal メジャー番号
liscalminor=0                                                    # liscal マイナー番号

# CLUSTERPRO Edition
edition=$(clpcfget -g ${XML_PATH_EDITION})
ret_editioncf=$?
if [[ ${ret_editioncf} -ne 0 ]]; then
    edition=SSS
fi

# Get kernel version
kernel=$(uname -r)

# khb
log_write "start - (khb copy)"
drvcp_main_cp "khb" "clpkhb"
khb_ret=$?
log_write "end - (khb copy)"

# ka
if [[ ${khb_ret} -eq ${CMD_SUCCESS} ]]; then  #khbのドライバが存在 または コピー&ロードに成功した場合のみ
    log_write "start - (ka copy)"
    drvcp_main_cp "ka" "clpka"
    ka_ret=$?
    log_write "end - (ka copy)"
fi

# liscal
if [[ ${edition} = "X" ]]; then
    log_write "start - (liscal copy)"
    drvcp_main_cp "md" "liscal"
    liscal_ret=$?
    log_write "end - (liscal copy)"
elif [[ ${edition} = "SSS" ]]; then
    log_write "liscal is not supported. (edition: ${edition})"
    liscal_ret=${CMD_SUCCESS}
else
    log_write "Edition is not supported. (edition: =${edition})"
    liscal_ret=${CMD_FAILED}
fi

# Unload all drivers
if [[ ${unload_liscal_flag} -eq 1 ]]; then
    err_cmd=$(rmmod liscal 2>&1)
    ret=$?
    if [[ ${ret} -ne 0 ]]; then
        log_write "${err_cmd}"
    fi
    if [[ ${ret} -eq 0 ]]; then
        log_write "rmmod liscal success."
    else 
        log_write "rmmod liscal failed."
    fi
fi
if [[ ${unload_ka_flag} -eq 1 ]]; then
    err_cmd=$(rmmod clpka 2>&1)
    ret=$?
    if [[ ${ret} -ne 0 ]]; then
        log_write "${err_cmd}"
    fi
    if [[ ${ret} -eq 0 ]]; then
        log_write "rmmod clpka success."
    else 
        log_write "rmmod clpka failed."
    fi
fi
if [[ ${unload_khb_flag} -eq 1 ]]; then
    err_cmd=$(rmmod clpkhb 2>&1)
    ret=$?
    if [[ ${ret} -ne 0 ]]; then
        log_write "${err_cmd}"
    fi
    if [[ ${ret} -eq 0 ]]; then
        log_write "rmmod clpkhb success."
    else 
        log_write "rmmod clpkhb failed."
    fi
fi

# Return value
if [[ ${khb_ret} -eq ${CMD_SUCCESS} && ${ka_ret} -eq ${CMD_SUCCESS} && ${liscal_ret} -eq ${CMD_SUCCESS} ]]; then
    log_write "command success."
    ret=${CMD_SUCCESS}
else
    log_write "command failed."
    ret=${CMD_FAILED}
fi

# Log end
log_end "${ret}"