#!/usr/bin/env python3
# coding: utf-8

import argparse
import ctypes
import ipaddress
import json
import logging
import os
import re
import subprocess
import sys
import traceback
import xml.etree.ElementTree as ET
from collections import defaultdict

################################################################################

VALUE_LEN = 256
CONTENTS_LEN = 1024
PATH_LEN = 1024
HOSTNAME_LEN = 256
DEVICE_LEN = 1024
TYPENAME_LEN = 32
RSCNAME_LEN = TYPENAME_LEN
CLNAME_LEN = TYPENAME_LEN
GRPNAME_LEN = TYPENAME_LEN

HBADEVICEID_LEN = 256
HBAINSTANSID_LEN = 256
DEV_PATH_LEN = CONTENTS_LEN
DRIVE_LETTER_LEN = 4
GUID_LEN = VALUE_LEN

PINGNP_IPADDR_NUM = 16
PINGNP_GRP_NUM = 16

INTID_MIN = 0
# INTID_MAX = 99
# PUBID_MIN = 100
PUBID_MAX = 199
# COMID_MIN = 200
# COMID_MAX = 299
DISKID_MIN = 300
DISKID_MAX = 399
MDCID_MIN = 400
MDCID_MAX = 499
# BMCID_MIN = 600
# BMCID_MAX = 699
WITNESSID_MIN = 700
WITNESSID_MAX = 799
DISKNPID_MIN = 10100
DISKNPID_MAX = 10199
PINGNPID_MIN = 10200
PINGNPID_MAX = 10299
MAJONPID_MIN = 10300
MAJONPID_MAX = 10399
HTTPNPID_MIN = 10700
HTTPNPID_MAX = 10799

PORT_NUM_MIN = 0
PORT_NUM_MAX = 65535

MESSAGE = {
    'success_cmd': 'Command succeeded.',
    'err_user_admin': 'Log in as administrator.',
    'err_user_root': 'Log in as root.',
    'err_not_found_file': '\'{}\' is not found.',
    'err_not_exist_object': 'The specified object does not exist. \'{}\'',
    'err_not_exist_element': 'The specified element \'{}\' does not exist in \'{}\'.',
    'err_not_exist_path': 'The specified path does not exist in a config file.',
    'err_clpconf': 'Invalid config file. Use the \'create\' option.',
    'err_clpconf_exist': 'The config file already exists.',
    'err_non_configurable': 'Non-configurable elements specified.',
    'err_specify_value': 'Invalid value specified. Specify as follows: <resource type>@<resource name>',
    'err_invalid_path': 'Invalid path specified.',
    'err_cnt': 'Cannot register a \'{}\' any more.',
    'err_arg_required': 'The following arguments are required :{}',
    'err_arg_not_allowed': 'Argument {}: allowed only with argument \'{}\'',
    'err_arg_choices': 'Argument {}: invalid choice: \'{}\' (choose from {})',
    'err_arg_range': 'Argument {}: invalid value: \'{}\' (The value must be in the range [{}, {}])',
    'err_arg_length': 'Argument {}: invalid value: \'{}\' (The length must be less than or equal to {})',
    'err_arg_exist': 'Argument {}: \'{}\' already exists.',
    'err_arg_not_exist': 'Argument {}: \'{}\' does not exist.',
    'err_arg_ref_sameobj': 'Argument {}: cannot specify a dependency to the same object.',
    'err_arg_ipaddr': 'Argument {}: does not appear to be an IPv4.',
    'err_positive_int': 'Invalid value: \'{}\' (The value must be greater than or equal to 0)',
    'err_internal': 'Internal error.',
}

CFADM_RETCODE = {
    'success': 0,
    'failure': 1,
}
CFSET_RETCODE = {
    'success': 0,
    'err_cmd': 1,
}

CLPCONF = 'clp.conf'

PROG = os.path.abspath(__file__)
if os.path.islink(PROG):
    PROG = os.readlink(PROG)

PROG_DIR = os.path.dirname(PROG)
CMD_CFSET = PROG_DIR + '/clpcfset'

POLICY = {
    'windows': PROG_DIR + '/../etc/cfadm/allpolicy_win.json',
    'linux': PROG_DIR + '/../etc/cfadm/allpolicy_lin.json',
}
IS_PRODUCT_SSS = False

DISPLAY_TARGETS_FOR_ROOT_CHILD_ELEMENTS = [
    'all',
    'cluster',
    'pm',
    'webmgr',
    'messages',
    'servergroup',
    'webalert',
    'rm',
    'group',
    'resource',
    'monitor',
    'server',
    'forcestop',
    'heartbeat',
    'networkpartition',
    'xclrule',
    'haltp',
    'mdagent',
    'mddriver',
    'diskhb',
    'rc',
    'reg',
    'jra',
    'rstd',
    'diskagent',
    'diskfltr',
    'diskperf',
    'alertservice',
]

ID_ATTRIBUTE = [
    'device',
    'list',
    'hba',
    'grp',
    'servers',
    'netdev',
    'svgpolicy',
    'smtpsrv',
    'snmpsrv',
    'env',
    'nm',
    'vol',
    'targetgrp',
    'diskcap',
]

CHANGE_ALL_TO_XPATH_FORMAT_AS_NAME_ATTR_REGEX \
    = ['@(.*?)(/|$)', '[@name="\\1"]\\2']

CHANGE_FROM_NAME_ATTR_TO_ID_ATTR_REGEX \
    = [r'({})\[@name'.format(('|').join(ID_ATTRIBUTE)), '\\1[@id']

SERVER_GROUP = '^servergroup$'
SERVER_GROUP_POLICY = '^servergroup@.+/policy$'
GROUP_RESOURCE = '^group@.+/resource$'
RESOURCE_DEPEND = '^resource/.+/depend$'
MONITOR_RELATION = '^monitor/.+/relation/name$'
MONITOR_TARGET = '^monitor/(?!.mrw@).?/target$'
WEBMGR_CLIENTLIST_IP = '^webmgr/security/clientlist/ip$'

BLANK = ''
LOCAL_SERVER = 'LocalServer'
ALL_GROUP = BLANK  # all group is blank tag

REF_OBJECT_REGEX = [
    {'path' : re.compile(r'{}'.format(SERVER_GROUP_POLICY)), 'include_type' : False, 'object' : ['srv']},
    {'path': re.compile(r'{}'.format(GROUP_RESOURCE)), 'include_type': True, 'object': ['rsc']},
    {'path' : re.compile(r'{}'.format(RESOURCE_DEPEND)),     'include_type' : False, 'object' : ['rsc', [BLANK]]},
    {'path' : re.compile(r'{}'.format(MONITOR_RELATION)),    'include_type' : False, 'object' : ['grp', 'rsc', [LOCAL_SERVER, ALL_GROUP]]},
    {'path' : re.compile(r'{}'.format(MONITOR_TARGET)),      'include_type' : False, 'object' : ['rsc', [BLANK]]},
]

CMD_RSCDEP_REGEX = re.compile(RESOURCE_DEPEND)

ADD_ATTR_AND_BLANK_TAG = [
    SERVER_GROUP,
    SERVER_GROUP_POLICY,
    GROUP_RESOURCE,
    # RESOURCE_DEPEND,
    WEBMGR_CLIENTLIST_IP,
]
ADD_ATTR_AND_BLANK_TAG_REGEX = re.compile(r'{}'.format(('|').join(ADD_ATTR_AND_BLANK_TAG)))

ACTION_ADD = 'add'
ACTION_DEL = 'del'

SERVEROS = ['windows', 'linux']
CHARSET = {
    'windows': ['ASCII', 'SJIS', 'GB2312'],
    'linux': ['ASCII', 'EUC-JP', 'GB2312'],
}
ENCODE = ['ASCII', 'SJIS', 'EUC-JP', 'GB2312']

DEVICE_TYPE = {
    'windows': ['lan', 'mdc', 'witness', 'disknp', 'ping', 'http', 'majo'],
    'linux': ['lan', 'mdc', 'witness', 'disk', 'ping', 'http'],
}

DEVICE_ID_RANGE = {
    'lan': {'min': INTID_MIN, 'max': PUBID_MAX},
    'mdc': {'min': MDCID_MIN, 'max': MDCID_MAX},
    'witness': {'min': WITNESSID_MIN, 'max': WITNESSID_MAX},
    # 'bmc'     : {'min' : BMCID_MIN,     'max' : BMCID_MAX},
    'disknp': {'min': DISKNPID_MIN, 'max': DISKNPID_MAX},
    'disk': {'min': DISKID_MIN, 'max': DISKID_MAX},
    'ping': {'min': PINGNPID_MIN, 'max': PINGNPID_MAX},
    'http': {'min': HTTPNPID_MIN, 'max': HTTPNPID_MAX},
    'majo': {'min': MAJONPID_MIN, 'max': MAJONPID_MAX},
}

HB_DEVICE_ID_RANGE = {
    'lankhb': {'min': INTID_MIN, 'max': PUBID_MAX},
    'lanhb': {'min': INTID_MIN, 'max': PUBID_MAX},
    'diskhb': {'min': DISKID_MIN, 'max': DISKID_MAX},
    # 'bmchb'     : {'min' : BMCID_MIN,     'max' : BMCID_MAX},
    # 'comhb'     : {'min' : COMID_MIN,     'max' : COMID_MAX},
    'witnesshb': {'min': WITNESSID_MIN, 'max': WITNESSID_MAX},
}

NP_DEVICE_ID_RANGE = {
    'disknp': {'min': DISKNPID_MIN, 'max': DISKNPID_MAX},
    'pingnp': {'min': PINGNPID_MIN, 'max': PINGNPID_MAX},
    'httpnp': {'min': HTTPNPID_MIN, 'max': HTTPNPID_MAX},
    # 'comnp'  : {'min' : COMNPID_MIN,  'max' : COMNPID_MAX},
    'majonp': {'min': MAJONPID_MIN, 'max': MAJONPID_MAX},
}

GRP_TYPE = ['failover', 'ManagementGroup']

################################################################################


class CfAdmError(Exception):
    def __init__(self, retcode, msg):
        self.retcode = retcode
        self.msg = msg


class CfAdmArgumentParser(argparse.ArgumentParser):
    def error(self, message):
        message = message[0].upper() + message[1:]
        self.exit(CFADM_RETCODE['failure'], '{}\n'.format(message))

    def print_usage(self, fd=sys.stderr):
        super().print_usage(fd)
        self.exit(CFADM_RETCODE['failure'])

    def print_help(self):
        super().print_help()
        self.exit(CFADM_RETCODE['failure'])


def deep_merge(dict1, dict2):
    dict3 = dict2.copy()
    for k, v in dict1.items():
        if k in dict3 and type(dict3[k]) is dict and type(v) is dict:
            dict3[k] = deep_merge(v, dict3[k])
        else:
            dict3[k] = v

    return dict3


def change_dict_keys_to_have_attr(dict1, dict2):
    remove_key = []
    for k in dict1:
        try:
            if '@' in k:
                d = k.split('@')[0]
                dict2[k] = dict2[d].copy()
                remove_key.append(d)
            else:
                if type(dict1[k]) is dict:
                    change_dict_keys_to_have_attr(dict1[k], dict2[k])
        except KeyError:
            continue

    for k in list(set(remove_key)):
        dict2.pop(k)


def convert_xml_to_dict(element):

    # if the attribute exists, the value connected by '@' is used as the dict key
    key = element.tag if not element.attrib else \
            element.tag + '@' + ''.join(list(element.attrib.values()))

    _dict = {key: None}
    child_element = list(element)

    if child_element:
        ddict = defaultdict(list)
        for child_dict in map(convert_xml_to_dict, child_element):
            for k, v in child_dict.items():
                ddict[k].append(v)
        d = {}
        for k, v in ddict.items():
            d[k] = v[0] if len(v) == 1 else v
        _dict[key] = d

    if element.text:
        text = element.text.strip()
        if text:
            _dict[key] = text

    return _dict


def get_encoding(file):
    encoding = None
    is_cr = False
    with open(file, "rb") as openfile:
        # 1バイトずつ読み込みCR, CRLFを探す
        for readbyte in iter(lambda: openfile.read(1), b''):
            if (readbyte == b'\r'):
                is_cr = True
            elif (readbyte == b'\n'):
                encoding = "SJIS" if is_cr else "EUC-JP"
                break
            else:
                is_cr = False
    logging.debug(f"Open File: {file}")
    logging.debug(f"  - encoding: {encoding}")
    return encoding


class Config:
    def __init__(self):
        self.clpconf, clpconf_data = self._get_clpconf()
        self.policy, policy_data = self._get_policy(clpconf_data['serveros'])
        config_data = deep_merge(clpconf_data, policy_data)
        for key in config_data:
            setattr(self, key, config_data[key])
        self.config = self._make_config(self.clpconf, self.policy)

    def _make_config(self, clpconf_xml, policy):
        clpconf_dict = convert_xml_to_dict(clpconf_xml)['root']
        change_dict_keys_to_have_attr(clpconf_dict, policy)
        config = deep_merge(clpconf_dict, policy)
        self._delete_elements(config)

        return config

    def _get_clpconf(self):
        try:
            with open(CLPCONF, 'r', encoding=get_encoding(CLPCONF)) as f:
                root = ET.fromstring(f.read())

            serveros = root.find('./all/serveros').text

        except FileNotFoundError:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_found_file'].format(CLPCONF))

        except Exception:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_clpconf'])

        if serveros not in ['windows', 'linux']:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_clpconf'])

        srv = {}
        srv['name'] = [n.get('name') for n in root.findall('./server')]
        srv['priority'] = [int(n.text) for n in root.findall('./server/priority')]
        for n in srv['name']:
            srv[n] = {}
            srv[n]['devid'] = [int(n.get('id')) for n in root.findall('./server/[@name="' + n + '"]/device')]
            srv[n]['devcnt'] = len(srv[n]['devid'])
            srv[n]['hbaid'] = [int(n.get('id')) for n in root.findall('./server[@name="' + n + '"]/hba')]
        srv['use_cnt'] = len(srv['name'])

        forcestop = {}
        forcestop['use_type'] = [n.text for n in root.findall('./forcestop/env')]

        hb = {}
        hb['use_type'] = [n.get('name') for n in root.findall('./heartbeat/types')]
        hb['devid'] = [int(m.text) for n in hb['use_type'] for m in root.findall('./heartbeat/' + n + '/device')]
        hb['priority'] = [int(m.text) for n in hb['use_type'] for m in root.findall('./heartbeat/' + n + '/priority')]
        for n in hb['use_type']:
            hb[n] = {}
            hb[n]['devid'] = [int(m.text) for m in root.findall('./heartbeat/' + n + '/device')]
            hb[n]['priority'] = [int(m.text) for m in root.findall('./heartbeat/' + n + '/priority')]
        hb['use_cnt'] = len(hb['devid'])

        np = {}
        np['use_type'] = [n.get('name') for n in root.findall('./networkpartition/types')]
        np['devid'] = [int(m.text) for n in np['use_type'] for m in root.findall('./networkpartition/' + n + '/device')]
        np['priority'] = [int(m.text) for n in np['use_type'] for m in root.findall('./networkpartition/' + n + '/priority')]
        for n in np['use_type']:
            np[n] = {}
            np[n]['devid'] = [int(m.text) for m in root.findall('./networkpartition/' + n + '/device')]
            np[n]['priority'] = [int(m.text) for m in root.findall('./networkpartition/' + n + '/priority')]
        np['use_cnt'] = len(np['devid'])

        grp = {}
        grp['exist_manage'] = True if 'cluster' in [n.text for n in root.findall('./group/type')] else False
        grp['name'] = [n.get('name') for n in root.findall('./group') if n.get('name') is not None]
        for n in grp['name']:
            grp[n] = {}
            grp[n]['rscname'] = [m.get('name').split('@')[1] \
                for m in root.findall('./group[@name="' + n + '"]/resource') if '@' in m.get('name')]
            grp[n]['use_rsccnt'] = len(grp[n]['rscname'])
        grp['use_cnt'] = len(grp['name'])

        rsc = {}
        rsc['use_type'] = [n.get('name') for n in root.findall('./resource/types')]
        rsc['name'] = [m.get('name') for n in rsc['use_type'] for m in root.findall('./resource/' + n)]
        for n in rsc['use_type']:
            rsc[n] = {}
            rsc[n]['name'] = [m.get('name') for m in root.findall('./resource/' + n)]
            # for m in rsc[n]['name']:
            #    rsc[n][m] = {}
            #    rsc[n][m]['depend'] = [m.get('name') for m in root.findall('./resource/' + n + '[@name="' + m +'"]/depend')]
        # rsc['use_cnt'] = len(rsc['name'])

        mon = {}
        mon['use_type'] = [n.get('name') for n in root.findall('./monitor/types')]
        mon['name'] = [m.get('name') for n in mon['use_type'] for m in root.findall('./monitor/' + n)]
        for n in mon['use_type']:
            mon[n] = {}
            mon[n]['name'] = [m.get('name') for m in root.findall('./monitor/' + n)]
        mon['use_cnt'] = len(mon['name'])

        data = dict(serveros=serveros, forcestop=forcestop, srv=srv, hb=hb, np=np, grp=grp, rsc=rsc, mon=mon)
        return root, data

    def _get_policy(self, serveros):
        global IS_PRODUCT_SSS
        try:
            with open(POLICY[serveros], 'r', encoding='utf-8') as f:
                root = json.load(f)['root']

        except FileNotFoundError:
            file = os.path.basename(POLICY[serveros])
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_found_file'].format(file))

        srv = {}
        srv['max_cnt'] = int(root['all']['num']['server'])

        forcestop = {}
        IS_PRODUCT_SSS = "SSS" in root["all"]["edition"]
        if not IS_PRODUCT_SSS:
            forcestop['all_type'] = [n.strip() for n in root['forcestop']['envitem']['list'].split(',') \
                                    if n.strip() != 'disabled']

        hb = {}
        hb['all_type'] = [n for n in root['heartbeat']]
        hb['max_cnt'] = int(root['all']['num']['hbr'])

        np = {}
        np['all_type'] = [n for n in root['networkpartition']]
        np['max_cnt'] = int(root['all']['num']['npr'])

        grp = {}
        grp['max_cnt'] = int(root['all']['num']['group'])

        exclusion = ['nas', 'spool', 'vm']
        rsc = {}
        rsc['all_type'] = [n for n in root['resource'] if n not in exclusion]
        rsc['max_cnt'] = int(root['all']['num']['resource'])

        exclusion = ['jra', 'bmcw', 'nasw', 'osmw', 'spoolw', 'sybasew', 'vmw']
        mon = {}
        mon['all_type'] = [n for n in root['monitor'] if n not in exclusion]
        mon['max_cnt'] = int(root['all']['num']['monitor'])

        data = dict(srv=srv, forcestop=forcestop, hb=hb, np=np, grp=grp, rsc=rsc, mon=mon)
        return root, data

    def _delete_elements(self, config):
        elements = [
            {'target': config, 'use_children': DISPLAY_TARGETS_FOR_ROOT_CHILD_ELEMENTS},
            {'target': config['resource'], 'use_children': self.rsc['use_type']},
            {'target': config['monitor'], 'use_children': self.mon['use_type']},
            {'target': config['heartbeat'], 'use_children': self.hb['use_type']},
            {'target' : config['networkpartition'], 'use_children' : self.np['use_type']}
        ]
        if not IS_PRODUCT_SSS:
            elements.append({'target': config['forcestop'], 'use_children': self.forcestop['use_type']})

        for element in elements:
            if not element['target']:
                continue
            for child in list(element['target']):
                if child.split('@')[0] not in element['use_children']:
                    element['target'].pop(child)

        # delete from policy dict if not exist in clp.conf
        for n in ['server', 'group', 'resource', 'monitor', 'forcestop', 'heartbeat', 'networkpartition']:
            exist_object = False
            for m in self.clpconf:
                if n in m.tag:
                    exist_object = True
                    break

            if not exist_object:
                for m in list(config):
                    if n == m:
                        config.pop(n)

        exist_server = False
        for n in self.clpconf:
            if 'server' == n.tag:
                exist_server = True

        if not exist_server:
            config.pop('servergroup')

    def _get_element(self, path):
        if path is None:
            return self.config

        element = self.config
        exist_path = ''
        for child in path.split('/'):
            try:
                element = element[child]
            except (KeyError, TypeError):  # TypeError is for NoneType
                err_msg = MESSAGE['err_not_exist_element'].format(child, '/root' + exist_path)
                raise CfAdmError(CFADM_RETCODE['failure'], err_msg)
            exist_path += '/' + child

        return element

    def get_node_value(self, elements: dict, is_empty_ok=False) -> dict:
        result = {}
        for key, path in elements.items():
            value = self._get_element(path)
            if not isinstance(value, str) and not is_empty_ok:
                raise CfAdmError(
                    CFADM_RETCODE['failure'],
                    MESSAGE['err_non_configurable'])
            result[key] = value
        logging.debug("get_node_value")
        for key, val in result.items():
            logging.debug(f" - {key}: {val}")
        return result

    def show_child_elements(self, path):
        element = self._get_element(path)

        if type(element) is str or element is None:
            result = '[{}]'.format(element)
        else:
            temp = []
            for child_key, child_value in sorted(element.items()):
                if type(element[child_key]) is dict:
                    temp.append(child_key)
                else:
                    temp.append('{} [{}]'.format(child_key, child_value))
            result = '\n'.join(temp)

        if result:
            print(result)

    def check_refer_object(self, path, value):
        for regex in REF_OBJECT_REGEX:
            match = regex['path'].search(path)
            if match is None:
                continue

            if not regex['include_type']:
                obj_name = value
            else:
                match_type = False
                if '@' not in value or len(value.split('@')) != 2:
                    raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_specify_value'])

                obj_type, obj_name = value.split('@')
                for obj_key in regex['object']:
                    if obj_type in getattr(self, obj_key)['use_type']:
                        match_type = True

                if not match_type:
                    raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_exist_object'].format(obj_type))

            for obj_key in regex['object']:
                if type(obj_key) is list:
                    if obj_name in obj_key:
                        return
                else:
                    if obj_name in getattr(self, obj_key)['name']:
                        return

            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_exist_object'].format(obj_name))

    def check_path(self, path, action, check=True):
        if path is None:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_invalid_path'])

        if action == ACTION_ADD:
            if check:
                self._check_configurable_element(path)
        else:  # ACTION_DEL
            # search the config dict to detect element name errors(for error message)
            self._get_element(path)
            self.check_exist_in_clpconf(path)

    def _check_configurable_element(self, path):
        element = self._get_element(path)

        if element is not None and type(element) is not str:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_non_configurable'])

    def check_exist_in_clpconf(self, path):
        xpath = self._change_path_to_xpath(path)

        if self.clpconf.find(xpath) is None:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_exist_path'])

    def _change_path_to_xpath(self, path):
        xpath = './' + path
        xpath = re.sub(*CHANGE_ALL_TO_XPATH_FORMAT_AS_NAME_ATTR_REGEX, xpath)
        xpath = re.sub(*CHANGE_FROM_NAME_ATTR_TO_ID_ATTR_REGEX, xpath)
        # logging.debug('xpath : {}'.format(xpath))
        return xpath


def _is_positive_int(value):
    # Not exactly a positive integer, but a number greater than zero
    if not value.isdecimal() or int(value) < 0:
        raise argparse.ArgumentTypeError(MESSAGE['err_positive_int'].format(value))
    return int(value)


def _is_ipaddr(value, item_name, parser, including_port=False):
    ipaddr = value
    if including_port:
        if ':' not in value or len(value.split(':')) != 2:
            parser.error(MESSAGE['err_arg_ipaddr'].format(item_name) + ':port num')

        ipaddr, portnum = value.split(':')
        if not portnum.isdecimal() or not (PORT_NUM_MIN <= int(portnum) <= PORT_NUM_MAX):
            parser.error(MESSAGE['err_arg_range'].format(item_name, portnum, PORT_NUM_MIN, PORT_NUM_MAX))
    try:
        ipaddress.ip_address(ipaddr)
    except ValueError:
        parser.error(MESSAGE['err_arg_ipaddr'].format(item_name))


def _to_help_msg(_list):
    return '{' + ','.join(_list) + '}'


def _create(create_parser):

    create_parser.add_argument('clsname')
    create_parser.add_argument('charset')
    create_parser.add_argument('-e', dest='encode', metavar='encode')
    create_parser.add_argument('-s', dest='serveros', metavar='serveros', choices=SERVEROS)

    args = create_parser.parse_args()

    if os.path.isfile(CLPCONF):
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_clpconf_exist'])

    if len(args.clsname) >= CLNAME_LEN:
        create_parser.error(MESSAGE['err_arg_length'].format('clsname', args.clsname, CLNAME_LEN - 1))

    if args.serveros:
        serveros = args.serveros
    else:
        serveros = 'windows' if os.name == 'nt' else 'linux'

    if args.charset not in CHARSET[serveros]:
        create_parser.error(MESSAGE['err_arg_choices'].format('charset', args.charset, CHARSET[serveros]))

    if args.encode and args.encode not in ENCODE:
        create_parser.error(MESSAGE['err_arg_choices'].format('encode', args.encode, ENCODE))

    if args.encode is None and args.serveros:
        args.encode = args.charset

    cmdline = [CMD_CFSET, 'create', args.clsname, args.charset, args.encode, args.serveros]
    execute_command(cmdline)


def _add(add_parser):

    subparsers = add_parser.add_subparsers()
    add_srv_parser = subparsers.add_parser('srv', parents=[cmn_parser], help='see `add srv -h`')
    if config.serveros == 'windows':
        add_hba_parser = subparsers.add_parser('hba', parents=[cmn_parser], help='see `add hba -h`')
    add_device_parser    = subparsers.add_parser('device',    parents=[cmn_parser], help='see `add device -h`')
    if not IS_PRODUCT_SSS:
        add_forcestop_parser = subparsers.add_parser('forcestop', parents=[cmn_parser], help='see `add forcestop -h`')
    add_hb_parser = subparsers.add_parser('hb', parents=[cmn_parser], help='see `add hb -h`')
    add_np_parser = subparsers.add_parser('np', parents=[cmn_parser], help='see `add np -h`')
    add_grp_parser = subparsers.add_parser('grp', parents=[cmn_parser], help='see `add grp -h`')
    add_rsc_parser = subparsers.add_parser('rsc', parents=[cmn_parser], help='see `add rsc -h`')
    add_rscdep_parser    = subparsers.add_parser('rscdep',    parents=[cmn_parser], help='see `add rscdep -h`')
    add_mon_parser = subparsers.add_parser('mon', parents=[cmn_parser], help='see `add mon -h`')

    commands = {
        'srv': {'func': _add_srv, 'parser': add_srv_parser},
        'device': {'func': _add_device, 'parser': add_device_parser},
        'hb': {'func': _add_hb, 'parser': add_hb_parser},
        'np': {'func': _add_np, 'parser': add_np_parser},
        'grp': {'func': _add_grp, 'parser': add_grp_parser},
        'rsc': {'func': _add_rsc, 'parser': add_rsc_parser},
        'rscdep': {'func': _add_rscdep, 'parser': add_rscdep_parser},
        'mon': {'func': _add_mon, 'parser': add_mon_parser},
    }

    if config.serveros == 'windows':
        commands['hba'] = {'func': _add_hba, 'parser': add_hba_parser}
    if not IS_PRODUCT_SSS:
        commands['forcestop'] = {'func': _add_forcestop, 'parser': add_forcestop_parser}

    sys.argv = sys.argv[1:]
    if len(sys.argv) > 0 and sys.argv[0] in commands:
        commands[sys.argv[0]]['func'](commands[sys.argv[0]]['parser'])

    elif len(sys.argv) > 0 and ('-h' in sys.argv or '--help' in sys.argv):
        add_parser.print_help()

    else:
        add_parser.print_usage()


def _add_srv(add_srv_parser):

    add_srv_parser.add_argument('srvname')
    add_srv_parser.add_argument('priority', type=_is_positive_int)

    args = add_srv_parser.parse_args()

    if config.srv['use_cnt'] == config.srv['max_cnt']:
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('server'))

    if len(args.srvname) >= HOSTNAME_LEN:
        add_srv_parser.error(MESSAGE['err_arg_length'].format('srvname', args.srvname, HOSTNAME_LEN - 1))

    if args.srvname in config.srv['name']:
        add_srv_parser.error(MESSAGE['err_arg_exist'].format('srvname', args.srvname))

    if args.priority >= config.srv['max_cnt']:
        add_srv_parser.error(MESSAGE['err_arg_range'].format('priority', args.priority, 0, config.srv['max_cnt'] - 1))

    if args.priority in config.srv['priority']:
        add_srv_parser.error(MESSAGE['err_arg_exist'].format('priority', args.priority))

    cmdline = [CMD_CFSET, 'add', 'srv', args.srvname, args.priority]
    execute_command(cmdline)


def _add_hba(add_hba_parser):

    add_hba_parser.add_argument('srvname')
    add_hba_parser.add_argument('hbaid', type=_is_positive_int)
    add_hba_parser.add_argument('portnum', type=_is_positive_int)
    add_hba_parser.add_argument('hbadevid')
    add_hba_parser.add_argument('hbainstanceid')

    args = add_hba_parser.parse_args()

    if args.srvname not in config.srv['name']:
        add_hba_parser.error(MESSAGE['err_arg_not_exist'].format('srvname', args.srvname))

    if args.hbaid in config.srv[args.srvname]['hbaid']:
        add_hba_parser.error(MESSAGE['err_arg_exist'].format('hbaid', args.hbaid))

    if not (PORT_NUM_MIN <= args.portnum <= PORT_NUM_MAX):
        add_hba_parser.error(MESSAGE['err_arg_range'].format('portnum', args.portnum, PORT_NUM_MIN, PORT_NUM_MAX))

    if len(args.hbadevid) >= HBADEVICEID_LEN:
        add_hba_parser.error(MESSAGE['err_arg_length'].format('hbadevid', args.hbadevid, HBADEVICEID_LEN - 1))

    if len(args.hbainstanceid) >= HBAINSTANSID_LEN:
        add_hba_parser.error(MESSAGE['err_arg_length'].format('hbainstanceid', args.hbainstanceid, HBAINSTANSID_LEN - 1))

    cmdline = [CMD_CFSET, 'add', 'hba', args.srvname, args.hbaid, args.portnum, args.hbadevid, args.hbainstanceid]
    execute_command(cmdline)


def _add_device(add_device_parser):

    choice_devtype = DEVICE_TYPE[config.serveros]
    output_devtype = 'witness or disknp' if config.serveros == 'windows' else 'witness'

    add_device_parser.add_argument('srvname')
    add_device_parser.add_argument('devtype', metavar='devtype', choices=choice_devtype, help=_to_help_msg(choice_devtype))
    add_device_parser.add_argument('devid', type=_is_positive_int)
    add_device_parser.add_argument('info')
    add_device_parser.add_argument('extend', nargs='?', help='required if devtype is {}'.format(output_devtype))

    args = add_device_parser.parse_args()

    if args.extend and args.devtype not in ['witness', 'disknp']:
        add_device_parser.error(MESSAGE['err_arg_not_allowed'].format('extend', 'devtype is ' + output_devtype))
    if args.extend is None and args.devtype in ['witness', 'disknp']:
        add_device_parser.error(MESSAGE['err_arg_required'].format('extend'))

    if args.srvname not in config.srv['name']:
        add_device_parser.error(MESSAGE['err_arg_not_exist'].format('srvname', args.srvname))

    devid_min = DEVICE_ID_RANGE[args.devtype]['min']
    devid_max = DEVICE_ID_RANGE[args.devtype]['max']

    register_devid = args.devid + devid_min
    if not (devid_min <= register_devid <= devid_max):
        add_device_parser.error(MESSAGE['err_arg_range'].format('devid', args.devid, 0, devid_max - devid_min))

    if register_devid in config.srv[args.srvname]['devid']:
        add_device_parser.error(MESSAGE['err_arg_exist'].format('devid', args.devid))

    if args.devtype in ['lan', 'mdc']:
        _is_ipaddr(args.info, 'info', add_device_parser)

    if args.devtype in ['witness', 'ping', 'http', 'majo']:
        if not args.info.isdecimal() or (int(args.info) != 0 and int(args.info) != 1):
            add_device_parser.error(MESSAGE['err_arg_choices'].format('info', args.info, [0, 1]))

    if args.devtype == 'disk' and len(args.info) >= DEV_PATH_LEN:
        add_device_parser.error(MESSAGE['err_arg_length'].format('info', args.info, DEV_PATH_LEN - 1))

    if args.devtype == 'disknp' and len(args.info) >= GUID_LEN:
        add_device_parser.error(MESSAGE['err_arg_length'].format('info', args.info, GUID_LEN - 1))

    if args.devtype == 'witness':
        _is_ipaddr(args.extend, 'extend', add_device_parser, including_port=True)

    if args.devtype == 'disknp' and len(args.extend) >= DRIVE_LETTER_LEN:
        add_device_parser.error(MESSAGE['err_arg_length'].format('extend', args.extend, DRIVE_LETTER_LEN - 1))

    cmdline = [CMD_CFSET, 'add', 'device', args.srvname, args.devtype, args.devid, args.info, args.extend]
    execute_command(cmdline)


def _add_forcestop(add_forcestop_parser):

    add_forcestop_parser.add_argument('env', metavar='env',
        choices=config.forcestop['all_type'],
                                      help=_to_help_msg(config.forcestop['all_type']))
    args = add_forcestop_parser.parse_args()

    if config.forcestop['use_type'] and config.forcestop['use_type'][0] != 'disabled':
        add_forcestop_parser.error(MESSAGE['err_cnt'].format('force stop resource'))

    cmdline = [CMD_CFSET, 'add', 'forcestop', args.env]
    execute_command(cmdline)


def _add_hb(add_hb_parser):

    # "bmchb" and "comhb" are not supported
    hbtype = [c for c in config.hb['all_type'] if c != 'bmchb' and c != 'comhb']

    # add hb common argument
    add_hb_common_parser = CfAdmArgumentParser(add_help=False)
    add_hb_common_parser.add_argument('devid', type=_is_positive_int)
    add_hb_common_parser.add_argument('priority', type=_is_positive_int)

    subparsers = add_hb_parser.add_subparsers()

    parser = {}; commands = {}
    for name in hbtype:
        parser[name] = subparsers.add_parser(
                            name,
                            parents=[cmn_parser, add_hb_common_parser],
                            help='see `add hb {} -h`'.format(name))
        commands[name] = {'func': eval('_add_hb_' + name), 'parser': parser[name]}

    sys.argv = sys.argv[1:]
    if len(sys.argv) > 0 and sys.argv[0] in commands:

        # add individual argument
        add_hb_child_parser = commands[sys.argv[0]]['parser']
        commands[sys.argv[0]]['func'](add_hb_child_parser)

        args = add_hb_child_parser.parse_args()

        if config.hb['use_cnt'] == config.hb['max_cnt']:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('heartbeat'))

        input_hbtype = sys.argv[0]
        devid_min = HB_DEVICE_ID_RANGE[input_hbtype]['min']
        devid_max = HB_DEVICE_ID_RANGE[input_hbtype]['max']

        register_devid = args.devid + devid_min
        if not (devid_min <= register_devid <= devid_max):
            add_hb_child_parser.error(MESSAGE['err_arg_range'].format('devid', args.devid, 0, devid_max - devid_min))

        if register_devid in config.hb['devid']:
            add_hb_child_parser.error(MESSAGE['err_arg_exist'].format('devid', args.devid))

        if args.priority > devid_max - devid_min:
            add_hb_child_parser.error(MESSAGE['err_arg_range'].format('priority', args.priority, 0, devid_max - devid_min))

        if args.priority in config.hb['priority']:
            add_hb_child_parser.error(MESSAGE['err_arg_exist'].format('priority', args.priority))

        cmdline = [CMD_CFSET, 'add', 'hb', input_hbtype, args.devid, args.priority]

        if hasattr(args, 'host'):
            _is_ipaddr(args.host, 'host', add_hb_child_parser, including_port=True)
            cmdline.append(args.host)

        execute_command(cmdline)

    elif len(sys.argv) > 0 and ('-h' in sys.argv or '--help' in sys.argv):
        add_hb_parser.print_help()

    else:
        add_hb_parser.print_usage()


def _add_hb_lankhb(add_hb_lankhb_parser):
    pass


def _add_hb_lanhb(add_hb_lanhb_parser):
    pass


def _add_hb_bmchb(add_hb_bmchb_parser):
    pass


def _add_hb_diskhb(add_hb_diskhb_parser):
    pass


def _add_hb_witnesshb(add_hb_witnesshb_parser):
    add_hb_witnesshb_parser.add_argument('host')


def _add_np(add_np_parser):

    # "comnp" is not supported
    nptype = [c for c in config.np['all_type'] if c != 'comnp']

    add_np_common_parser = CfAdmArgumentParser(add_help=False)
    add_np_common_parser.add_argument('devid', type=_is_positive_int)
    add_np_common_parser.add_argument('priority', type=_is_positive_int)

    subparsers = add_np_parser.add_subparsers()

    parser = {}; commands = {}
    for name in nptype:
        parser[name] = subparsers.add_parser(
                            name,
                            parents=[cmn_parser, add_np_common_parser],
                            help='see `add np {} -h`'.format(name))
        commands[name] = {'func': eval('_add_np_' + name), 'parser': parser[name]}

    sys.argv = sys.argv[1:]
    if len(sys.argv) > 0 and sys.argv[0] in commands:

        # add individual argument
        add_np_child_parser = commands[sys.argv[0]]['parser']
        commands[sys.argv[0]]['func'](add_np_child_parser)

        args = add_np_child_parser.parse_args()

        if config.np['use_cnt'] == config.np['max_cnt']:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('networkpartition'))

        input_nptype = sys.argv[0]
        devid_min = NP_DEVICE_ID_RANGE[input_nptype]['min']
        devid_max = NP_DEVICE_ID_RANGE[input_nptype]['max']

        register_devid = args.devid + devid_min
        if not (devid_min <= register_devid <= devid_max):
            add_np_child_parser.error(MESSAGE['err_arg_range'].format('devid', args.devid, 0, devid_max - devid_min))

        if register_devid in config.np['devid']:
            add_np_child_parser.error(MESSAGE['err_arg_exist'].format('devid', args.devid))

        if args.priority > devid_max - devid_min:
            add_np_child_parser.error(MESSAGE['err_arg_range'].format('priority', args.priority, 0, devid_max - devid_min))

        if args.priority in config.np['priority']:
            add_np_child_parser.error(MESSAGE['err_arg_exist'].format('priority', args.priority))

        cmdline = [CMD_CFSET, 'add', 'np', input_nptype, args.devid, args.priority]

        if hasattr(args, 'grpid'):
            if not (0 <= args.grpid <= PINGNP_GRP_NUM - 1):
                add_np_child_parser.error(MESSAGE['err_arg_range'].format('grpid', args.grpid, 0, PINGNP_GRP_NUM - 1))
            cmdline.append(args.grpid)

        if hasattr(args, 'listid'):
            if not (0 <= args.listid <= PINGNP_IPADDR_NUM - 1):
                add_np_child_parser.error(MESSAGE['err_arg_range'].format('listid', args.listid, 0, PINGNP_IPADDR_NUM - 1))
            cmdline.append(args.listid)

        if hasattr(args, 'ipaddr'):
            _is_ipaddr(args.ipaddr, 'ipaddr', add_np_child_parser)
            cmdline.append(args.ipaddr)

        if hasattr(args, 'host') and args.host is not None:
            _is_ipaddr(args.host, 'host', add_np_child_parser, including_port=True)
            cmdline.append(args.host)

        execute_command(cmdline)

    elif len(sys.argv) > 0 and ('-h' in sys.argv or '--help' in sys.argv):
        add_np_parser.print_help()

    else:
        add_np_parser.print_usage()


def _add_np_majonp(add_np_majonp_parser):
    pass


def _add_np_disknp(add_np_disknp_parser):
    pass


def _add_np_pingnp(add_np_pingnp_parser):
    add_np_pingnp_parser.add_argument('grpid', type=_is_positive_int)
    add_np_pingnp_parser.add_argument('listid', type=_is_positive_int)
    add_np_pingnp_parser.add_argument('ipaddr')


def _add_np_httpnp(add_np_httpnp_parser):
    add_np_httpnp_parser.add_argument('--host', metavar='host')


def _add_grp(add_grp_parser):

    add_grp_parser.add_argument('grptype', metavar='grptype', choices=GRP_TYPE, help=_to_help_msg(GRP_TYPE))
    add_grp_parser.add_argument('grpname')

    args = add_grp_parser.parse_args()

    if config.grp['use_cnt'] == config.grp['max_cnt']:
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('group'))

    if args.grptype == 'ManagementGroup' and config.grp['exist_manage']:
        add_grp_parser.error(MESSAGE['err_cnt'].format('ManagementGroup'))

    if len(args.grpname) >= GRPNAME_LEN:
        add_grp_parser.error(MESSAGE['err_arg_length'].format('grpname', args.grpname, GRPNAME_LEN - 1))

    if args.grpname in config.grp['name']:
        add_grp_parser.error(MESSAGE['err_arg_exist'].format('grpname', args.grpname))

    cmdline = [CMD_CFSET, 'add', 'grp', args.grptype, args.grpname]
    execute_command(cmdline)


def _add_rsc(add_rsc_parser):

    add_rsc_parser.add_argument('grpname')
    add_rsc_parser.add_argument('rsctype')
    add_rsc_parser.add_argument('rscname')

    args = add_rsc_parser.parse_args()

    if args.grpname not in config.grp['name']:
        add_rsc_parser.error(MESSAGE['err_arg_not_exist'].format('grpname', args.grpname))

    if args.rsctype not in config.rsc['all_type']:
        add_rsc_parser.error(MESSAGE['err_arg_not_exist'].format('rsctype', args.rsctype))

    if len(args.rscname) >= RSCNAME_LEN:
        add_rsc_parser.error(MESSAGE['err_arg_length'].format('rscname', args.rscname, RSCNAME_LEN - 1))

    if args.rscname in config.rsc['name']:
        add_rsc_parser.error(MESSAGE['err_arg_exist'].format('rscname', args.rscname))

    if config.grp[args.grpname]['use_rsccnt'] == config.rsc['max_cnt']:
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('group resource'))

    cmdline = [CMD_CFSET, 'add', 'rsc', args.grpname, args.rsctype, args.rscname]
    execute_command(cmdline)


def _add_rscdep(add_rscdep_parser):

    add_rscdep_parser.add_argument('rsctype')
    add_rscdep_parser.add_argument('rscname')
    add_rscdep_parser.add_argument('deprscname')

    args = add_rscdep_parser.parse_args()

    if args.rsctype not in config.rsc['use_type']:
        add_rscdep_parser.error(MESSAGE['err_arg_not_exist'].format('rsctype', args.rsctype))

    if args.rscname not in config.rsc[args.rsctype]['name']:
        add_rscdep_parser.error(MESSAGE['err_arg_not_exist'].format('rscname', args.rscname))

    if args.deprscname != '' and args.deprscname not in config.rsc['name']:
        add_rscdep_parser.error(MESSAGE['err_arg_not_exist'].format('deprscname', args.deprscname))

    if args.rscname == args.deprscname:
        add_rscdep_parser.error(MESSAGE['err_arg_ref_sameobj'].format('deprscname'))

    # if args.deprscname in config.rsc[args.rsctype][args.rscname]['depend']:
    #    add_rscdep_parser.error(MESSAGE['err_arg_exist'].format('deprscname', args.deprscname))

    cmdline = [CMD_CFSET, 'add', 'rscdep', args.rsctype, args.rscname, args.deprscname]
    execute_command(cmdline)


def _add_mon(add_mon_parser):

    add_mon_parser.add_argument('montype')
    add_mon_parser.add_argument('monname')

    args = add_mon_parser.parse_args()

    if config.mon['use_cnt'] == config.mon['max_cnt']:
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_cnt'].format('monitor'))

    if args.montype not in config.mon['all_type']:
        add_mon_parser.error(MESSAGE['err_arg_not_exist'].format('montype', args.montype))

    if len(args.monname) >= RSCNAME_LEN:
        add_mon_parser.error(MESSAGE['err_arg_length'].format('monname', args.monname, RSCNAME_LEN - 1))

    if args.monname in config.mon['name']:
        add_mon_parser.error(MESSAGE['err_arg_exist'].format('monname', args.monname))

    cmdline = [CMD_CFSET, 'add', 'mon', args.montype, args.monname]
    execute_command(cmdline)


def _mod(mod_parser):
    global re_config
    global should_show_element
    mod_groups = mod_parser.add_mutually_exclusive_group()
    mod_parser.add_argument('-t', dest='path', metavar='path', nargs='?', required=True)
    mod_groups.add_argument('--set', dest='value', metavar='value')
    mod_parser.add_argument('--nocheck', action='store_true')
    mod_groups.add_argument('--delete', action='store_true')

    args = mod_parser.parse_args()

    if args.nocheck and (args.value is None or args.delete):
        pass  # ignore

    if args.path:
        args.path = args.path.strip('/')

    # show the list of child elements
    if args.value is None and not args.delete:
        should_show_element = True
        return config.show_child_elements(args.path)

    # check validity of path and value
    elif args.value is not None:
        config.check_path(args.path, action=ACTION_ADD, check=not args.nocheck)

        if len(args.value) >= CONTENTS_LEN:
            mod_parser.error(MESSAGE['err_arg_length'].format('value', args.value, CONTENTS_LEN - 1))

        config.check_refer_object(args.path, args.value)

        if CMD_RSCDEP_REGEX.search(args.path):
            rsctype, rscname = args.path.split('/')[1].split('@')
            cmdline = [CMD_CFSET, 'add', 'rscdep', rsctype, rscname, args.value]

        elif ADD_ATTR_AND_BLANK_TAG_REGEX.search(args.path):
            cmdline = [CMD_CFSET, 'add', 'clsparam', args.path + '@' + args.value, '']

        else:
            cmdline = [CMD_CFSET, 'add', 'clsparam', args.path, args.value]

    else:  # args.delete
        config.check_path(args.path, action=ACTION_DEL)
        cmdline = [CMD_CFSET, 'del', 'clsparam', args.path]

    # exec command
    execute_command(cmdline)

    # 活性・非活性タイムアウトの再計算(対象リソースの場合)
    calc_rsc = {
        "appli": set_act_deact_timeout_appli,
        "awsdns": set_act_deact_timeout_awsdns,
        "awseip": set_act_deact_timeout_awseip,
        "awssip": set_act_deact_timeout_awssip,
        "awsvip": set_act_deact_timeout_awsvip,
        "azuredns": set_act_deact_timeout_azuredns,
        "exec": set_act_deact_timeout_exec,
        "fip": set_act_deact_timeout_fip,
        "script": set_act_deact_timeout_script,
        "service": set_act_deact_timeout_service,
        "disk": set_act_deact_timeout_disk
    }
    expr = re.search(r'^resource/[^/]+@[^/]+/', args.path)
    if expr is None:
        logging.debug("Not a recalculated resource.")
        return

    rsc_base_path = expr.group()
    rsc_type = re.search(r'/.+@', rsc_base_path).group()[1:-1]
    if rsc_type in calc_rsc:
        logging.debug("Recalculate active, inactive or both timeouts.")
        logging.debug(f" - rsc: {rsc_type}")
        re_config = Config()
        act_func = calc_rsc[rsc_type]
        act_func(rsc_base_path)
        return
    logging.debug("Not a recalculated resource.")


def set_act_deact_timeout_appli(rsc_base_path):
    path_list = {
        "param_acttimeout": rsc_base_path + "parameters/acttimeout",
        "param_deacttimeout": rsc_base_path + "parameters/deacttimeout"
    }
    value = re_config.get_node_value(path_list)
    param_acttimeout = int(value["param_acttimeout"])
    acttimeout = param_acttimeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    param_deacttimeout = int(value["param_deacttimeout"])
    deacttimeout = param_deacttimeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_awsdns(rsc_base_path):
    path_list = {
        "awsclitimeout": rsc_base_path + "parameters/awsclitimeout",
        "waitinsync": rsc_base_path + "parameters/waitinsync",
        "waitinsynctimeout": rsc_base_path + "parameters/waitinsynctimeout"
    }
    value = re_config.get_node_value(path_list)
    awsclitimeout = int(value["awsclitimeout"])
    waitinsync = int(value["waitinsync"])
    waitinsynctimeout = int(value["waitinsynctimeout"])
    acttimeout = awsclitimeout + (waitinsynctimeout if waitinsync else 0) + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    deacttimeout = int(awsclitimeout) + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_awseip(rsc_base_path):
    path_list = {
        "awsclitimeout": rsc_base_path + "parameters/awsclitimeout"
    }
    value = re_config.get_node_value(path_list)
    awsclitimeout = int(value["awsclitimeout"])
    acttimeout = 2 * awsclitimeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    deacttimeout = awsclitimeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_awssip(rsc_base_path):
    srvos = re_config.get_node_value({"srvos": "all/serveros"})["srvos"]
    if srvos == "linux":
        path_list = {
            "start_timeout": rsc_base_path + "parameters/timeout/start",
            "stop_timeout": rsc_base_path + "parameters/timeout/stop"
        }
    else:
        path_list = {
            "start_timeout": rsc_base_path + "parameters/acttimeout",
            "stop_timeout": rsc_base_path + "parameters/deacttimeout"
        }
    value = re_config.get_node_value(path_list)
    start_timeout = int(value["start_timeout"])
    acttimeout = start_timeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    stop_timeout = int(value["stop_timeout"])
    deacttimeout = stop_timeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_awsvip(rsc_base_path):
    srvos = re_config.get_node_value({"srvos": "all/serveros"})["srvos"]
    if srvos == "linux":
        path_list = {
            "start_timeout": rsc_base_path + "parameters/timeout/start",
            "stop_timeout": rsc_base_path + "parameters/timeout/stop"
        }
    else:
        path_list = {
            "start_timeout": rsc_base_path + "parameters/acttimeout",
            "stop_timeout": rsc_base_path + "parameters/deacttimeout"
        }
    value = re_config.get_node_value(path_list)
    start_timeout = int(value["start_timeout"])
    acttimeout = start_timeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    stop_timeout = int(value["stop_timeout"])
    deacttimeout = stop_timeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_azuredns(rsc_base_path):
    path_list = {
        "azureclitimeout": rsc_base_path + "parameters/azureclitimeout"
    }
    value = re_config.get_node_value(path_list)
    azureclitimeout = int(value["azureclitimeout"])
    acttimeout = 7 * azureclitimeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    deacttimeout = 3 * azureclitimeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_exec(rsc_base_path):
    path_list = {
        "start_timeout": rsc_base_path + "parameters/timeout/start",
        "stop_timeout": rsc_base_path + "parameters/timeout/stop"
    }
    value = re_config.get_node_value(path_list)
    start_timeout = int(value["start_timeout"])
    acttimeout = start_timeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    stop_timeout = int(value["stop_timeout"])
    deacttimeout = stop_timeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_fip(rsc_base_path):
    srvos = re_config.get_node_value({"srvos": "all/serveros"})["srvos"]
    if srvos == "windows":
        set_act_timeout_fip_windows(rsc_base_path)
    else:
        set_act_deact_timeout_fip_linux(rsc_base_path)


def set_act_timeout_fip_windows(rsc_base_path):
    path_list = {
        "pinginterval": rsc_base_path + "parameters/pinginterval",
        "pingtimeout": rsc_base_path + "parameters/pingtimeout",
        "pingretry": rsc_base_path + "parameters/pingretry",
        "waittimeout": rsc_base_path + "parameters/waittimeout"
    }
    value = re_config.get_node_value(path_list)
    pinginterval = int(value["pinginterval"])
    pingtimeout = int(value["pingtimeout"])
    pingretry = int(value["pingretry"])
    waittimeout = int(value["waittimeout"])
    acttimeout = int(pingtimeout / 1000) * (pingretry + 1) \
                    + pinginterval * pingretry + waittimeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_fip_linux(rsc_base_path):
    path_list = {
        "ifconfig_timeout": rsc_base_path + "parameters/ifconfig/timeout",
        "ping_interval": rsc_base_path + "parameters/ping/interval",
        "ping_timeout": rsc_base_path + "parameters/ping/timeout",
        "ping_retry": rsc_base_path + "parameters/ping/retry",
        "arp_retry": rsc_base_path + "parameters/arp/retry",
        "chk_ifconfig_retry": rsc_base_path + "parameters/check/ifconfig/retry",
        "chk_ifconfig_interval": rsc_base_path + "parameters/check/ifconfig/interval",
        "chk_ifconfig_exe": rsc_base_path + "parameters/check/ifconfig/execute",
        "chk_ping_interval": rsc_base_path + "parameters/check/ping/interval",
        "chk_ping_retry": rsc_base_path + "parameters/check/ping/retry",
        "chk_ping_exe": rsc_base_path + "parameters/check/ping/execute"
    }
    value = re_config.get_node_value(path_list)
    ifconfig_timeout = int(value["ifconfig_timeout"])
    ping_interval = int(value["ping_interval"])
    ping_timeout = int(value["ping_timeout"])
    ping_retry = int(value["ping_retry"])
    arp_retry = int(value["arp_retry"])
    acttimeout = \
        ifconfig_timeout * 2 \
        + ping_timeout * (ping_retry + 1) \
        + ping_interval * ping_retry \
        + arp_retry \
        + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    chk_ifconfig_retry = int(value["chk_ifconfig_retry"])
    chk_ifconfig_interval = int(value["chk_ifconfig_interval"])
    chk_ifconfig_exe = int(value["chk_ifconfig_exe"])
    chk_ping_interval = int(value["chk_ping_interval"])
    chk_ping_retry = int(value["chk_ping_retry"])
    chk_ping_exe = int(value["chk_ping_exe"])
    deacttimeout = \
        ifconfig_timeout * 2 \
        + (ifconfig_timeout * (chk_ifconfig_retry + 1) \
        + chk_ifconfig_interval * chk_ifconfig_retry) * chk_ifconfig_exe \
        + (chk_ping_retry + 1 + chk_ping_interval * chk_ping_retry) * chk_ping_exe \
        + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_script(rsc_base_path):
    path_list = {
        "param_acttimeout": rsc_base_path + "parameters/acttimeout",
        "param_deacttimeout": rsc_base_path + "parameters/deacttimeout"
    }
    value = re_config.get_node_value(path_list)
    param_acttimeout = int(value["param_acttimeout"])
    acttimeout = param_acttimeout + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    param_deacttimeout = int(value["param_deacttimeout"])
    deacttimeout = param_deacttimeout + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_service(rsc_base_path):
    path_list = {
        "param_acttimeout": rsc_base_path + "parameters/acttimeout",
        "param_actwaittime": rsc_base_path + "parameters/actwaittime",
        "param_deacttimeout": rsc_base_path + "parameters/deactwaittime",
        "param_deactwaittime": rsc_base_path + "parameters/deacttimeout"
    }
    value = re_config.get_node_value(path_list)
    param_acttimeout = int(value["param_acttimeout"])
    param_actwaittime = int(value["param_actwaittime"])
    acttimeout = param_acttimeout + param_actwaittime + 180
    acttimeout_path = rsc_base_path + "act/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)

    param_deacttimeout = int(value["param_deacttimeout"])
    param_deactwaittime = int(value["param_deactwaittime"])
    deacttimeout = param_deacttimeout + param_deactwaittime + 180
    deacttimeout_path = rsc_base_path + "deact/timeout"
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def set_act_deact_timeout_disk(rsc_base_path):
    path_list = {
        "disk_type": rsc_base_path + "parameters/disktype",
        "fsck_xfstimeout": rsc_base_path + "parameters/fsck/xfstimeout",
        "fsck_fixopt": rsc_base_path + "parameters/fsck/fixopt",
        "fsck_timeout": rsc_base_path + "parameters/fsck/timeout",
        "fsck_rebuildopt": rsc_base_path + "parameters/fsck/rebuildopt",
        "disable_disk_quota": rsc_base_path + "parameters/disable/disk/quota",
        "disable_disk_raw": rsc_base_path + "parameters/disable/disk/raw",
        "disable_disk_bind": rsc_base_path + "parameters/disable/disk/bind",
        "disable_lvm_roset": rsc_base_path + "parameters/disable/lvm/roset",
        "disable_lvm_quota": rsc_base_path + "parameters/disable/lvm/quota",
        "disable_lvm_raw": rsc_base_path + "parameters/disable/lvm/raw",
        "disable_lvm_bind": rsc_base_path + "parameters/disable/lvm/bind",
        "roset_timeout": rsc_base_path + "parameters/roset/timeout",
        "mount_timeout": rsc_base_path + "parameters/mount/timeout",
        "mount_retry": rsc_base_path + "parameters/mount/retry",
        "relpath_timeout": rsc_base_path + "parameters/relpath/timeout",
        "raw_timeout": rsc_base_path + "parameters/raw/timeout",
        "quotacheck_timeout": rsc_base_path + "parameters/quotacheck/timeout",
        "quotaon_timeout": rsc_base_path + "parameters/quotaon/timeout",
        "bind_timeout": rsc_base_path + "parameters/bind/timeout",
        "bind_retry": rsc_base_path + "parameters/bind/retry",
        "quotaoff_timeout": rsc_base_path + "parameters/quotaoff/timeout",
        "unbind_timeout": rsc_base_path + "parameters/unbind/timeout",
        "unbind_retry": rsc_base_path + "parameters/unbind/retry",
        "unmount_timeout": rsc_base_path + "parameters/umount/timeout",
        "unmount_retry": rsc_base_path + "parameters/umount/retry",
        "unmount_interval": rsc_base_path + "parameters/umount/interval",
        "enable_exec_unbind": rsc_base_path + "parameters/unbind/execute"
    }
    value = re_config.get_node_value(path_list)
    disk_type = value["disk_type"]
    fsck_xfstimeout = int(value["fsck_xfstimeout"])
    fsck_fixopt = int(value["fsck_fixopt"])
    fsck_timeout = int(value["fsck_timeout"])
    fsck_rebuildopt = int(value["fsck_rebuildopt"])
    disable_disk_quota = int(value["disable_disk_quota"])
    disable_disk_raw = int(value["disable_disk_raw"])
    disable_disk_bind = int(value["disable_disk_bind"])
    disable_lvm_roset = int(value["disable_lvm_roset"])
    disable_lvm_quota = int(value["disable_lvm_quota"])
    disable_lvm_raw = int(value["disable_lvm_raw"])
    disable_lvm_bind = int(value["disable_lvm_bind"])
    roset_timeout = int(value["roset_timeout"])
    mount_timeout = int(value["mount_timeout"])
    mount_retry = int(value["mount_retry"])
    relpath_timeout = int(value["relpath_timeout"])
    raw_timeout = int(value["raw_timeout"])
    quotacheck_timeout = int(value["quotacheck_timeout"])
    quotaon_timeout = int(value["quotaon_timeout"])
    bind_timeout = int(value["bind_timeout"])
    bind_retry = int(value["bind_retry"])
    quotaoff_timeout = int(value["quotaoff_timeout"])
    relpath_timeout = int(value["relpath_timeout"])
    unbind_timeout = int(value["unbind_timeout"])
    unbind_retry = int(value["unbind_retry"])
    unmount_timeout = int(value["unmount_timeout"])
    unmount_retry = int(value["unmount_retry"])
    unmount_interval = int(value["unmount_interval"])
    enable_exec_unbind = int(value["enable_exec_unbind"])
    disable_fsck = 0
    disable_relpath = 0
    disable_mount = 0

    acttimeout_path = rsc_base_path + "act/timeout"
    deacttimeout_path = rsc_base_path + "deact/timeout"

    if disk_type == "raw":
        acttimeout = roset_timeout * 2 + raw_timeout + bind_timeout + 180
        deacttimeout = unbind_timeout * (unbind_retry + 1) \
                        * enable_exec_unbind + roset_timeout + 180
    else:
        file_system_path = rsc_base_path + "parameters/fs"
        file_system = re_config.get_node_value(
            {"file_system": file_system_path}, is_empty_ok=True)["file_system"]
        if file_system is None:
            # ファイルシステムが未登録の場合は再計算を実施しない
            logging.debug("File system not registered.")
            return
        if file_system == "xfs":
            rebuild_reiserfs = False
            fsck_timeout = fsck_xfstimeout
        else:
            rebuild_reiserfs = fsck_fixopt

        rebuild_tree_option = (fsck_rebuildopt == 1)
        x = 2 if rebuild_tree_option or rebuild_reiserfs else 1

        if disk_type == "disk":
            disable_roset = 0
            disable_quota = disable_disk_quota
            disable_raw = disable_disk_raw
            disable_bind = disable_disk_bind
        else:
            disable_roset = disable_lvm_roset
            disable_quota = disable_lvm_quota
            disable_raw = disable_lvm_raw
            disable_bind = disable_lvm_bind

        acttimeout = \
            (1 - disable_roset) * (roset_timeout * 2) \
            + (1 - disable_fsck) * (fsck_timeout * (x)) \
            + (1 - disable_mount) * (mount_timeout * (1 + mount_retry)) \
            + (1 - disable_quota) * (quotacheck_timeout + quotaon_timeout) \
            + (1 - disable_raw) * raw_timeout \
            + (1 - disable_bind) * (bind_timeout * (1 + bind_retry)) \
            + 180

        deacttimeout = \
            (1 - disable_roset) * roset_timeout \
            + (1 - disable_mount) * (unmount_timeout * (1 + unmount_retry) \
                                    + (unmount_interval * unmount_retry)) \
            + (1 - disable_quota) * quotaoff_timeout \
            + (1 - disable_relpath) * (relpath_timeout * unmount_retry) \
            + (1 - disable_bind) * (unbind_timeout * (1 + unbind_retry)) \
            + 180
    # acttimeout
    cmdline = [CMD_CFSET, 'add', 'clsparam', acttimeout_path, acttimeout]
    execute_command(cmdline)
    # deacttimeout
    cmdline = [CMD_CFSET, 'add', 'clsparam', deacttimeout_path, deacttimeout]
    execute_command(cmdline)


def _del(del_parser):

    subparsers = del_parser.add_subparsers()
    del_srv_parser = subparsers.add_parser('srv', parents=[cmn_parser], help='see `del srv -h`')
    if config.serveros == 'windows':
        del_hba_parser = subparsers.add_parser('hba', parents=[cmn_parser], help='see `del hba -h`')
    del_device_parser    = subparsers.add_parser('device',    parents=[cmn_parser], help='see `del device -h`')
    if not IS_PRODUCT_SSS:
        del_forcestop_parser = subparsers.add_parser('forcestop', parents=[cmn_parser], help='see `del forcestop -h`')
    del_hb_parser = subparsers.add_parser('hb', parents=[cmn_parser], help='see `del hb -h`')
    del_np_parser = subparsers.add_parser('np', parents=[cmn_parser], help='see `del np -h`')
    del_grp_parser = subparsers.add_parser('grp', parents=[cmn_parser], help='see `del grp -h`')
    del_rsc_parser = subparsers.add_parser('rsc', parents=[cmn_parser], help='see `del rsc -h`')
    del_rscdep_parser    = subparsers.add_parser('rscdep',    parents=[cmn_parser], help='see `del rscdep -h`')
    del_mon_parser = subparsers.add_parser('mon', parents=[cmn_parser], help='see `del mon -h`')

    commands = {
        'srv': {'func': _del_srv, 'parser': del_srv_parser},
        'device': {'func': _del_device, 'parser': del_device_parser},
        'hb': {'func': _del_hb, 'parser': del_hb_parser},
        'np': {'func': _del_np, 'parser': del_np_parser},
        'grp': {'func': _del_grp, 'parser': del_grp_parser},
        'rsc': {'func': _del_rsc, 'parser': del_rsc_parser},
        'rscdep': {'func': _del_rscdep, 'parser': del_rscdep_parser},
        'mon': {'func': _del_mon, 'parser': del_mon_parser},
    }

    if config.serveros == 'windows':
        commands['hba'] = {'func': _del_hba, 'parser': del_hba_parser}
    if not IS_PRODUCT_SSS:
        commands['forcestop'] = {'func': _del_forcestop, 'parser': del_forcestop_parser}

    sys.argv = sys.argv[1:]
    if len(sys.argv) > 0 and sys.argv[0] in commands:
        commands[sys.argv[0]]['func'](commands[sys.argv[0]]['parser'])

    elif len(sys.argv) > 0 and ('-h' in sys.argv or '--help' in sys.argv):
        del_parser.print_help()

    else:
        del_parser.print_usage()


def _del_srv(del_srv_parser):

    del_srv_parser.add_argument('srvname')

    args = del_srv_parser.parse_args()

    if args.srvname not in config.srv['name']:
        del_srv_parser.error(MESSAGE['err_arg_not_exist'].format('srvname', args.srvname))

    cmdline = [CMD_CFSET, 'del', 'srv', args.srvname]
    execute_command(cmdline)


def _del_hba(del_hba_parser):

    del_hba_parser.add_argument('srvname')
    del_hba_parser.add_argument('hbaid', type=_is_positive_int)

    args = del_hba_parser.parse_args()

    if args.srvname not in config.srv['name']:
        del_hba_parser.error(MESSAGE['err_arg_not_exist'].format('srvname', args.srvname))

    if args.hbaid not in config.srv[args.srvname]['hbaid']:
        del_hba_parser.error(MESSAGE['err_arg_not_exist'].format('hbaid', args.hbaid))

    cmdline = [CMD_CFSET, 'del', 'hba', args.srvname, args.hbaid]
    execute_command(cmdline)


def _del_device(del_device_parser):

    del_device_parser.add_argument('srvname')
    del_device_parser.add_argument('devid', type=_is_positive_int)

    args = del_device_parser.parse_args()

    if args.srvname not in config.srv['name']:
        del_device_parser.error(MESSAGE['err_arg_not_exist'].format('srvname', args.srvname))

    if args.devid not in config.srv[args.srvname]['devid']:
        del_device_parser.error(MESSAGE['err_arg_not_exist'].format('devid', args.devid))

    cmdline = [CMD_CFSET, 'del', 'device', args.srvname, args.devid]
    execute_command(cmdline)


def _del_forcestop(del_forcestop_parser):

    del_forcestop_parser.parse_args()

    path = 'forcestop'
    config.check_exist_in_clpconf(path)

    cmdline = [CMD_CFSET, 'del', 'forcestop']
    execute_command(cmdline)


def _del_hb(del_hb_parser):

    del_hb_parser.add_argument('hbtype')
    del_hb_parser.add_argument('devid', type=_is_positive_int)

    args = del_hb_parser.parse_args()

    if args.hbtype not in config.hb['use_type']:
        del_hb_parser.error(MESSAGE['err_arg_not_exist'].format('hbtype', args.hbtype))

    if args.devid not in config.hb[args.hbtype]['devid']:
        del_hb_parser.error(MESSAGE['err_arg_not_exist'].format('devid', args.devid))

    cmdline = [CMD_CFSET, 'del', 'hb', args.hbtype, args.devid]
    execute_command(cmdline)


def _del_np(del_np_parser):

    del_np_parser.add_argument('nptype')
    del_np_parser.add_argument('devid', type=_is_positive_int)

    args = del_np_parser.parse_args()

    if args.nptype not in config.np['use_type']:
        del_np_parser.error(MESSAGE['err_arg_not_exist'].format('nptype', args.nptype))

    if args.devid not in config.np[args.nptype]['devid']:
        del_np_parser.error(MESSAGE['err_arg_not_exist'].format('devid', args.devid))

    cmdline = [CMD_CFSET, 'del', 'np', args.nptype, args.devid]
    execute_command(cmdline)


def _del_grp(del_grp_parser):

    del_grp_parser.add_argument('grpname')

    args = del_grp_parser.parse_args()

    if args.grpname not in config.grp['name']:
        del_grp_parser.error(MESSAGE['err_arg_not_exist'].format('grpname', args.grpname))

    cmdline = [CMD_CFSET, 'del', 'grp', args.grpname]
    execute_command(cmdline)


def _del_rsc(del_rsc_parser):

    del_rsc_parser.add_argument('grpname')
    del_rsc_parser.add_argument('rsctype')
    del_rsc_parser.add_argument('rscname')

    args = del_rsc_parser.parse_args()

    if args.grpname not in config.grp['name']:
        del_rsc_parser.error(MESSAGE['err_arg_not_exist'].format('grpname', args.grpname))

    if args.rsctype not in config.rsc['use_type']:
        del_rsc_parser.error(MESSAGE['err_arg_not_exist'].format('rsctype', args.rsctype))

    if args.rscname not in config.rsc[args.rsctype]['name'] or \
       args.rscname not in config.grp[args.grpname]['rscname']:
        del_rsc_parser.error(MESSAGE['err_arg_not_exist'].format('rscname', args.rscname))

    cmdline = [CMD_CFSET, 'del', 'rsc', args.grpname, args.rsctype, args.rscname]
    execute_command(cmdline)


def _del_rscdep(del_rscdep_parser):

    del_rscdep_parser.add_argument('rsctype')
    del_rscdep_parser.add_argument('rscname')

    args = del_rscdep_parser.parse_args()

    if args.rsctype not in config.rsc['use_type']:
        del_rscdep_parser.error(MESSAGE['err_arg_not_exist'].format('rsctype', args.rsctype))

    if args.rscname not in config.rsc[args.rsctype]['name']:
        del_rscdep_parser.error(MESSAGE['err_arg_not_exist'].format('rscname', args.rscname))

    path = 'resource/' + args.rsctype + '@' + args.rscname + '/depend'
    config.check_exist_in_clpconf(path)

    cmdline = [CMD_CFSET, 'del', 'rscdep', args.rsctype, args.rscname]
    execute_command(cmdline)


def _del_mon(del_mon_parser):

    del_mon_parser.add_argument('montype')
    del_mon_parser.add_argument('monname')

    args = del_mon_parser.parse_args()

    if args.montype not in config.mon['use_type']:
        del_mon_parser.error(MESSAGE['err_arg_not_exist'].format('montype', args.montype))

    if args.monname not in config.mon[args.montype]['name']:
        del_mon_parser.error(MESSAGE['err_arg_not_exist'].format('monname', args.monname))

    cmdline = [CMD_CFSET, 'del', 'mon', args.montype, args.monname]
    execute_command(cmdline)


def _delete_none_in_cmdline(func):
    def inner(*args):
        cmdline = [str(c) for c in args[0] if c is not None]
        return func(cmdline)

    return inner


def _make_backup_file(func):
    def inner(*args):
        import shutil
        backupfile = CLPCONF + '.bak'
        if is_delbakfile:
            os.remove(backupfile)
        if os.path.isfile(CLPCONF) and not os.path.isfile(backupfile):
            shutil.copy2(CLPCONF, backupfile)
        return func(args[0])
    return inner


@_delete_none_in_cmdline
@_make_backup_file
def execute_command(cmdline):
    logging.debug('execute command : {}'.format(cmdline))
    try:
        proc = subprocess.Popen(cmdline, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    except FileNotFoundError:
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_not_found_file'].format(CMD_CFSET))

    result = proc.communicate()
    _, stderr = result[0], result[1]
    retcode = proc.returncode
    logging.debug('retcode : {}'.format(retcode))

    if retcode != CFSET_RETCODE['success']:
        if retcode != CFSET_RETCODE['err_cmd']:  # command 'clpcfset' usage
            logging.debug(stderr.decode())
        # raise CfAdmError(CFADM_RETCODE['failure'], stderr.decode())
        raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_internal'])


def check_user():

    if os.name == 'nt':  # windows
        if not ctypes.windll.shell32.IsUserAnAdmin():
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_user_admin'])

    else:  # linux
        if os.geteuid() != 0 or os.getuid() != 0:
            raise CfAdmError(CFADM_RETCODE['failure'], MESSAGE['err_user_root'])


def main():

    global is_delbakfile
    global config
    global cmn_parser
    global should_show_element

    loglevel = logging.DEBUG if '--debug' in sys.argv else logging.WARNING
    logging.basicConfig(level=loglevel,
                        format='%(levelname)s - %(message)s',
                        stream=sys.stdout)
    is_delbakfile = "--backup" in sys.argv
    should_show_element = False

    try:
        check_user()

        os.environ['COLUMNS'] = '100'  # for argparse output columns
        cmn_parser = CfAdmArgumentParser(add_help=False)
        cmn_parser.add_argument('--debug', action='store_true', help=argparse.SUPPRESS)
        cmn_parser.add_argument('--backup', action='store_true')

        main_parser = CfAdmArgumentParser()
        subparsers = main_parser.add_subparsers()

        create_parser = subparsers.add_parser('create', parents=[cmn_parser], help='see `create -h`')
        add_parser = subparsers.add_parser('add', parents=[cmn_parser], help='see `add -h`')
        mod_parser    = subparsers.add_parser('mod',    parents=[cmn_parser], help='see `mod -h`', allow_abbrev=False)
        del_parser = subparsers.add_parser('del', parents=[cmn_parser], help='see `del -h`')

        commands = {
            'create': {'func': _create, 'parser': create_parser},
            'add': {'func': _add, 'parser': add_parser},
            'mod': {'func': _mod, 'parser': mod_parser},
            'del': {'func': _del, 'parser': del_parser},
        }

        sys.argv = sys.argv[1:]
        if len(sys.argv) > 0 and sys.argv[0] in commands:
            if sys.argv[0] != 'create':
                config = Config()

            commands[sys.argv[0]]['func'](commands[sys.argv[0]]['parser'])
            if not should_show_element:
                print(MESSAGE['success_cmd'])
            return CFADM_RETCODE['success']

        elif len(sys.argv) > 0 and ('-h' in sys.argv or '--help' in sys.argv):
            main_parser.print_help()

        else:
            main_parser.print_usage()

    except CfAdmError as err:
        print(err.msg, file=sys.stderr)
        return err.retcode

    except Exception:
        logging.debug(traceback.format_exc())
        print(MESSAGE['err_internal'], file=sys.stderr)
        return CFADM_RETCODE['failure']


if __name__ == '__main__':
    sys.dont_write_bytecode = True
    sys.exit(main())
