#/bin/sh

# cfget return code
CFGET_SUCCESS=0			# success
CFGET_ERR_POLICY=8		# error policy
CFGET_ERR_XMLEXIST=11	# xmlpath not exist

# fwctrl return code
FWCTRL_SUCCESS=0
FWCTRL_ERR_OPT=1
FWCTRL_ERR_INSTALL=2
FWCTRL_NOT_ROOT=3
FWCTRL_FIREWALLD_ERR_PORT=4
FWCTRL_FIREWALLD_ERR_ZONE=5
FWCTRL_ERR_GET_XML=126
FWCTRL_ERR_CMDNOTFOUND=127

# xml path
XML_INSTALLPATH="/root/all/path"	# install path
XML_SSS="/root/sss/type"			# sss

# typeA
PORT_API="/root/cluster/api/port"					# TCP
PORT_TRNS="/root/cluster/trns/port"					# TCP
PORT_HB="/root/cluster/heartbeat/port/recv"			# UDP
PORT_HTTP="/root/webmgr/http/port"					# TCP
PORT_ALERT="/root/webalert/daemon/udpport"			# UDP
PORT_MDAGENT="/root/mdagent/port"					# TCP 本体のみ
PORT_KHB="/root/cluster/heartbeat/khbport/recv"		# UDP 本体のみ
PORT_IB="/root/cluster/api/ibport"					# TCP
PORT_RSTD_HTTP="/root/cluster/rstd/http/port"		# TCP
PORT_RSTD_SERVICE="/root/cluster/rstd/service/port"	# TCP

# typeB
# ミラーディスクリソース
PORT_MD="/root/resource/md@{name}/parameters/mddriver/port"				# TCP
PORT_MD_HB="/root/resource/md@{name}/parameters/mddriver/hbport"		# TCP
PORT_MD_ACK2="/root/resource/md@{name}/parameters/mddriver/ack2port"	# TCP
# ハイブリッドディスクリソース
PORT_HD="/root/resource/hd@{name}/parameters/mddriver/port"				# TCP
PORT_HD_HB="/root/resource/hd@{name}/parameters/mddriver/hbport"		# TCP
PORT_HD_ACK2="/root/resource/hd@{name}/parameters/mddriver/ack2port"	# TCP
# Azureプローブポートリソース
PORT_AZURE="/root/resource/azurepp@{name}/parameters/probeport"			# TCP
# GoogleCloud仮想IPリソース
PORT_GC="/root/resource/gcvip@{name}/parameters/probeport"				# TCP
# OracleCloud仮想 IPリソース
PORT_OC="/root/resource/ocvip@{name}/parameters/probeport"				# TCP

# typeC
EVENT_METHOD="/root/cluster/event/method"
PORT_EVENT="/root/cluster/event/port"		# UDP
JRA_PATH="/root/jra/path/java"
PORT_JRA="/root/jra/admin/port"				# TCP

# check root
if [ ${EUID:-${UID}} != 0 ]; then
    echo "Log in as root"
    exit ${FWCTRL_NOT_ROOT}
fi

# Source function library.
if [ -f /etc/rc.d/init.d/functions ]
then
	. /etc/rc.d/init.d/functions
elif [ -f /etc/rc.status ]
then
	. /etc/rc.status
fi

. /opt/nec/clusterpro/bin/clpfunctions

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/opt/nec/clusterpro/bin
LD_LIBRARY_PATH=/opt/nec/clusterpro/lib
MALLOC_CHECK_=2

export PATH
export LD_LIBRARY_PATH
export MALLOC_CHECK_

targetos=`/bin/uname`
clp_filedel log fwctrl $targetos

#------------------------------------------------
#	functions
#------------------------------------------------

##
#	clpcfget command execution
# 	 arg1 : option(g:string, e:enm)
# 	 arg2 : xml path
#	 arg3 : policy type
##
function clpcfget_exe ()
{
	local cfget_opt=$1
	local xml_path=$2
	local policy_type=$3

	cfget=""
	cfget_exitcode=""
	local cmdline="clpcfget -${cfget_opt} ${xml_path}"
	if [ "${policy_type}" != "" ];
	then
		cmdline="${cmdline} -p ${policy_type}"
	fi
	cfget=`${cmdline}`
	cfget_exitcode=`echo $?`
}

##
#	check command exist
# 	 arg1 : command
##
function check_exist_cmd ()
{
	which $1 > /dev/null 2>&1
	if [ $? -ne 0 ]; then
		end_log "Unsupported environment." "${FWCTRL_ERR_CMDNOTFOUND}"
		exit ${FWCTRL_ERR_CMDNOTFOUND}
	fi
}

##
#	get typeA
##
function get_typeA ()
{
	xml_path_list_typeA=(
		${PORT_API}
		${PORT_TRNS}
		${PORT_HB}
		${PORT_HTTP}
		${PORT_ALERT}
		${PORT_MDAGENT}
		${PORT_KHB}
		${PORT_IB}
		${PORT_RSTD_HTTP}
		${PORT_RSTD_SERVICE}
	)

	for xml_path in ${xml_path_list_typeA[@]};
	do
		if [ ${PORT_HTTP} = ${xml_path} ]; then
			clpcfget_exe "g" "${xml_path}" "webmgr";
		elif [ ${PORT_ALERT} = ${xml_path} ]; then
			clpcfget_exe "g" "${xml_path}" "webalert";
		elif [ ${PORT_MDAGENT} = ${xml_path} ]; then
			if [ $sssFlg -eq 1 ]; then
				continue
			fi
			clpcfget_exe "g" "${xml_path}" "mdagent";
		else
			if [ $sssFlg -eq 1 -a ${PORT_KHB} = ${xml_path} ]; then
				continue
			fi
			clpcfget_exe "g" "${xml_path}" "cluster";
		fi

		if [ "${cfget_exitcode}" -ne "${CFGET_SUCCESS}" ]; then
			end_log "${cfget}(${xml_path})" "${FWCTRL_ERR_GET_XML}"
			exit ${FWCTRL_ERR_GET_XML}
		fi
		port_array["${xml_path}"]=${cfget}
	done
}

##
#	get typeB
# 	 arg1 : resource type
# 	 arg2 : xml path
##
function get_typeB ()
{
	local resource_type="/root/resource/$1"
	local xml_path=""
	# リソース名取得
	clpcfget_exe "e" "${resource_type}" "";

	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ]; then
		# リソース名からポート番号取得
		for resource_name in ${cfget};
		do
			xml_path=${2//\{name\}/${resource_name}}
			clpcfget_exe "g" "${xml_path}" "$1";
			if [ "${cfget_exitcode}" -ne "${CFGET_SUCCESS}" ]; then
				end_log "${cfget}(${xml_path})" "${FWCTRL_ERR_GET_XML}"
				exit ${FWCTRL_ERR_GET_XML}
			fi
			port_array["${xml_path}"]=${cfget}
		done
	elif [ ${cfget_exitcode} -eq ${CFGET_ERR_XMLEXIST} ]; then
		# リソースの登録がない場合は何もしない
		:
	else
		end_log "${cfget}(${resource_type})" "${FWCTRL_ERR_GET_XML}"
		exit ${FWCTRL_ERR_GET_XML}
	fi
}

##
#	get typeC
##
function get_typeC ()
{
	# /root/cluster/event/method が 1 の時に、ポートの許可設定をおこなう。(UDP)
	# ⇒/root/cluster/event/port 
	clpcfget_exe "g" "${EVENT_METHOD}" "cluster";
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ]; then
		if [ ${cfget} -eq 1 ]; then
			clpcfget_exe "g" "${PORT_EVENT}" "cluster";
			if [ "${cfget_exitcode}" -ne "${CFGET_SUCCESS}" ]; then
				end_log "${cfget}(${PORT_EVENT})" "${FWCTRL_ERR_GET_XML}"
				exit ${FWCTRL_ERR_GET_XML}
			fi
			port_array[${PORT_EVENT}]=${cfget}
		fi
	else
		end_log "${cfget}(${EVENT_METHOD})" "${FWCTRL_ERR_GET_XML}"
		exit ${FWCTRL_ERR_GET_XML}
	fi

	# /root/jra/path/java の値に有効なパスが設定されている時に、ポートの許可設定をおこなう。(TCP)
	# ⇒/root/jra/admin/port
	# JVM監視を設定したときのみパスのチェックを行う
	clpcfget_exe "g" "${JRA_PATH}" "";
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ]; then
		if [ -e "${cfget}" ]; then
			clpcfget_exe "g" "${PORT_JRA}" "jra";
			if [ "${cfget_exitcode}" -ne "${CFGET_SUCCESS}" ]; then
				end_log "${cfget}(${PORT_JRA})" "${FWCTRL_ERR_GET_XML}"
				exit ${FWCTRL_ERR_GET_XML}
			fi
			port_array[${PORT_JRA}]=${cfget}
		else
			end_log "Not exist java install path.(${cfget})" "${FWCTRL_ERR_GET_XML}"
			exit ${FWCTRL_ERR_GET_XML}
		fi
	elif [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ]; then
		# 何もしない
		:
	else
		end_log "${cfget}(${JRA_PATH})" "${FWCTRL_ERR_GET_XML}"
		exit ${FWCTRL_ERR_GET_XML}
	fi
}

##
#	make xml
##
function make_xml()
{
	echo "<?xml version=\"1.0\" encoding=\"utf-8\"?>" >> ${XMLFILE}
	echo "<service>" >> ${XMLFILE}
	echo "  <short>${PRODUCTNAME}</short>" >> ${XMLFILE}
	echo "  <description>This allows cluster services to communicate each other.</description>" >> ${XMLFILE}
	# 取得したポート番号をxmlファイルに追加
	clp_logwrite log "-----register port-----" fwctrl
	for key in ${!port_array[@]};
	do
		# echo "${key} : ${port_array[${key}]}"
		if [ ${PORT_HB} = ${key} -o ${PORT_ALERT} = ${key} -o ${PORT_KHB} = ${key} -o ${PORT_EVENT} = ${key} ]; then
			# UDP
			echo "  <port protocol=\"udp\" port=\"${port_array[${key}]}\"/> <!-- ${key} -->" >> ${XMLFILE}
			clp_logwrite log "${key} : ${port_array[${key}]}(UDP)" fwctrl
		else
			# TCP
			echo "  <port protocol=\"tcp\" port=\"${port_array[${key}]}\"/> <!-- ${key} -->" >> ${XMLFILE}
			clp_logwrite log "${key} : ${port_array[${key}]}(TCP)" fwctrl
		fi
	done
	clp_logwrite log "-----register port-----" fwctrl
	echo "</service>" >> ${XMLFILE}
}

##
#	register xml
##
function register_xml ()
{
	local fwCmd=""
	if [ ${fwFlg} -eq 0 ]; then
		fwCmd="firewall-offline-cmd"
	else
		fwCmd="firewall-cmd --permanent"
	fi
	# サービス削除
	${fwCmd} --delete-service=${PRODUCTNAME} > /dev/null 2>&1
	if [ ${fwFlg} -eq 1 ]; then
		firewall-cmd --reload > /dev/null 2>&1
	fi
	# サービス作成
	firewalldMsg=`${fwCmd} --new-service-from-file="${XMLFILE}" --name=${PRODUCTNAME} 2>&1 > /dev/null`
	firewalldRet=$?
	if [ ${firewalldRet} -ne 0 ]; then
		# port番号不正
		echo "Failed to register rule(${PRODUCTNAME}). Invalid port."
		clp_logwrite log "${firewalldMsg}" fwctrl
		clp_logwrite log "clpfwctrl end(${FWCTRL_FIREWALLD_ERR_PORT})" fwctrl
		exit ${FWCTRL_FIREWALLD_ERR_PORT}
	fi
	# サービス追加
	firewalldMsg=`${fwCmd} --add-service=${PRODUCTNAME} ${optZone} 2>&1 > /dev/null`
	firewalldRet=$?
	if [ ${firewalldRet} -ne 0 ]; then
		# ZONE名不正
		echo "Failed to register rule(${PRODUCTNAME}). Invalid zone."
		clp_logwrite log "${firewalldMsg}" fwctrl
		clp_logwrite log "clpfwctrl end(${FWCTRL_FIREWALLD_ERR_ZONE})" fwctrl
		exit ${FWCTRL_FIREWALLD_ERR_ZONE}
	fi
	if [ ${fwFlg} -eq 1 ]; then
		firewall-cmd --reload > /dev/null 2>&1
	fi

	# 設定状況確認
	if [ ${fwFlg} -eq 0 ]; then
		checkRule=`firewall-offline-cmd --list-services ${optZone}` > /dev/null 2>&1
	else
		checkRule=`firewall-cmd --list-services ${optZone}` > /dev/null 2>&1
	fi
	clp_logwrite log "list services(${checkRule})" fwctrl
}

##
#	print_usage	--help -h
##
function print_usage()
{
	echo ""
	echo "Usage:"
	echo "  clpfwctrl.sh --add                : add"
	echo "  clpfwctrl.sh --add --zone=<ZONE>  : add"
	echo "  clpfwctrl.sh --remove             : remove"
	echo ""
}

##
#	print end log
# 	 arg1 : message
# 	 arg2 : exit code
##
function end_log()
{
	echo "$1"
	clp_logwrite log "$1" fwctrl
	clp_logwrite log "clpfwctrl end($2)" fwctrl
}

# functions End

#------------------------------------------------
# Main
#------------------------------------------------

fwFlg=0			# 0:fw stop, 1:fw running
optFlg=0		# 0:add, 1:remove
sssFlg=0		# 0:本体, 1:SSS
optZone=""
firewalldMsg=""
firewalldRet=0

clp_logwrite log "$0 $* start" fwctrl
# コマンド存在チェック
check_exist_cmd "firewall-cmd";
check_exist_cmd "firewall-offline-cmd";

# 引数チェック
if [ "$#" -ge 3 ]
then
	end_log "Invalid option." "${FWCTRL_ERR_OPT}"
	exit ${FWCTRL_ERR_OPT}
fi

if [ "$1" = "--add" ]; then
	if [ "$2" != "" ]; then
		# --zone or --profile のチェック
		if [ "--zone=" = "${2:0:7}" ]; then
			optZone=$2
		elif [ "--profile=" = "${2:0:10}" ]; then
			optZone=${2//--profile/--zone}
		else
			end_log "Invalid option." "${FWCTRL_ERR_OPT}"
			exit ${FWCTRL_ERR_OPT}
		fi
	fi
	optFlg=0
elif [ "$1" = "--remove" ]; then
	optFlg=1
elif [ "$1" = "--help" -o "$1" = "-h" ]; then
	print_usage
	clp_logwrite log "print usage." fwctrl
	clp_logwrite log "clpfwctrl end(${FWCTRL_SUCCESS})" fwctrl
	exit ${FWCTRL_SUCCESS}
else
	end_log "Invalid option." "${FWCTRL_ERR_OPT}"
	exit ${FWCTRL_ERR_OPT}
fi

# インストールパス
clpcfget_exe "g" "${XML_INSTALLPATH}" "all";
if [ "${cfget_exitcode}" -ne "${CFGET_SUCCESS}" ]; then
	end_log "${cfget}(${XML_INSTALLPATH})" "${FWCTRL_ERR_GET_XML}"
	exit ${FWCTRL_ERR_GET_XML}
fi
INSTALLPATH=${cfget}
# 製品名
PRODUCTNAME=`basename ${INSTALLPATH}`
# workディレクトリ
WORKDIR=${INSTALLPATH}/work/fwctrl
# xmlファイル
XMLFILE=${WORKDIR}/${PRODUCTNAME}.xml

# 本体 or SSS のチェック
clpcfget_exe "g" "${XML_SSS}" "sss";
if [ "${cfget_exitcode}" -eq "${CFGET_ERR_POLICY}" -o "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ]; then
	:
elif [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ]; then
	if [ "sss" = "${cfget}" ]; then
		sssFlg=1
	else
		end_log "${cfget}(${XML_SSS})" "${FWCTRL_ERR_INSTALL}"
		exit ${FWCTRL_ERR_INSTALL}
	fi
else
	end_log "${cfget}(${XML_SSS})" "${FWCTRL_ERR_GET_XML}"
	exit ${FWCTRL_ERR_GET_XML}
fi

# firewalld起動確認
firewall-cmd --state > /dev/null 2>&1
if [ $? -ne 0 ]; then
	clp_logwrite log "firewalld is not running." fwctrl
	fwFlg=0
else
	clp_logwrite log "firewalld is running." fwctrl
	fwFlg=1
fi

if [ ${optFlg} -eq 1 ]; then # --remove の場合
	if [ ${fwFlg} -eq 0 ]; then
		firewall-offline-cmd --delete-service=${PRODUCTNAME} > /dev/null 2>&1
	else
		firewall-cmd --permanent --delete-service=${PRODUCTNAME} > /dev/null 2>&1
		firewall-cmd --reload > /dev/null 2>&1
	fi
else # --add の場合
	# フォルダチェック
	if [ -d ${WORKDIR} ]; then
		rm -rf ${WORKDIR}
	fi
	mkdir -p ${WORKDIR}

	# ポート番号取得用連想配列
	declare -A port_array;

	# typeA
	get_typeA;

	# typeB
	if [ ${sssFlg} -ne 1 ]; then
		# ミラーディスクリソース
		get_typeB "md" "${PORT_MD}";
		get_typeB "md" "${PORT_MD_HB}";
		get_typeB "md" "${PORT_MD_ACK2}";
		# ハイブリッドディスクリソース
		get_typeB "hd" "${PORT_HD}";
		get_typeB "hd" "${PORT_HD_HB}";
		get_typeB "hd" "${PORT_HD_ACK2}";
		# Azureプローブポートリソース
		get_typeB "azurepp" "${PORT_AZURE}";
		# GoogleCloud仮想IPリソース
		get_typeB "gcvip" "${PORT_GC}";
		# OracleCloud仮想 IPリソース
		get_typeB "ocvip" "${PORT_OC}";
	fi

	# typeC
	get_typeC;

	# xmlファイル作成
	make_xml;

	# xmlファイル登録
	register_xml;
fi
end_log "Command succeeded." "${FWCTRL_SUCCESS}"
exit ${FWCTRL_SUCCESS}

# Main End
