#!/bin/bash
##
#	Validation Check
##

#------------------------------------------------
#	command exit code
#------------------------------------------------
EXITCODE_CMD_INVOKED_CANNOT_EXECUTE=126
EXITCODE_CMD_NOT_FOUND=127

#------------------------------------------------
#	return code
#------------------------------------------------
RET_SUCCESS=0	# success
RET_FAIL_OPT=1	# option error


#------------------------------------------------
#	functions
#------------------------------------------------
# function return code
FUNC_RET_SUCCESS=0
FUNC_RET_FAIL=1

##
#	Read Message File
##
declare -a msg_table_id=()
declare -a msg_table_item=()
declare -a msg_table_msg1=()
declare -a msg_table_msg2=()
already_read=false

function read_msg_file ()
{
	local i=0
	local wk_id
	local wk_item
	local wk_msg1
	local wk_msg2

	item=""
	msg1=""
	msg2=""

	if ! "${already_read}";
	then
		while read line;
		do
			if [[ "${line}" =~ ^[\s\t]*ID[\s\t]*=[\s\t]*([0-9]+)[\s\t]*$ ]];
			then
				# set message for ID, ITEM, MSG1, MSG2

				# ID
				msg_table_id=("${msg_table_id[@]}" "${line#ID=}")

				# ITEM
				read wk_item
				msg_table_item=("${msg_table_item[@]}" "${wk_item#ITEM=}")

				# MSG1
				read wk_msg1
				msg_table_msg1=("${msg_table_msg1[@]}" "${wk_msg1#MSG1=}")

				# MSG2
				read wk_msg2
				msg_table_msg2=("${msg_table_msg2[@]}" "${wk_msg2#MSG2=}")
			fi
		done < ${msgfpath}

		if [ `echo ${#msg_table_id[@]}` -gt 0 ];
		then
			already_read=true
		fi
	fi

	for wk_id in ${msg_table_id[@]};
	do
		if [ "${id}" == "${wk_id}" ];
		then
			# get messages for ID from message table
			item=${msg_table_item[i]}
			msg1=${msg_table_msg1[i]}
			msg2=${msg_table_msg2[i]}
			break
		fi
		i=$(( i + 1 ))
	done
}


##
#	clpcfget command execution
#	 arg1 : option(g:string, e:enum)
#	 arg2 : policy type
#	 arg3 : server name
##
CFGET_SUCCESS=0		# success
CFGET_ERR_XMLEXIST=11	# xmlpath not exist

function clpcfget_exe ()
{
	local cfget_opt=$1
	local policy_type=$2
	local server_name=$3

	cfget=""
	cfget_exitcode=""
	local cmdline="clpcfget -${cfget_opt} ${xml_path} -i ${ipath}"
	if [ "${policy_type}" != "" ];
	then
		cmdline="${cmdline} -p ${policy_type}"
	fi
	if [ "${server_name}" != "" ];
	then
		cmdline="${cmdline} -s ${server_name}"
	fi
	cfget=`${cmdline}`
	cfget_exitcode=`echo $?`
}


##
#	Set Message : Could Not Check
##
function set_msg_couldnotchk ()
{
	local rep_msg=$1
	result_summary="FALSE"
	result="-"
	if [ "${rep_msg}" != "" ];
	then
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rep_msg}|"`
	fi
	result_msg2=${checkerr_msg}
}

##
#	Set Message : Not Check Target
##
function set_msg_notchktarget ()
{
	local rep_msg=$1
	result="-"
	if [ "${rep_msg}" != "" ];
	then
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rep_msg}|"`
	fi
	result_msg2=${notchktarget_msg}
}


##
#	Output Message : result summary [ID,Server,,Msg1,Result]
##
function output_msg_result_summary ()
{
	local wk_result=""
	case ${result_summary} in
		"TRUE" ) wk_result="PASS" ;;
		"FALSE") wk_result="FAIL" ;;
		*      ) wk_result="-"    ;;
	esac
	echo "\"${id}\",\"${server}\",\"\",\"${msg1}\",\"${wk_result}\"" >> "${output}"
}

##
#	Output Message : datetime [ID,Server,,Msg1,Datetime]
##
function output_msg_datetime ()
{
	local datetime=`date '+%Y/%m/%d %H:%M:%S'`
	echo "\"${id}\",\"${server}\",\"\",\"${msg1}\",\"${datetime}\"" >> "${output}"
}

##
#	Output Message : normal [ID,Server,Item,Msg1,Result,Msg2]
#	 arg1 : output msg1
#	 arg2 : output msg2(error msg)
##
function output_msg ()
{
	local wk_outputmsg1=$1
	local wk_outputmsg2=$2
	local wk_result=""
	case ${result} in
		"TRUE" ) wk_result="PASS"
		         wk_outputmsg2="" ;;
		"FALSE") wk_result="FAIL" ;;
		*      ) wk_result="-"    ;;
	esac

	# remove last $newline
	wk_outputmsg2="${wk_outputmsg2%${newline}}"

	echo "\"${id}\",\"${server}\",\"${item}\",\"${wk_outputmsg1}\",\"${wk_result}\",\"${wk_outputmsg2}\"" >> "${output}"
}


##
#	Check
##

## ping
##  arg1 : IP address
function ping_check ()
{
	local ipaddr=$1
	local ping_exitcode=""
	local ping6_exitcode=""

	# ping : ipv4
	ping ${ipaddr} -c 3 > /dev/null 2>&1
	ping_exitcode=`echo $?`
	if [ "${ping_exitcode}" -eq 0 ];
	then
		return ${FUNC_RET_SUCCESS}
	elif [ "${ping_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${ping_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${ping_exitcode}
	else
		# ping : ipv6
		ping6 ${ipaddr} -c 3 > /dev/null 2>&1
		ping6_exitcode=`echo $?`
		if [ "${ping6_exitcode}" -eq 0 ];
		then
			return ${FUNC_RET_SUCCESS}
		elif [ "${ping6_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${ping6_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			return ${ping6_exitcode}
		fi
	fi

	return ${FUNC_RET_FAIL}
}

## port number
##  arg1 : port number
function port_check ()
{
	local port=$1
	local eport_min
	local eport_max
	local rports
	local trash
	local sysctl_exitcode=""

	# 不正なポート番号ならエラー
	if [ ! "${port}" -gt 0 ];
	then
		return ${FUNC_RET_FAIL}
	fi

	# エフェメラルポートの範囲取得
	sysctl net.ipv4.ip_local_port_range > /dev/null 2>&1
	sysctl_exitcode=`echo $?`
	if [ "${sysctl_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${sysctl_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${sysctl_exitcode}
	fi
	read trash trash eport_min eport_max <<< `sysctl net.ipv4.ip_local_port_range`

	if [ ! "${eport_min}" -ge 0 -a "${eport_max}" -ge 0 ];
	then
		# 取得失敗なら次に進む
		:
	elif [ ! "${port}" -ge "${eport_min}" -a "${port}" -le "${eport_max}" ];
	then
		# 対象ポートが範囲外なら正常終了
		return ${FUNC_RET_SUCCESS}
	fi

	# 予約ポートの取得
	sysctl net.ipv4.ip_local_reserved_ports > /dev/null 2>&1
	sysctl_exitcode=`echo $?`
	if [ "${sysctl_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${sysctl_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${sysctl_exitcode}
	fi
	read trash trash rports <<< `sysctl net.ipv4.ip_local_reserved_ports`

	# 予約ポートとの比較
	while read -d, rport;
	do
		# 対象ポートが予約済みなら正常終了
		echo ${rport} | grep "-" > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			# 単一ポート
			if [ "${port}" == "${rport}" ];
			then
				return ${FUNC_RET_SUCCESS}
			fi
		else
			# 範囲指定
			rport1=`echo ${rport} | awk -F- '{print $1}'`
			rport2=`echo ${rport} | awk -F- '{print $2}'`
			if [ "${rport1}" -le "${port}" -a "${port}" -le "${rport2}" ];
			then
				return ${FUNC_RET_SUCCESS}
			fi
		fi
	done <<< "${rports},"

	return ${FUNC_RET_FAIL}
}

## device
#  block device
#   arg1 : device name
function block_device_check ()
{
	local dev=$1
	if [ ! -b "${dev}" ];
	then
		return ${FUNC_RET_FAIL}
	fi
	return ${FUNC_RET_SUCCESS}
}
#  character device
#   arg1 : device name
function character_device_check ()
{
	local dev=$1
	if [ ! -c "${dev}" ];
	then
		return ${FUNC_RET_FAIL}
	fi
	return ${FUNC_RET_SUCCESS}
}

## kernel
function kernel_check ()
{
	local kernel_ver=`uname -r`
	local drv_dir="/opt/nec/clusterpro/drivers"
	local ka_dir
	local khb_dir
	local md_dir

	# check ka driver
	ka_dir="${drv_dir}/ka/distribution/current"
	if [ -d "${ka_dir}" ];
	then
		if [ ! -f "${ka_dir}/clpka-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	# check khb driver
	khb_dir="${drv_dir}/khb/distribution/current"
	if [ -d "${khb_dir}" ];
	then
		if [ ! -f "${khb_dir}/clpkhb-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	# check liscal driver
	md_dir="${drv_dir}/md/distribution/current"
	if [ -d "${md_dir}" ];
	then
		if [ ! -f "${md_dir}/liscal-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}

## file system
#  disk
#   arg1 : device name
#   arg2 : file system
function disk_file_system_check ()
{
	local dev=$1
	local fs=$2
	local parted_exitcode=""

	# check file system
	#  -m : Number:Start:End:Size:File system:Name:Flags
	parted -ms ${dev} p | grep -E "([0-9]+):(.*):(.*):(.*):(${fs}):(.*):(.*)" > /dev/null 2>&1
	if [ $? -ne 0 ];
	then
		lsblk -dnf ${dev} | grep ${fs} > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}


#------------------------------------------------
# Main
#------------------------------------------------
export LANG=C

## get option
ipath=""
opath=""
while getopts ":i:o:" opt;
do
	case ${opt} in
		i ) ipath=${OPTARG} ;;		# clp.conf
		o ) opath=${OPTARG} ;;		# output dir
		\?) exit ${RET_FAIL_OPT} ;;
	esac
done

## server, server list
server_fullname=`uname -n`		# get hostname (use output file path)
server=`echo ${server_fullname%%.*}`	# delete domain name

# enum server
xml_path="/root/server"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	server_list=${cfget}
else
	server_list=""
fi

## output file path
output="${opath}/${server_fullname}.csv"

## message string
result_summary="TRUE"		# 総合結果("TRUE" or "FALSE")
newline="<br/>"			# 改行文字
checkerr_msg=""			# メッセージ:エラー
notchktarget_msg=""		# メッセージ:チェック対象外
msg_cp="CP"			# クラスタパーティション
msg_dp="DP"			# データパーティション
rep="!REP!"			# 置換文字

## message file path
msgfpath="/opt/nec/clusterpro/etc/validation/resultmsg.us"

# get value (charset)
xml_path="/root/all/charset"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ "${cfget}" == "EUC-JP" ];
	then
		msgfpath="/opt/nec/clusterpro/etc/validation/resultmsg.jp"
	fi
fi

#--------------------
# ID : 3
#  get message(check error)
#--------------------
id=3
read_msg_file
checkerr_msg=${msg2}

#--------------------
# ID : 4
#  get message(not check target)
#--------------------
id=4
read_msg_file
notchktarget_msg=${msg2}

#--------------------
# ID : 1
#  start time
#--------------------
id=1
read_msg_file
output_msg_datetime

#--------------------
# ID : 1000
#  check NP resolution ip address(ping)
#--------------------
id=1000
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""

# enum pingnp
xml_path="/root/networkpartition/pingnp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	output_flg=true
	for pingnp_name in ${cfget};	# for : 1
	do
		# get value (device)
		xml_path="/root/networkpartition/pingnp@${pingnp_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			# get value (info)
			xml_path="/root/server@${server}/device@${cfget}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					# enum grp
					xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp"
					clpcfget_exe "e"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						for grp_id in ${cfget};	# for : 2
						do
							# enum list
							xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list"
							clpcfget_exe "e"
							if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
							then
								for list_id in ${cfget};	# for : 3
								do
									# get value (IP address)
									xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list@${list_id}/ip"
									clpcfget_exe "g"
									if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
									then
										ping_check "${cfget}"
										exitcode=`echo $?`
										if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
										then
											# failed command execute
											checkerr_flg=true
											break 3		# -> for : 1
										elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
										then
											result_summary="FALSE"
											result="FALSE"
											result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
										fi
									else
										# failed to get value
										checkerr_flg=true
										break 3		# -> for : 1
									fi
								done
							else
								# failed to enum pingnp/grp/list
								checkerr_flg=true
								break 2		# -> for : 1
							fi
						done
					else
						# failed to enum pingnp/grp
						checkerr_flg=true
						break 1		# -> for : 1
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
				break 1		# -> for : 1
			fi
		else
			# failed to get value
			checkerr_flg=true
			break 1		# -> for : 1
		fi
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum pingnp
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1010
#  check port number
#--------------------
id=1010
xml_path_list=(
	"/root/cluster/api/port"		# 29001 (TCP)
	"/root/cluster/api/ibport"		# 29008 (TCP)
	"/root/cluster/trns/port"		# 29002 (TCP)
	"/root/webmgr/http/port"		# 29003 (TCP)
	"/root/cluster/rstd/http/port"		# 29009 (TCP)
	"/root/cluster/rstd/service/port"	# 29010 (TCP)
	"/root/cluster/heartbeat/port/recv"	# 29002 (UDP)
	"/root/cluster/heartbeat/khbport/recv"	# 29006 (UDP)
	"/root/webalert/daemon/udpport"		# 29003 (UDP)
)

## get message
read_msg_file

## check
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""

for xml_path in ${xml_path_list[@]};
do
	# get value (port)
	clpcfget_exe "g"
	if [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない場合はポリシータイプを指定して再取得
	then
		# policy type
		pol=`echo ${xml_path#/root/}`
		pol=`echo ${pol%%/*}`
		# get value (port)
		clpcfget_exe "g" "${pol}"
	fi

	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		port_check "${cfget}"
		exitcode=`echo $?`
		if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			# failed command execute
			checkerr_flg=true
			break
		elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
		fi
	else
		# failed to get value
		checkerr_flg=true
		break
	fi
done

if "${checkerr_flg}";
then
	# could not check
	set_msg_couldnotchk
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 1011
#  check port number(mirror)
#--------------------
id=1011
## get message
read_msg_file

## check
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""

# get value (port)
xml_path="/root/mdagent/port"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない場合はポリシータイプを指定して再取得
then
	# policy type
	pol=`echo ${xml_path#/root/}`
	pol=`echo ${pol%%/*}`
	# get value (port)
	clpcfget_exe "g" "${pol}"
fi

if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	port_check "${cfget}"
	exitcode=`echo $?`
	if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		# failed command execute
		checkerr_flg=true
	elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
	then
		result_summary="FALSE"
		result="FALSE"
		result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
	fi
else
	# failed to get value
	checkerr_flg=true
fi

if "${checkerr_flg}";
then
	# could not check
	set_msg_couldnotchk
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 1012
#  check port number(log)
#--------------------
id=1012
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""

# get value (method)
xml_path="/root/cluster/event/method"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ "${cfget}" -eq 1 ];	# 0:UNIX DOMAIN, 1:UDP, 2:MSGQ
	then
		# method : UDP
		output_flg=true
		# get value (port)
		xml_path="/root/cluster/event/port"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			port_check "${cfget}"
			exitcode=`echo $?`
			if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
			then
				# failed command execute
				checkerr_flg=true
			elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi
	fi
else
	# failed to get value
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2000
#  check fip ip address(ping)
#--------------------
id=2000
## get message
read_msg_file

## check
# enum fip
xml_path="/root/resource/fip"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for fip_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${fip_name}|"`
		result_msg2=""

		# get value (IP address)
		xml_path="/root/resource/fip@${fip_name}/parameters/ip"
		clpcfget_exe "g" "" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			cfget=`echo ${cfget%%/*}`
			cfget=`echo ${cfget%%%*}`
			ping_check "${cfget}"
			exitcode=`echo $?`
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
			then
				# failed command execute
				checkerr_flg=true
			fi
		elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
		then
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum fip
	set_msg_couldnotchk "Floating IP resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2001
#  check vip ip address(ping)
#--------------------
id=2001
## get message
read_msg_file

## check
# enum vip
xml_path="/root/resource/vip"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for vip_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${vip_name}|"`
		result_msg2=""

		# get value (IP address)
		xml_path="/root/resource/vip@${vip_name}/parameters/ip"
		clpcfget_exe "g" "" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			cfget=`echo ${cfget%%/*}`
			ping_check "${cfget}"
			exitcode=`echo $?`
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
			then
				# failed command execute
				checkerr_flg=true
			fi
		elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
		then
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum vip
	set_msg_couldnotchk "Virtual IP resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2010
#  check disk drive letter
#--------------------
id=2010
## get message
read_msg_file

## check
# enum disk
xml_path="/root/resource/disk"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for disk_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${disk_name}|"`
		result_msg2=""

		# get value (device)
		xml_path="/root/resource/disk@${disk_name}/parameters/device"
		clpcfget_exe "g" "" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			block_device_check "${cfget}"
			if [ $? -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			fi

			# get value (disktype)
			xml_path="/root/resource/disk@${disk_name}/parameters/disktype"
			clpcfget_exe "g" "disk" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" == "vxvm" ];
				then
					# get value (rawdevice)
					xml_path="/root/resource/disk@${disk_name}/parameters/rawdevice"
					clpcfget_exe "g" "" "${server}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						character_device_check "${cfget}"
						if [ $? -ne "${FUNC_RET_SUCCESS}" ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2011
#  check md drive letter
#--------------------
id=2011
## get message
read_msg_file

## check
# enum md
xml_path="/root/resource/md"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for md_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${md_name}|"`
		result_msg2=""

		# get value (dppath)
		xml_path="/root/resource/md@${md_name}/parameters/diskdev/dppath"
		clpcfget_exe "g" "" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			block_device_check "${cfget}"
			if [ $? -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${msg_dp} ${cfget}|"`
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if ! "${checkerr_flg}";
		then
			# get value (cppath)
			xml_path="/root/resource/md@${md_name}/parameters/diskdev/cppath"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				block_device_check "${cfget}"
				if [ $? -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${msg_cp} ${cfget}|"`
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Mirror disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2012
#  check hd drive letter
#--------------------
id=2012
## get message
read_msg_file

## check
# enum hd
xml_path="/root/resource/hd"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for hd_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${hd_name}|"`
		result_msg2=""

		# get value (dppath)
		xml_path="/root/resource/hd@${hd_name}/parameters/diskdev/dppath"
		clpcfget_exe "g" "" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			block_device_check "${cfget}"
			if [ $? -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${msg_dp} ${cfget}|"`
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if ! "${checkerr_flg}";
		then
			# get value (cppath)
			xml_path="/root/resource/hd@${hd_name}/parameters/diskdev/cppath"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				block_device_check "${cfget}"
				if [ $? -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${msg_cp} ${cfget}|"`
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Hybrid disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2020
#  check Azure probe port
#--------------------
id=2020
## get message
read_msg_file

## check
# enum azurepp
xml_path="/root/resource/azurepp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for azurepp_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${azurepp_name}|"`
		result_msg2=""

		# get value (port)
		xml_path="/root/resource/azurepp@${azurepp_name}/parameters/probeport"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			port_check "${cfget}"
			exitcode=`echo $?`
			if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
			then
				# failed command execute
				checkerr_flg=true
			elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Azure probe port resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2030
#  file system check
#--------------------
id=2030
## get message
read_msg_file

## check
# enum disk
xml_path="/root/resource/disk"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for disk_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${disk_name}|"`
		result_msg2=""

		# get value (disktype)
		xml_path="/root/resource/disk@${disk_name}/parameters/disktype"
		clpcfget_exe "g" "disk" "${server}"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			disk_type=${cfget}
			if [ "${disk_type}" == "disk" ];
			then
				notchktarget_flg=false
				# get value (file system)
				xml_path="/root/resource/disk@${disk_name}/parameters/fs"
				clpcfget_exe "g" "disk" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					file_system=${cfget}
					# get value (device)
					xml_path="/root/resource/disk@${disk_name}/parameters/device"
					clpcfget_exe "g" "" "${server}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						disk_file_system_check "${cfget}" "${file_system}"
						exitcode=`echo $?`
						if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
						then
							# failed command execute
							checkerr_flg=true
						elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=${msg2}
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			elif [ "${disk_type}" == "raw" ];
			then
				# OK(PASS)
				notchktarget_flg=false
			elif [ "${disk_type}" == "lvm" ];
			then
				# not check
				notchktarget_flg=true
			elif [ "${disk_type}" == "vxvm" ];
			then
				# not check
				notchktarget_flg=true
			else
				# could not check
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2040
#  VG presence check
#--------------------
id=2040
## get message
read_msg_file

## check
# enum volmgr
xml_path="/root/resource/volmgr"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for volmgr_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${volmgr_name}|"`
		result_msg2=""

		# get value (volume manager)
		xml_path="/root/resource/volmgr@${volmgr_name}/parameters/type"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			volmgr_type=${cfget}
			if [ "${volmgr_type}" == "lvm" ];
			then
				notchktarget_flg=false
				# get value (target name)
				xml_path="/root/resource/volmgr@${volmgr_name}/parameters/devname"
				clpcfget_exe "g"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					vgs ${cfget} > /dev/null 2>&1
					exitcode=`echo $?`
					if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
					then
						# failed command execute
						checkerr_flg=true
					elif [ "${exitcode}" -ne 0 ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			elif [ "${volmgr_type}" == "vxvm" ];
			then
				# not check
				notchktarget_flg=true
			elif [ "${volmgr_type}" == "zfspool" ];
			then
				# not check
				notchktarget_flg=true
			else
				# could not check
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Volume manager resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 3000
#  /dev/watchdog usage check
#--------------------
id=3000
## get message
read_msg_file

## check
# enum userw
xml_path="/root/monitor/userw"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for userw_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${userw_name}|"`
		result_msg2=""

		# get value (method)
		xml_path="/root/monitor/userw@${userw_name}/parameters/method"
		clpcfget_exe "g" "userw"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			if [ "${cfget}" == "softdog" ];
			then
				if [ -e /dev/watchdog ];
				then
					lsof /dev/watchdog > /dev/null 2>&1
					exitcode=`echo $?`
					if [ "${exitcode}" -eq 0 ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=${msg2}
					elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
					then
						# failed command execute
						checkerr_flg=true
					fi
				fi
			else
				# not check
				set_msg_notchktarget
			fi
		elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
		then
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum userw
	set_msg_couldnotchk "User mode monitor"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4000
#  check kernel mode LAN heartbeat resource ip address(ping)
#--------------------
id=4000
## get message
read_msg_file

## check
# enum lankhb
xml_path="/root/heartbeat/lankhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for lankhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${lankhb_name}|"`
		result_msg2=""

		# get value (device)
		xml_path="/root/heartbeat/lankhb@${lankhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					for server_name in ${server_list};
					do
						# get value (info)
						xml_path="/root/server@${server_name}/device@${device}/info"
						clpcfget_exe "g"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if [ "${cfget}" != "" ];
							then
								notchktarget_flg=false
								ping_check "${cfget}"
								exitcode=`echo $?`
								if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
								then
									# failed command execute
									checkerr_flg=true
									break
								elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
								fi
							fi
						else
							# failed to get value
							checkerr_flg=true
							break
						fi
					done
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/lankhb
	set_msg_couldnotchk "Kernel Mode"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4001
#  check LAN heartbeat resource ip address(ping)
#--------------------
id=4001
## get message
read_msg_file

## check
# enum lanhb
xml_path="/root/heartbeat/lanhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for lanhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${lanhb_name}|"`
		result_msg2=""

		# get value (device)
		xml_path="/root/heartbeat/lanhb@${lanhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					for server_name in ${server_list};
					do
						# get value (info)
						xml_path="/root/server@${server_name}/device@${device}/info"
						clpcfget_exe "g"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if [ "${cfget}" != "" ];
							then
								notchktarget_flg=false
								ping_check "${cfget}"
								exitcode=`echo $?`
								if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
								then
									# failed command execute
									checkerr_flg=true
									break
								elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
								fi
							fi
						else
							# failed to get value
							checkerr_flg=true
							break
						fi
					done
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/lanhb
	set_msg_couldnotchk "User Mode"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4010
#  check DISK drive letter
#--------------------
id=4010
## get message
read_msg_file

## check
# enum diskhb
xml_path="/root/heartbeat/diskhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for diskhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${diskhb_name}|"`
		result_msg2=""

		# get value (device)
		xml_path="/root/heartbeat/diskhb@${diskhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					notchktarget_flg=false
					block_device_check "${cfget}"
					if [ $? -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/diskhb
	set_msg_couldnotchk "DISK"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 5000
#  check AWSCLI
#--------------------
id=5000
xml_path_list=(
	"/root/resource/awsdns"
	"/root/resource/awseip"
	"/root/resource/awsvip"
)

## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""

for xml_path in ${xml_path_list[@]};
do
	# get value (port)
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		output_flg=true
		aws ec2 describe-availability-zones > /dev/null 2>&1
		exitcode=`echo $?`
		if [ "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			# command not found
			checkerr_flg=true
		elif [ "${exitcode}" -ne 0 ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=${msg2}
		fi
		break
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to get value
		output_flg=true
		checkerr_flg=true
		break
	fi
done

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 5001
#  check OS Boot timeout
#--------------------
id=5001
## get message
read_msg_file

## check
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
timeout_val=""
multi_val=1

# get value (heartbeat timeout)
xml_path="/root/cluster/heartbeat/timeout"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ -f /etc/default/grub ];
	then
		# grub2 (in seconds)
		multi_val=1000
		#  pattern : GRUB_TIMEOUT=X
		timeout_val=`grep GRUB_TIMEOUT /etc/default/grub | grep -v "#" | awk -F= '{print $2}'`
		if [ "${timeout_val}" == "" ];
		then 
			#  pattern : GRUB_TIMEOUT X
			timeout_val=`grep GRUB_TIMEOUT /etc/default/grub | grep -v "#" | awk '{print $2}'`
		fi
	elif [ -f /boot/grub/menu.lst ];
	then
		# grub (in seconds)
		multi_val=1000
		#  pattern : timeout=X
		timeout_val=`grep timeout /boot/grub/menu.lst | grep -v "#" | awk -F= '{print $2}'`
		if [ "${timeout_val}" == "" ];
		then 
			#  pattern : timeout X
			timeout_val=`grep timeout /boot/grub/menu.lst | grep -v "#" | awk '{print $2}'`
		fi
	elif [ -f /etc/lilo.conf ];
	then
		# lilo (in 1/10 seconds)
		multi_val=100
		#  pattern : timeout=X
		timeout_val=`grep timeout /etc/lilo.conf | grep -v "#" | awk -F= '{print $2}'`
		if [ "${timeout_val}" == "" ];
		then 
			#  pattern : timeout X
			timeout_val=`grep timeout /etc/lilo.conf | grep -v "#" | awk '{print $2}'`
			if [ "${timeout_val}" == "" ];
			then 
				#  pattern : delay=X
				timeout_val=`grep delay /etc/lilo.conf | grep -v "#" | awk -F= '{print $2}'`
				if [ "${timeout_val}" == "" ];
				then 
					#  pattern : delay X
					timeout_val=`grep delay /etc/lilo.conf | grep -v "#" | awk '{print $2}'`
				fi
			fi
		fi
	fi

	if [ "${timeout_val}" != "" ];
	then
		timeout_val=$(( timeout_val * multi_val ))
		if [ "${cfget}" -gt "${timeout_val}" ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=${msg2}
		fi
	else
		checkerr_flg=true
	fi
else
	# failed to get value
	checkerr_flg=true
fi

if "${checkerr_flg}";
then
	# could not check
	set_msg_couldnotchk
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 5002
#  SELinux disable
#--------------------
id=5002
## get message
read_msg_file

## check
result="TRUE"
result_msg1=${msg1}
result_msg2=""

selinux_mode=`getenforce`
exitcode=`echo $?`
if [ "${exitcode}" -ne 0 ];
then
	# could not check
	set_msg_couldnotchk
elif [ "${selinux_mode}" == "Enforcing" ];
then
	result_summary="FALSE"
	result="FALSE"
	result_msg2=${msg2}
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 5003
#  Kernel
#--------------------
id=5003
## get message
read_msg_file

## check
result="TRUE"
result_msg1=${msg1}
result_msg2=""

kernel_check
if [ $? -ne "${FUNC_RET_SUCCESS}" ];
then
	result_summary="FALSE"
	result="FALSE"
	result_msg2=${msg2}
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 6000
#  Recovery action check for deactivation failure
#--------------------
id=6000
## get message
read_msg_file

## check
# enum types
xml_path="/root/resource/types"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for types_name in ${cfget};
	do
		# enum resource
		xml_path="/root/resource/${types_name}"
		clpcfget_exe "e"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			for resource_name in ${cfget};
			do
				result="TRUE"
				result_msg1=`echo ${msg1} | sed -e "s|${rep}|(${resource_name})|"`
				result_msg2=""

				# get value (deact action)
				xml_path="/root/resource/${types_name}@${resource_name}/deact/action"
				clpcfget_exe "g" "${types_name}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					if [ "${cfget}" -eq 0 -o "${cfget}" -eq 1 ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=${msg2}
					fi
				else
					# failed to enum resource
					set_msg_couldnotchk
				fi

				# output result
				output_msg "${result_msg1}" "${result_msg2}"
			done
		else
			# failed to enum resource
			set_msg_couldnotchk "${types_name}"

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		fi
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum types
	set_msg_couldnotchk
	result_msg1=`echo ${msg1} | sed -e "s|${rep}||"`

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2
#  end time
#--------------------
id=2
read_msg_file
output_msg_datetime

#--------------------
# ID : 0
#  result summary
#--------------------
id=0
read_msg_file
output_msg_result_summary


# end
