#!/bin/bash
# ==========================================================
# [OCI 強制停止スクリプト]
#
#
# ==========================================================

export OCI_CLI_SUPPRESS_FILE_PERMISSIONS_WARNING="True"

SubCommand=$1
InstanceId=$2
FstMode=$3

ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"

CommandStop="stop"
CommandCheckStop="check-stop"
CommandMonitor="monitor"

ModeStop="stop"
ModeReboot="reboot"

ExpectStatus="STOPPED"

ShmMaxSize=$((64 - 1))

# AlertSuccess="Command succeeded."
AlertOCINotFound="The OCI CLI command is not found."
AlertOCIFailed="The OCI CLI command failed. (%1)"
AlertUnExpectVMStatus="Stopping instances failed to be completed."
AlertInaternalError="Internal error occurred."

OCIUnknownError="Internal error"


#
# OCIコマンド実行
#
function InvokeOCICommand {
    ociCmdLine=$1
    if ! ociResult=$(eval "$ociCmdLine 2>&1"); then
        echo "$ociResult" >&2
        if ! cause=$(ExtractOCIErrorCause "$ociResult" $ShmMaxSize); then
            cause="$OCIUnknownError"
        fi
        SetAlertLog "${AlertOCIFailed/\%1/$cause}"
        echo "The OCI CLI command failed." >&2
        return 1
    fi
    echo "$ociResult"
}


#
# アラート出力文字列の設定
#
function SetAlertLog {
    message=$1
    result=$(clpshmnmset --descript --fst -m "$message" 2>&1)
    shmNmExit=$?
    if [[ $shmNmExit -ne 0 ]]; then
        echo "The 'clpshmnmset' command failed. ($shmNmExit)" >&2
        echo "================= [ ClpShmNmSet raw message ] =================" >&2
        echo "$result" >&2
        echo "===============================================================" >&2
        # 処理継続
    fi
}


#
# EXITのトラップ
#
function ExitTrap {
    exitCode=$?
    echo "OCI forced-stop script exit: $exitCode"
    exit $exitCode
}


############################################################
#
# 処理開始
#
############################################################
trap ExitTrap EXIT

#-------------------------------------------------------
# OCIコマンドの存在確認
#-------------------------------------------------------
if ! command -v "oci" >/dev/null; then
    SetAlertLog "$AlertOCINotFound"
    echo "The OCI CLI command is not found." >&2
    exit 1
fi

#-------------------------------------------------------
# 強制停止の実行
#-------------------------------------------------------
if [[ "$SubCommand" == "$CommandStop" ]]; then
    if [[ "$FstMode" == "$ModeStop" ]]; then
        # Stop: 停止
        ociCmdLine="oci compute instance action --action STOP --instance-id $InstanceId"

    elif [[ "$FstMode" == "$ModeReboot" ]]; then
        # Reboot: 再起動
        ociCmdLine="oci compute instance action --action RESET --instance-id $InstanceId"
    else
        SetAlertLog "$AlertInaternalError"
        echo "Unknown FstMode: $FstMode" >&2
        exit 1
    fi
    # **** OCIコマンド実行 ****
    if ! InvokeOCICommand "$ociCmdLine" >/dev/null; then
        exit 1
    fi
    echo "Performed the instance stop operation. ($FstMode)"

#-------------------------------------------------------
# 強制停止実行後のスタータスをチェック
#-------------------------------------------------------
elif [[ "$SubCommand" == "$CommandCheckStop" ]]; then
    query='data.\"lifecycle-state\"'
    ociCmdLine="oci compute instance get --instance-id $InstanceId --query $query"
    # **** OCIコマンド実行 ****
    if ! vmStatus=$(InvokeOCICommand "$ociCmdLine" | sed 's/"//g'); then
        exit 1
    fi
    echo "vmStatus: $vmStatus"
    if [[ "$vmStatus" != "$ExpectStatus" ]]; then
        SetAlertLog "$AlertUnExpectVMStatus"
        echo "VM is not in '$ExpectStatus' state. vmStatus: $vmStatus" >&2
        exit 1
    fi
    echo "VM has entered a stopped state. ($FstMode)"

#-------------------------------------------------------
# 強制停止機能を実行可能かチェック
#-------------------------------------------------------
elif [[ "$SubCommand" == "$CommandMonitor" ]]; then
    # NOTE: 同等の実行権限が必要なupdateコマンドを実行して確認
    for ociCmdLine in \
            "oci compute instance update --instance-id $InstanceId" \
            "oci compute instance get --instance-id $InstanceId"; do
        # **** OCIコマンド実行 ****
        if ! InvokeOCICommand "$ociCmdLine" >/dev/null; then
            exit 1
        fi
    done
    echo "There were no issues during the periodic instance stop confirmation check."

else
    SetAlertLog "$AlertInaternalError"
    echo "Unknown SubCommand: $SubCommand" >&2
    exit 1
fi

exit 0
