#!/bin/bash
#
##
# clpgcdns-watch.sh
##
# monitoring the clpgcdns
##


##
# function
##


# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --mon -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# gcloud cli error handling
function gcloudcli_err_handle {
	ERR_DETAIL=$1
	GCLOUDCLI_DETAIL=$2
	echo "${GCLOUDCLI_DETAIL}"
	GCLOUDCLI_DETAIL=$(ExtractGCPErrorCause "${GCLOUDCLI_DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		GCLOUDCLI_DETAIL="${GCDNSW_ERR_GCLODCLI_MSG}"
	fi
	ALTMSG="${ERR_DETAIL}(${GCLOUDCLI_DETAIL})"
	ALTMSG=$(Trancate "${ALTMSG}" ${MAX_SHMRM_SIZE})
	set_alert_message "${ALTMSG}"
}


# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# monitoring the clpgcdns
##


echo "clpgcdns watch script start."


##
# ret code
##
# success
GCDNSW_SUCCESS_CODE=0					# succeed
# error
GCDNSW_ERR_GET_RECORD_CODE=56			# get dns record has failed
GCDNSW_ERR_NO_RECORD_CODE=57			# there is no designated dns record
# warning
GCDNSW_WARN_GET_VALUE_CODE=150			# failed to get the setting value
GCDNSW_WARN_GET_ZONENAME_CODE=151			# get zone name has failed
GCDNSW_WARN_GET_RECORDNAME_CODE=152		# get record name has failed
GCDNSW_WARN_GET_RECORDTYPE_CODE=153		# get record type has failed
GCDNSW_WARN_GET_RECORDTTL_CODE=154		# get ttl has failed
GCDNSW_WARN_GET_RECORDIP_CODE=155			# get ip address has failed
GCDNSW_WARN_SCH_GCLOUDCLI_CODE=158		# failed to search gcloud cli path


##
# resultmsg
##
# error
GCDNSW_ERR_GCLODCLI_MSG="The gcloud CLI command failed."
GCDNSW_ERR_GET_RECORD_MSG="Failed to obtain the record set."
GCDNSW_ERR_NO_RECORD_MSG="No record set to be monitored."
# warning
GCDNSW_WARN_GET_VALUE_MSG="Failed to obtain the setting value."
GCDNSW_WARN_GET_ZONENAME_MSG="Failed to obtain the zone name."
GCDNSW_WARN_GET_RECORDNAME_MSG="Failed to obtain the DNS name."
GCDNSW_WARN_GET_RECORDTYPE_MSG="Failed to obtain the record type."
GCDNSW_WARN_GET_RECORDTTL_MSG="Failed to obtain the TTL."
GCDNSW_WARN_GET_RECORDIP_MSG="Failed to obtain the IP address."
GCDNSW_WARN_SCH_GCLOUDCLI_MSG="The gcloud CLI command is not found."


##
# set the value to use for the shmrmset command
##
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="gcdnsw"						# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# gcloud cli path search
##
which gcloud > /dev/null 2>&1
SCH_GCLOUDCLI_EXIT_CODE=$?
if [[ ${SCH_GCLOUDCLI_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${GCDNSW_WARN_SCH_GCLOUDCLI_MSG}"
	echo "failed to search gcloud cli path." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_SCH_GCLOUDCLI_CODE}
fi


##
# set value
##
TARGET_RSC="${CLP_OCF_PARAM1}"			# resource name
SRV_FULLNAME="`uname -n`"				# get hostname (use output file path)
SRV_NAME="`echo ${SRV_FULLNAME%%.*}`"	# delete domain name

if [ -z "$TARGET_RSC" ]
then
	set_alert_message "${GCDNSW_WARN_GET_VALUE_MSG}"
	echo "failed to get the setting value." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_VALUE_CODE}
fi

##
# get target resource config
##
ZONE_NAME="`clpcfget -g /root/resource/gcdns@\"${TARGET_RSC}\"/parameters/zone_name -p gcdns`"
if [ $? -ne 0 ]
then
	set_alert_message "${GCDNSW_WARN_GET_ZONENAME_MSG}"
	echo "$ZONE_NAME" | head -n 1 1>&2
	echo "get zone name has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_ZONENAME_CODE}
fi

RECORD_NAME="`clpcfget -g /root/resource/gcdns@\"${TARGET_RSC}\"/parameters/dns_name -p gcdns`"
if [ $? -ne 0 ]
then
	set_alert_message "${GCDNSW_WARN_GET_RECORDNAME_MSG}"
	echo "$RECORD_NAME" | head -n 1 1>&2
	echo "get record name has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_RECORDNAME_CODE}
fi

RECORD_TYPE="`clpcfget -g /root/resource/gcdns@\"${TARGET_RSC}\"/parameters/record_type -p gcdns`"
if [ $? -ne 0 ]
then
	set_alert_message "${GCDNSW_WARN_GET_RECORDTYPE_MSG}"
	echo "$RECORD_TYPE" | head -n 1 1>&2
	echo "get record type has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_RECORDTYPE_CODE}
fi

RECORD_TTL="`clpcfget -g /root/resource/gcdns@\"${TARGET_RSC}\"/parameters/record_ttl -p gcdns`"
if [ $? -ne 0 ]
then
	set_alert_message "${GCDNSW_WARN_GET_RECORDTTL_MSG}"
	echo "$RECORD_TTL" | head -n 1 1>&2
	echo "get ttl has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_RECORDTTL_CODE}
fi

RECORD_IP="`clpcfget -g /root/resource/gcdns@\"${TARGET_RSC}\"/parameters/record_ip -p gcdns -s \"${SRV_NAME}\"`"
if [ $? -ne 0 ]
then
	set_alert_message "${GCDNSW_WARN_GET_RECORDIP_MSG}"
	echo "$RECORD_IP" | head -n 1 1>&2
	echo "get ip address has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_WARN_GET_RECORDIP_CODE}
fi

echo "get config has succeeded (zone=${ZONE_NAME} name=${RECORD_NAME} type=${RECORD_TYPE} ttl=${RECORD_TTL} ip=${RECORD_IP})."


##
# get dns record
##
CURRENT="`gcloud dns record-sets list -z=\"${ZONE_NAME}\" --name=\"${RECORD_NAME}\" --type=\"${RECORD_TYPE}\" \
	--format=\"table[no-heading](name,type,ttl, rrdatas.list():label=DATA)\" 2>&1`"
if [ $? -ne 0 ]
then
	gcloudcli_err_handle "${GCDNSW_ERR_GET_RECORD_MSG}" "${CURRENT}"
	echo "get dns record has failed." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_ERR_GET_RECORD_CODE}
elif [ -z "$CURRENT" ]
then
	set_alert_message "${GCDNSW_ERR_NO_RECORD_MSG} (get dns record)"
	echo "there is no designated dns record." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_ERR_NO_RECORD_CODE}
fi


##
# ttl ip check
##
CURRENT_TTL="`echo ${CURRENT} | awk '{print $3}'`"
CURRENT_IP="`echo ${CURRENT} | awk '{print $4}'`"

if [ "$CURRENT_TTL" -ne "$RECORD_TTL" ]
then
	set_alert_message "${GCDNSW_ERR_NO_RECORD_MSG} (ttl check)"
	echo "CURRENT_TTL = ${CURRENT_TTL} / RECORD_TTL = ${RECORD_TTL}" 1>&2
	echo "ttl is not suitable." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_ERR_NO_RECORD_CODE}
fi

if [ "$CURRENT_IP" != "$RECORD_IP" ]
then
	set_alert_message "${GCDNSW_ERR_NO_RECORD_MSG} (ip check)"
	echo "CURRENT_IP = ${CURRENT_IP} / RECORD_IP = ${RECORD_IP}" 1>&2
	echo "ip address is not suitable." 1>&2
	echo "clpgcdns watch script has failed." 1>&2
	exit ${GCDNSW_ERR_NO_RECORD_CODE}
fi

##
# end
##
echo "clpgcdns watch script has succeeded."

exit ${GCDNSW_SUCCESS_CODE}

