#!/bin/bash

# clpcloudutilの呼び出し
scriptdir="$(cd $(dirname $0); pwd)"
. "${scriptdir}/clpcloudutil.sh"

export PATH=$PATH:/usr/local/bin

CURL_TIMEOUT=5
AWS_API_VERSION=2019-10-01
AWS_TOKEN_TTL=60
AWS_CLI_TIMEOUT=10
AZURE_API_VERSION=2019-11-01

PFTYPE_AWS=1
PFTYPE_AZURE=2
PFTYPE_GCP=3
PFTYPE_OCI=4

#==================================
# AWS環境情報取得
#==================================
get_aws_info()
{
    token=$(curl -is --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}" -X PUT "http://169.254.169.254/latest/api/token" \
            -H "X-aws-ec2-metadata-token-ttl-seconds: ${AWS_TOKEN_TTL}" \
            | tr -d "\r" | (
                read -r _ statcode _
                test "${statcode:=0}" -eq 200 || exit 1
                while read -r line
                do
                    test "${line}" = "Server: EC2ws" && ok=1
                done
                echo "${line}"
                test "${ok}"
            ))

    if [ "$?" -ne 0 ]; then
        return
    fi

    echo "PFTYPE=AWS"

    # ec2_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" ec2`"
    # rt53_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" route53`"

    # ----INSTANCE-ID----
    instance_id=$(curl -sS "http://169.254.169.254/${AWS_API_VERSION}/meta-data/instance-id" \
        -H "X-aws-ec2-metadata-token: ${token}" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}") \
        || echo "Could not get [instance-id]" >&2
    [ -n "${instance_id}" ] && echo "INSTANCE_ID=${instance_id}"

    interfaces="http://169.254.169.254/${AWS_API_VERSION}/meta-data/network/interfaces"
    macs=$(curl -sS "${interfaces}/macs" -H "X-aws-ec2-metadata-token: ${token}" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}")

    if [ "$?" -ne 0 ]; then
        echo "Could not get [macs]" >&2
        exit 1
    fi

    # ----VPC-ID----
    for mac in ${macs}
    do
        vpc_id=$(curl -sS "${interfaces}/macs/${mac}vpc-id" \
            -H "X-aws-ec2-metadata-token: ${token}" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}") \
            || echo "Could not get [vpc-id]" >&2
        [ -n "${vpc_id}" ] && echo "VPC_ID=${vpc_id}"
    done

    # ----ENI-ID----
    for mac in ${macs}
    do
        eni_id=$(curl -sS "${interfaces}/macs/${mac}interface-id" \
            -H "X-aws-ec2-metadata-token: ${token}" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}") \
            || echo "Could not get [eni-id]" >&2
        [ -n "${eni_id}" ] && echo "ENI_ID=${eni_id}"
    done

#    # ----EIP-ALLOCATION-ID----
#    eip_allocation_ids=$(aws ec2 describe-addresses ${ec2_cmdopt} --output text \
#        --cli-read-timeout "${AWS_CLI_TIMEOUT}" \
#        --cli-connect-timeout "${AWS_CLI_TIMEOUT}" \
#        --query 'Addresses[].AllocationId') \
#        || echo "Could not get [eip-allocation-id]" >&2
#
#    for key in ${eip_allocation_ids}
#    do
#        echo "EIP_ALLOCATION_ID=${key}"
#    done
#
#    # ----HOSTED-ZONE-ID----
#    hosted_zone_ids=$(aws route53 list-hosted-zones ${rt53_cmdopt} --output text \
#        --cli-read-timeout "${AWS_CLI_TIMEOUT}" \
#        --cli-connect-timeout "${AWS_CLI_TIMEOUT}" \
#        --query 'HostedZones[].Id') \
#        || echo "Could not get [hosted-zone-id]" >&2
#
#    for key in ${hosted_zone_ids}
#    do
#        echo "HOSTED_ZONE_ID=${key##*/}"
#    done

    # ----AZ-NAME----
    az_name=$(curl -sS "http://169.254.169.254/${AWS_API_VERSION}/meta-data/placement/availability-zone" \
        -H "X-aws-ec2-metadata-token: ${token}" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}") \
        || echo "Could not get [availability-zone]" >&2
    [ -n "${az_name}" ] && echo "AZ_NAME=${az_name}"

    exit 0
}


#==================================
# Azure環境情報取得
#==================================
get_azure_info()
{
    curl -is "http://169.254.169.254/metadata/instance?api-version=${AZURE_API_VERSION}&format=text" \
        -H "Metadata:true" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}" \
        | tr -d "\r" | (
            read -r _ statcode _
            [ "${statcode:=0}" -eq 200 ] && grep "^Server: IMDS" >/dev/null
        )

    if [ "$?" -ne 0 ]; then
        return
    fi

    echo "PFTYPE=Azure"
    exit 0
}


#==================================
# GCP環境情報取得
#==================================
get_gcp_info()
{
    curl -is "http://169.254.169.254/computeMetadata/v1/" \
        -H "Metadata-Flavor: Google" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}" \
        | tr -d "\r" | (
            read -r _ statcode _
            [ "${statcode:=0}" -eq 200 ] && grep "^Metadata-Flavor: Google" >/dev/null
        )

    if [ "$?" -ne 0 ]; then
        return
    fi

    echo "PFTYPE=GCP"
    exit 0
}


#==================================
# OCI環境情報取得
#==================================
get_oci_info()
{
    curl -Is "http://169.254.169.254/opc/v1/instance/" --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}" \
        | tr -d "\r" | (
            read -r _ statcode _
            [ "${statcode:=0}" -eq 200 ]
        )

    if [ "$?" -ne 0 ]; then
        return
    fi

    echo "PFTYPE=OCI"
    exit 0
}


#==================================
# ログ採取用インスタンスメタデータ取得 (AWS)
#==================================
function get_instance_meta_data_aws {
    # セッショントークンを取得 (IMDS v2)
    token_header="X-aws-ec2-metadata-token-ttl-seconds: $AWS_TOKEN_TTL"
    token_uri="http://169.254.169.254/latest/api/token"
    token=$(curl -X PUT -s -w "\n" -H "$token_header" "$token_uri" --noproxy "*" -m $CURL_TIMEOUT)
    if [ $? -ne 0 ]; then
        echo "Could not retrieve the 'token'." >&2
        exit 1
    fi

    header="X-aws-ec2-metadata-token: $token"
    # ami-id
    uri="http://169.254.169.254/$AWS_API_VERSION/meta-data/ami-id"
    if ! ami_id=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'ami-id'." >&2
    fi

    # instance-type
    uri="http://169.254.169.254/$AWS_API_VERSION/meta-data/instance-type"
    if ! instance_type=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'instance-type'." >&2
    fi

    # availability-zone
    uri="http://169.254.169.254/$AWS_API_VERSION/meta-data/placement/availability-zone"
    if ! availability_zone=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'availability-zone'." >&2
    fi

    # region
    uri="http://169.254.169.254/$AWS_API_VERSION/meta-data/placement/region"
    if ! region=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'region'." >&2
    fi

    # ファイル出力
    file_name="aws_metadata"
    {
        echo "[meta-data]"
        echo "    ami-id            : $ami_id"
        echo "    instance-type     : $instance_type"
        echo ""
        echo "[meta-data/placement]"
        echo "    availability-zone : $availability_zone"
        echo "    region            : $region"
    } > "$file_name"
}


#==================================
# ログ採取用インスタンスメタデータ取得 (Azure)
#==================================
function get_instance_meta_data_azure {
    base_uri="http://169.254.169.254/metadata"
    header="Metadata: true"

    # compute
    uri="$base_uri/instance/compute?api-version=$AZURE_API_VERSION"
    if ! compute=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'compute'." >&2
    fi

    # imageReference
    uri="$base_uri/instance/compute/storageProfile/imageReference?api-version=$AZURE_API_VERSION"
    if ! imageReference=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'imageReference'." >&2
    fi

    location=$(echo "$compute" | sed -n 's/.*"location"\s*:\s*"\([^"]*\)".*/\1/p')
    vmSize=$(echo "$compute" | sed -n 's/.*"vmSize"\s*:\s*"\([^"]*\)".*/\1/p')
    zone=$(echo "$compute" | sed -n 's/.*"zone"\s*:\s*"\([^"]*\)".*/\1/p')

    id=$(echo "$imageReference" | sed -n 's/.*"id"\s*:\s*"\([^"]*\)".*/\1/p')
    offer=$(echo "$imageReference" | sed -n 's/.*"offer"\s*:\s*"\([^"]*\)".*/\1/p')
    publisher=$(echo "$imageReference" | sed -n 's/.*"publisher"\s*:\s*"\([^"]*\)".*/\1/p')
    sku=$(echo "$imageReference" | sed -n 's/.*"sku"\s*:\s*"\([^"]*\)".*/\1/p')
    version=$(echo "$imageReference" | sed -n 's/.*"version"\s*:\s*"\([^"]*\)".*/\1/p')

    # ファイル出力
    file_name="azure_metadata"
    {
        echo "[metadata/instance/compute]"
        echo "    location  : $location"
        echo "    vmSize    : $vmSize"
        echo "    zone      : $zone"
        echo ""
        echo "[metadata/instance/compute/storageProfile/imageReference]"
        echo "    id        : $id"
        echo "    offer     : $offer"
        echo "    publisher : $publisher"
        echo "    sku       : $sku"
        echo "    version   : $version"
    } > "$file_name"
}


#==================================
# ログ採取用インスタンスメタデータ取得 (GCP)
#==================================
function get_instance_meta_data_gcp {
    header="Metadata-Flavor: Google"

    # image
    uri="http://169.254.169.254/computeMetadata/v1/instance/image"
    if ! image=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'image'."
    fi

    # machine-type
    uri="http://169.254.169.254/computeMetadata/v1/instance/machine-type"
    if ! machine_type=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'machine-type'."
    fi

    # zone
    uri="http://169.254.169.254/computeMetadata/v1/instance/zone"
    if ! zone=$(curl -s -H "$header" "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'zone'."
    fi

    # ファイル出力
    file_name="gcp_metadata"
    {
        echo "[computeMetadata/v1/instance]"
        echo "    image        : $image"
        echo "    machine-type : $machine_type"
        echo "    zone         : $zone"
    } > "$file_name"
}


#==================================
# ログ採取用インスタンスメタデータ取得 (OCI)
#==================================
function get_instance_meta_data_oci {
    # instance
    uri="http://169.254.169.254/opc/v1/instance"
    if ! instance=$(curl -s "$uri" --noproxy "*" -m $CURL_TIMEOUT); then
        echo "Could not retrieve the 'instance'."
    fi

    availabilityDomain=$(echo "$instance" | sed -n 's/.*"availabilityDomain"\s*:\s*"\([^"]*\)".*/\1/p')
    image=$(echo "$instance" | sed -n 's/.*"image"\s*:\s*"\([^"]*\)".*/\1/p')
    region=$(echo "$instance" | sed -n 's/.*"region"\s*:\s*"\([^"]*\)".*/\1/p')
    shape=$(echo "$instance" | sed -n 's/.*"shape"\s*:\s*"\([^"]*\)".*/\1/p')

    # ファイル出力
    file_name="oci_metadata"
    {
        echo "[opc/v1/instance]"
        echo "    availabilityDomain : $availabilityDomain"
        echo "    image              : $image"
        echo "    region             : $region"
        echo "    shape              : $shape"
    } > "$file_name"
}


# ---------------------------- 処理開始 ----------------------------
sub_command="$1"

clpcloudutil_env_init "${clpcloudutil_true}" "${scriptdir}/../aws/clpaws_setting.conf"

#==================================
# クラウド環境情報取得
#==================================
if [ "${sub_command}" = "get-cloud-info" ]; then

    curl -IsS --noproxy "169.254.169.254" -m "${CURL_TIMEOUT}" "http://169.254.169.254/" > /dev/null

    if [ "$?" -ne 0 ]; then
        echo "PFTYPE=UNKNOWN"
        exit 0
    fi

    get_aws_info
    get_azure_info
    get_gcp_info
    get_oci_info

    echo "PFTYPE=UNKNOWN"


#==================================
# Amazon Cloudwatchメトリクス送信
#==================================
elif [ "${sub_command}" = "aws-put-metric-data" ]; then

    if [ "$#" -lt 3 ]; then
        echo "Invalid Option!!" >&2
        exit 1
    fi

    namespace="$2"
    metric_data=("${@:3}")
    cw_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" cloudwatch`"

    # awscliの存在確認結果
    which aws > /dev/null 2>&1
    if [ "$?" -ne 0 ]; then
        echo "The 'aws' command does not exist." >&2
        exit 1
    fi

    message="`aws cloudwatch put-metric-data --namespace "${namespace}" --metric-data "${metric_data[@]}" ${cw_cmdopt} 2>&1`"

    if [ "$?" -ne 0 ]; then
        echo "Command Error. [aws cloudwatch put-metric-data]" >&2
        echo $message >&2
        regexp="A.* error .* \(AccessDenied\) .*"
        if [[ $message =~ $regexp ]]; then
            exit 2
        else
            exit 1
        fi
    fi

#==================================
# Amazon SNSトピック発行
#==================================
elif [ "${sub_command}" = "aws-publish-topic" ]; then

    if [ "$#" -ne 4 ]; then
        echo "Invalid Option!!" >&2
        exit 1
    fi

    topic_arn="$2"
    message="$3"
    region="$4"
    sns_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" sns`"

    # awscliの存在確認結果
    which aws > /dev/null 2>&1
    if [ "$?" -ne 0 ]; then
        echo "The 'aws' command does not exist." >&2
        exit 1
    fi

    cmdline="aws sns publish ${sns_cmdopt} --output text --topic-arn \"${topic_arn}\" --message \"${message}\""
    if [[ -n "$region" ]]; then
        cmdline="$cmdline --region $region"
    fi

    eval "$cmdline"
    if [ "$?" -ne 0 ]; then
        echo "Command Error. [aws sns publish]" >&2
        echo "  - topic_arn: $topic_arn" >&2
        echo "  - message: $message" >&2
        echo "  - region: $region" >&2
        exit 1
    fi

#==================================
# AWS環境設定情報収集
#==================================
elif [ "${sub_command}" = "collect-aws-info" ]; then

    if [ "$#" -ne 4 ]; then
        echo "Invalid Option!!" >&2
        exit 1
    fi

    pftype="$2"
    output_dir="$3"
    instance_id="$4"

    # 出力ディレクトリへ移動
    cd "$output_dir" || { echo "Directory not found. ($output_dir)" >&2; exit 1; }

    # インスタンスメタデータ取得
    if [ "${pftype}" = "${PFTYPE_AWS}" ]; then
        get_instance_meta_data_aws
    fi

    # awscliの存在確認結果
    which aws > "awscli_which" 2>&1
    if [ "$?" -ne 0 ]; then
        echo "The 'aws' command does not exist." >&2
        exit 0
    fi

    # awscliのバージョン情報
    aws --version > "awscli_version" 2>&1
    if [ "$?" -ne 0 ]; then
        echo "aws command failed. [aws --version]" >&2
    fi

    # awscliの設定情報
    aws configure list > "awscli_configure_list" 2>&1
    if [ "$?" -ne 0 ]; then
        echo "aws command failed. [aws configure list]" >&2
    fi

    if [ "${pftype}" = "${PFTYPE_AWS}" ]; then
        ec2_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" ec2`"
        # ENIの送信元/送信先チェック
        aws ec2 describe-network-interfaces \
            --filters Name=attachment.instance-id,Values="${instance_id}" ${ec2_cmdopt} \
            --output json > "awscli_describe_network_interfaces" 2>&1
        if [ "$?" -ne 0 ]; then
            echo "aws command failed. [aws ec2 describe-network-interfaces]" >&2
        fi

        # EC2の停止保護の有効/無効
        aws ec2 describe-instance-attribute \
            --instance-id "${instance_id}" \
            --attribute disableApiStop ${ec2_cmdopt} \
            --output json > "awscli_describe_instance_attribute" 2>&1
        if [ "$?" -ne 0 ]; then
            echo "aws command failed. [aws ec2 describe-instance-attribute]" >&2
        fi
    fi

#==================================
# AZURE環境設定情報収集
#==================================
elif [ "${sub_command}" = "collect-azure-info" ]; then
    pftype="$2"
    output_dir="$3"

    # 出力ディレクトリへ移動
    cd "$output_dir" || { echo "Directory not found. ($output_dir)" >&2; exit 1; }

    # インスタンスメタデータ取得
    if [ "${pftype}" = "${PFTYPE_AZURE}" ]; then
        get_instance_meta_data_azure
    fi

#==================================
# GCP環境設定情報収集
#==================================
elif [ "${sub_command}" = "collect-gcp-info" ]; then
    pftype="$2"
    output_dir="$3"

    # 出力ディレクトリへ移動
    cd "$output_dir" || { echo "Directory not found. ($output_dir)" >&2; exit 1; }

    # インスタンスメタデータ取得
    if [ "${pftype}" = "${PFTYPE_GCP}" ]; then
        get_instance_meta_data_gcp
    fi

#==================================
# OCI環境設定情報収集
#==================================
elif [ "${sub_command}" = "collect-oci-info" ]; then
    pftype="$2"
    output_dir="$3"

    # 出力ディレクトリへ移動
    cd "$output_dir" || { echo "Directory not found. ($output_dir)" >&2; exit 1; }

    # インスタンスメタデータ取得
    if [ "${pftype}" = "${PFTYPE_OCI}" ]; then
        get_instance_meta_data_oci
    fi

else
    echo "Invalid Sub Command!!" >&2
    exit 1

fi



exit 0
