#!/bin/bash
# ==========================================================
# [Azure 強制停止スクリプト]
#
#
# ==========================================================

SubCommand=$1
VmName=$2
RscGrp=$3
UserUri=$4
TenantId=$5
CertFile=$6
FstMode=$7

ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"

CommandStop="stop"
CommandCheckStop="check-stop"
CommandMonitor="monitor"

ModeStop="stop"
ModeStopAndDeallocate="stop and deallocate"
ModeStopOnly="stop only"
ModeReboot="reboot"

VmDeallocated="VM deallocated"
VmStopped="VM stopped"

ShmMaxSize=$((64 - 1))
AZCmdRetryMax=3

# AlertSuccess="Command succeeded."
AlertAZNotFound="The Azure CLI command is not found."
AlertAZLoginFailed="Failed to log in to the Azure."
AlertAZFailed="The Azure CLI command failed. (%1)"
# AlertAZTimeout="Timeout occurred."
AlertUnExpectVMStatus="Stopping instances failed to be completed."
AlertInaternalError="Internal error occurred."

AZUnknownError="Internal error"

AZNotLoginError1="Please run 'az login' to setup account."
AZNotLoginError2="Could not retrieve credential from local cache for service principal"
AZNotLoginErrorRegExp="${AZNotLoginError1}|${AZNotLoginError2}"


#
# AZログイン
#
function AZLogin {
    # 都度.azureディレクトリを削除する
    rm -rf "${HOME}/.azure"

    # サービスプリンシパルのファイルパスを指定するaz loginコマンドのオプション
    #  - 2.67.0以降: --certificate
    #  - 2.66.0以前: -p (--password)
    if ! az login --service-principal -u "$UserUri" --certificate "$CertFile" --tenant "$TenantId" >/dev/null; then
        echo "The az login --certificate command failed. Retrying with the az login -p command." >&2
        if ! az login --service-principal -u "$UserUri" -p "$CertFile" --tenant "$TenantId" >/dev/null; then
            echo "The az login -p command also failed." >&2
            SetAlertLog "$AlertAZLoginFailed"
            echo "AZ login failed." >&2
            return 1
        fi
    fi
}


#
# AZコマンド実行
#
function InvokeAZCommand {
    azCmdLine=$1
    azCmdRetryCnt=1
    while true; do
        if ! azResult=$(eval "$azCmdLine 2>&1"); then
            echo "$azResult" >&2
            # 別プロセスでAZログアウトしたことによりAZコマンドが実行できなかった場合は、
            # 再度AZログインを行った後にAZコマンドを実行する
            if [[ "$azResult" =~ $AZNotLoginErrorRegExp ]]; then
                if [[ $azCmdRetryCnt -ge $AZCmdRetryMax ]]; then
                    # 既定回数を超えた場合は「内部エラー」
                    SetAlertLog "$AlertInaternalError"
                    echo "The number of login attempts for AZ has exceeded the default limit." >&2
                    return 1
                fi
                sleep 0.1
                if ! AZLogin; then
                    return 1
                fi
                ((azCmdRetryCnt++))
                continue
            fi
            # AZログインエラー以外の異常時はエラーコードを抽出してアラートログへ出力
            if ! cause=$(ExtractAzureErrorCause "$azResult" $ShmMaxSize); then
                cause="$AZUnknownError"
            fi
            SetAlertLog "${AlertAZFailed/\%1/$cause}"
            echo "The AZ CLI command failed." >&2
            return 1
        fi
        break
    done
    echo "$azResult"
}


#
# アラート出力文字列の設定
#
function SetAlertLog {
    message=$1
    result=$(clpshmnmset --descript --fst -m "$message" 2>&1)
    shmNmExit=$?
    if [[ $shmNmExit -ne 0 ]]; then
        echo "The 'clpshmnmset' command failed. ($shmNmExit)" >&2
        echo "================= [ ClpShmNmSet raw message ] =================" >&2
        echo "$result" >&2
        echo "===============================================================" >&2
        # 処理継続
    fi
}


#
# EXITのトラップ
#
function ExitTrap {
    exitCode=$?
    echo "Azure forced-stop script exit: $exitCode"
    exit $exitCode
}


############################################################
#
# 処理開始
#
############################################################
trap ExitTrap EXIT

#-------------------------------------------------------
# AZコマンドの存在確認
#-------------------------------------------------------
if ! command -v "az" >/dev/null; then
    SetAlertLog "$AlertAZNotFound"
    echo "The Azure CLI command is not found." >&2
    exit 1
fi

#-------------------------------------------------------
# workディレクトリ作成
#-------------------------------------------------------
umask 033
installDir=$(cd "$(dirname "$0")/../.." && pwd)
workDir="${installDir}/work/forcestop/azure"
mkdir -p "$workDir"

#-------------------------------------------------------
# .azureディレクトリ生成パス設定
#-------------------------------------------------------
export HOME=$workDir

#-------------------------------------------------------
# AZログイン
#-------------------------------------------------------
if ! AZLogin; then
    exit 1
fi

#-------------------------------------------------------
# 強制停止の実行
#-------------------------------------------------------
if [[ "$SubCommand" == "$CommandStop" ]]; then
    if [[ "$FstMode" == "$ModeStop" || "$FstMode" == "$ModeStopAndDeallocate" ]]; then
        # Stop, Stop And Deallocate: リソース解放を伴うインスタンス停止
        azCmdLine="az vm deallocate -g $RscGrp -n $VmName --no-wait"
    elif [[ "$FstMode" == "$ModeStopOnly" ]]; then
        # Stop Only: リソース解放を伴わないインスタンス停止(※停止後も課金継続)
        azCmdLine="az vm stop -g $RscGrp -n $VmName --no-wait --skip-shutdown"
    elif [[ "$FstMode" == "$ModeReboot" ]]; then
        # Reboot: 再起動
        azCmdLine="az vm restart -g $RscGrp -n $VmName --no-wait"
    else
        SetAlertLog "$AlertInaternalError"
        echo "Unknown FstMode: $FstMode" >&2
        exit 1
    fi
    # **** AZコマンド実行 ****
    if ! InvokeAZCommand "$azCmdLine" >/dev/null; then
        exit 1
    fi
    echo "Performed the instance stop operation. ($FstMode)"

#-------------------------------------------------------
# 強制停止実行後のスタータスをチェック
#-------------------------------------------------------
elif [[ "$SubCommand" == "$CommandCheckStop" ]]; then
    if [[ "$FstMode" == "$ModeStop" || "$FstMode" == "$ModeStopAndDeallocate" ]]; then
        expectStatus="$VmDeallocated"
    elif [[ "$FstMode" == "$ModeStopOnly" ]]; then
        expectStatus="$VmStopped"
    else
        SetAlertLog "$AlertInaternalError"
        echo "Unknown FstMode: $FstMode" >&2
        exit 1
    fi
    format="tsv"
    query="instanceView.statuses[1].displayStatus"
    azCmdLine="az vm get-instance-view -g $RscGrp -n $VmName --query $query --output $format"
    # **** AZコマンド実行 ****
    if ! vmStatus=$(InvokeAZCommand "$azCmdLine"); then
        exit 1
    fi
    echo "vmStatus: $vmStatus"
    if [[ "$vmStatus" != "$expectStatus" ]]; then
        SetAlertLog "$AlertUnExpectVMStatus"
        echo "VM($VmName) is not in '$expectStatus' state. vmStatus: $vmStatus" >&2
        exit 1
    fi
    echo "VM($VmName) has entered a stopped state. ($FstMode)"

#-------------------------------------------------------
# 強制停止機能を実行可能かチェック
#-------------------------------------------------------
elif [[ "$SubCommand" == "$CommandMonitor" ]]; then
    # NOTE: 同等の実行権限が必要なupdateコマンドを実行して確認
    for azCmdLine in \
                "az vm update -g $RscGrp -n $VmName --no-wait" \
                "az vm get-instance-view -g $RscGrp -n $VmName"; do
        # **** AZコマンド実行 ****
        if ! InvokeAZCommand "$azCmdLine" >/dev/null; then
            exit 1
        fi
    done
    echo "There were no issues during the periodic instance stop confirmation check."

else
    SetAlertLog "$AlertInaternalError"
    echo "Unknown SubCommand: $SubCommand" >&2
    exit 1
fi

exit 0
