#!/bin/bash
#
##
# clpawsvip-start.sh
##
# [coding：utf-8]
##


##
# function
##


# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --rsc -g "${GROUP_NAME}" -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# aws cli error handling
function awscli_err_handle {
	DETAIL=$1
	echo "${DETAIL}"
	AWSCLI_ALTMSG=$(ExtractAWSErrorCause "${DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		AWSCLI_ALTMSG="${AWSVIP_ERR_AWSCLI_MSG}"
	fi
	set_alert_message "${AWSCLI_ALTMSG}"
}

# convert ip address notation to 32-bit value
function ipconv_32bit(){
	IFS=.
	CON_IP=($1)
	printf "%s\n" $(( (${CON_IP[0]} << 24) | (${CON_IP[1]} << 16) | (${CON_IP[2]} << 8) | ${CON_IP[3]} ))
	unset IFS
}

# converts network address in cidr notation to 32-bit value
function cidrconv_32bit(){
	printf "%s\n" $(( 0xFFFFFFFF ^ ((2 ** (32-$1))-1) ))
}

# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# start the clpawsvip
##

echo "clpawsvip start script start."


##
# ret code
##
AWSVIP_SUCCESS_CODE=0					# succeed
AWSVIP_ERR_AWSCLI_CODE=50				# failed in the awscli command
AWSVIP_ERR_SCHAWSCLI_CODE=51			# failed to search aws cli path
AWSVIP_ERR_GETOCFENV_CODE=52			# failed to get the ocf environment variable
AWSVIP_ERR_CHKSUB_CODE=53				# vip is belong to subnet of vpc
AWSVIP_ERR_ADDVIP_CODE=54				# failed to add virtual ip address
# AWSVIP_ERR_DELVIP_CODE=55				# failed to delete virtual ip address
AWSVIP_ERR_GETNICNAME_CODE=56			# failed to get nic name
AWSVIP_ERR_GETPRIIP_CODE=57				# failed to get the primary private ip address.
AWSVIP_ERR_CHKENIID_CODE=58				# eni id does not exist.


##
# resultmsg
##
AWSVIP_ERR_AWSCLI_MSG="The AWS CLI command failed."
AWSVIP_ERR_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSVIP_ERR_GETOCFENV_MSG="Failed to obtain the setting value."
AWSVIP_ERR_CHKSUB_MSG=""	#The VIP address ${VIRTUAL_IP} belongs to a VPC subnet.
AWSVIP_ERR_ADDVIP_MSG=""	#Failed to add the VIP address ${VIRTUAL_IP}.
# AWSVIP_ERR_DELVIP_MSG=""	#Failed to delete the VIP address ${VIRTUAL_IP}.
AWSVIP_ERR_GETNICNAME_MSG="Failed to obtain a NIC name."
AWSVIP_ERR_GETPRIIP_MSG="Failed to obtain a primary private IP address."
AWSVIP_ERR_CHKENIID_MSG=""	#ENI ID is invalid.(ENI ID=%1)


##
# set the value to use for the shmrmset command
##
GROUP_NAME="${CLP_GROUPNAME}"			# group name
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="awsvip"					# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${GROUP_NAME}" ] || [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# awscli path
##
export PATH=$PATH:/usr/local/bin


##
# awscli path search
##
which aws > /dev/null 2>&1

if [ $? -ne 0 ]
then
	set_alert_message "${AWSVIP_ERR_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_SCHAWSCLI_CODE}
fi


##
# set environment variables for use with the aws cli
##
clpcloudutil_env_init "${clpcloudutil_true}" "${CURRENT}/clpaws_setting.conf"


##
# get aws cli command line options
##
EC2_CMDOPT="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" ec2`"


##
# set ocf environment variable
##
VIRTUAL_IP="${CLP_OCF_PARAM1}"			# virtual ip address
VPC_ID="${CLP_OCF_PARAM2}"				# VPC ID
ENI_ID="${CLP_OCF_PARAM3}"				# ENI ID
PING_TIMEOUT="${CLP_OCF_PARAM4}"		# ping timeout

if [ -z "${VIRTUAL_IP}" ] || [ -z "${VPC_ID}" ] || [ -z "${ENI_ID}" ] || [ -z "${PING_TIMEOUT}" ]
then
	set_alert_message "${AWSVIP_ERR_GETOCFENV_MSG}"
	echo "failed to get the ocf environment variable." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_GETOCFENV_CODE}
fi

CIDR_BLOCK="/32"
VIP_CIDR_BLOCK="$VIRTUAL_IP$CIDR_BLOCK"


##
# check subnet
##

VPC_CIDR="`aws ec2 describe-vpcs --vpc-ids \"${VPC_ID}\" --query \"Vpcs[].CidrBlock\" ${EC2_CMDOPT} --output text 2>&1`"

if [ $? -ne 0 ]
then
	awscli_err_handle "${VPC_CIDR}"
	echo "failed to get the vpc cidrblock." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_AWSCLI_CODE}
fi

VPC_IP="`echo ${VPC_CIDR%/*}`"
VPC_PREFIX="`echo ${VPC_CIDR##*/}`"

echo "check subnet start. (vpc_ip=${VPC_IP} vpc_pre=${VPC_PREFIX})"

CHKSUB_NUM=$(ipconv_32bit ${VIRTUAL_IP})
CHKSUB_NET=$(( $(ipconv_32bit ${VPC_IP}) & $(cidrconv_32bit ${VPC_PREFIX}) ))
CHKSUB_BRD=$(( $(ipconv_32bit ${VPC_IP}) | (0xFFFFFFFF ^ $(cidrconv_32bit ${VPC_PREFIX})) ))

if [ "${CHKSUB_NET}" -le "${CHKSUB_NUM}" -a "${CHKSUB_NUM}" -le "${CHKSUB_BRD}" ]
then
	AWSVIP_ERR_CHKSUB_MSG="The VIP address \"${VIRTUAL_IP}\" belongs to a VPC subnet."
	set_alert_message "${AWSVIP_ERR_CHKSUB_MSG}"
	echo "vip ${VIRTUAL_IP} is belong to subnet of vpc ${VPC_ID}." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_CHKSUB_CODE}
fi


##
# check ENI ID
##
PRIMARY_IP=$(aws ec2 describe-network-interfaces --network-interface-ids "${ENI_ID}" \
	--query "NetworkInterfaces[].PrivateIpAddresses[?Primary==\`true\`].PrivateIpAddress" --output text ${EC2_CMDOPT} 2>&1)
GETPRIIP_RET=$?
if [ ${GETPRIIP_RET} -ne 0 ]
then
	awscli_err_handle "${PRIMARY_IP}"
	echo "failed to get the primary private ip address." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_AWSCLI_CODE}
elif [ -z "${PRIMARY_IP}" ]
then
	set_alert_message "${AWSVIP_ERR_GETPRIIP_MSG}"
	echo "primary ip address does not exist." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_GETPRIIP_CODE}
fi

CHKENIID_FLAG=0
OS_IP_LIST=$(ip -f inet -o addr show|cut -d\  -f 7 | cut -d/ -f 1)

IFS=$'\n'
for OS_IP in ${OS_IP_LIST}
do
	if [ "${PRIMARY_IP}" == "${OS_IP}" ]
	then
		echo "eni id ${ENI_ID} exists."
		CHKENIID_FLAG=$(( CHKENIID_FLAG + 1 ))
		break
	fi
done
unset IFS

if [ ${CHKENIID_FLAG} -eq 0 ]
then
	AWSVIP_ERR_CHKENIID_MSG="ENI ID is invalid.(ENI ID=\"${ENI_ID}\")"
	set_alert_message "${AWSVIP_ERR_CHKENIID_MSG}"
	echo "eni id ${ENI_ID} does not exist." 1>&2
	echo "primary IP address is ${PRIMARY_IP}." 1>&2
	OS_IP_LIST=$(echo ${OS_IP_LIST} | sed -e "s/\n//g")
	echo "ip address assigned to the os is ${OS_IP_LIST}." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_CHKENIID_CODE}
fi


##
# get routetable list
##
RT_IDLIST="`aws ec2 describe-route-tables \
	--filters Name=vpc-id,Values=\"${VPC_ID}\" Name=route.destination-cidr-block,Values=\"${VIP_CIDR_BLOCK}\" \
		--query \"RouteTables[].RouteTableId\" ${EC2_CMDOPT} --output text 2>&1`"
if [ $? -ne 0 ]
then
	awscli_err_handle "${RT_IDLIST}"
	echo "failed to get the routetable list." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_AWSCLI_CODE}
elif [ -z "${RT_IDLIST}" ]
then
	echo "route table of vpc ${VPC_ID} does not exist."
fi


##
# replace route
##
IFS=$' \t'

for RT_ID in `echo ${RT_IDLIST}`
do
	REPLACE_ROUTE="`aws ec2 replace-route --route-table-id \"${RT_ID}\" \
		--destination-cidr-block \"${VIP_CIDR_BLOCK}\" --network-interface-id \"${ENI_ID}\" ${EC2_CMDOPT} 2>&1`"
	if [ $? -ne 0 ]
	then
		awscli_err_handle "${REPLACE_ROUTE}"
		echo "route table ${RT_ID} failed to replace." 1>&2
		echo "clpawsvip start script has failed." 1>&2
		exit ${AWSVIP_ERR_AWSCLI_CODE}
	fi
done

unset IFS

echo "replace route has succeeded."


##
# check virtual ip exists
##
clpping -c ${PING_TIMEOUT} "${VIRTUAL_IP}" > /dev/null 2>&1
if [ $? -eq 0 ]
then
	echo "skip the process because the virtual ip address is exists."
	echo "virtual ip address is exists."
	echo "clpawsvip start script has succeeded."
	exit ${AWSVIP_SUCCESS_CODE}
fi


##
# nic name acquisition process
##
MAC_ADDRESS="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
				--query \"NetworkInterfaces[].[MacAddress]\" ${EC2_CMDOPT} --output text 2>&1`"
if [ $? -ne 0 ]
then
	awscli_err_handle "${MAC_ADDRESS}"
	echo "failed to get mac address." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_AWSCLI_CODE}
fi

NIC_NAME="`ip link | awk '/^[0-9]/ { iface=$2 } /link/ { mac=$2 } /brd/ { print iface, mac }' | grep \"${MAC_ADDRESS}\" | awk '{ sub(\":\", \"\") }  { print $1 }'`"
if [ $? -ne 0 ] || [ -z "${NIC_NAME}" ]
then
	set_alert_message "${AWSVIP_ERR_GETNICNAME_MSG}"
	echo "failed to get nic name (mac=${MAC_ADDRESS})." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_GETNICNAME_CODE}
fi

echo "obtaining a nic name has succeeded. (mac=${MAC_ADDRESS} nic=${NIC_NAME})"


##
# add virtual ip
##
ip address add "${VIP_CIDR_BLOCK}" dev "${NIC_NAME}"
if [ $? -ne 0 ]
then
	AWSVIP_ERR_ADDVIP_MSG="Failed to add the VIP address \"${VIRTUAL_IP}\"."
	set_alert_message "${AWSVIP_ERR_ADDVIP_MSG}"
	echo "failed to add virtual ip address." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_ADDVIP_CODE}
fi


##
# check virtual ip exists
##
clpping -c ${PING_TIMEOUT} "${VIRTUAL_IP}" > /dev/null 2>&1
if [ $? -ne 0 ]
then
	AWSVIP_ERR_ADDVIP_MSG="Failed to add the VIP address \"${VIRTUAL_IP}\"."
	set_alert_message "${AWSVIP_ERR_ADDVIP_MSG}"
	echo "failed to check virtual ip address." 1>&2
	echo "clpawsvip start script has failed." 1>&2
	exit ${AWSVIP_ERR_ADDVIP_CODE}
fi


##
# end
##
echo "clpawsvip start script has succeeded."

exit ${AWSVIP_SUCCESS_CODE}

