#!/bin/bash

#
#  usage : clpawsforcestop.sh <sub-command> <option>
#
#  sub-command :
#         stop <instance-id> <mode>
#         check-stop <instance-id>
#         monitor <instance-id> <mode>
#


# clpcloudutilの呼び出し
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"

# AWS CLIの設定
export PATH=$PATH:/usr/local/bin

# アラート関連の設定
max_shmnm_size=$((64 - 1))
awsfst_err_awscli_msg="The AWS CLI command failed."
awsfst_err_schawscli_msg="The AWS CLI command is not found."
awsfst_err_stopprotect_msg="Protection against stopping instances is enabled."
awsfst_err_stopinstance_msg="Stopping instances failed to be completed."
awsfst_err_internal_msg="Internal error occurred."

# インスタンス停止保護機能チェックのフラグ
# trueの場合アラートログ文字列を共有メモリへ設定する
# シェルスクリプトのif文の判定に合わせて 0 を true にする
monflag_true=0
monflag_false=1

# サブコマンドの取得
sub_command="$1"


#======================================
# アラートログ文字列を共有メモリへ設定する
#======================================
function set_alert_message {
    detail=$1
    result=$(clpshmnmset --descript --fst -m "${detail}" 2>&1)
    shmnmset_ret=$?
    if [ ${shmnmset_ret} -ne 0 ]
    then
        echo "An error has occurred with the 'clpshmnmset' command. (${shmnmset_ret})" 1>&2
        echo "${result}" | sed -n 2p 1>&2
        # 処理継続
    fi
}


#======================================
# AWS CLI エラー処理
#======================================
function awscli_err_handle {
    detail=$1
    echo "${detail}" 1>&2
    awscli_altmsg=$(ExtractAWSErrorCause "${detail}" ${max_shmnm_size})
    errcause_ret=$?
    if [ ${errcause_ret} -ne 0 ]
    then
        awscli_altmsg="${awsfst_err_awscli_msg}"
    fi
    set_alert_message "${awscli_altmsg}"
}


#======================================
# インスタンス停止保護機能チェック
#======================================
function chk_stop_protection {
    monflag=$1
    dapistop=$(aws ec2 describe-instance-attribute \
        --instance-id "${instance_id}" \
            --attribute disableApiStop \
                --query DisableApiStop.Value ${ec2_cmdopt} --output json 2>&1)
    dapistop_ret=$?
    if [ ${dapistop_ret} -ne 0 ]
    then
        echo "Command Error. [aws ec2 describe-instance-attribute]" 1>&2
        if [ "${monflag}" -eq ${monflag_true} ]
        then
            awscli_err_handle "${dapistop}"
            exit 1
        else
            echo "${dapistop}" 1>&2
        fi
    elif [ "${dapistop}" == "true" ]
    then
        echo "Instance stop protection is enabled." 1>&2
        if [ "${monflag}" -eq ${monflag_true} ]
        then
            set_alert_message "${awsfst_err_stopprotect_msg}"
            exit 1
        fi
    fi
}


#==================================
# AWS CLIの存在確認
#==================================
which aws > /dev/null 2>&1

if [ $? -ne 0 ]
then
    set_alert_message "${awsfst_err_schawscli_msg}"
    echo "AWSCLI Was Not Found." 1>&2
    exit 1
fi


#==================================
# AWS CLIで使用する環境変数の設定
#==================================
clpcloudutil_env_init "${clpcloudutil_false}" "${CURRENT}/clpaws_setting.conf"


#======================================
# AWS CLIコマンドラインオプションの取得
#======================================
ec2_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_false}\" ec2`"


#==================================
# AWS強制停止実行
#==================================
if [ "${sub_command}" == "stop" ]
then

    if [ "$#" -ne 3 ]
    then
        set_alert_message "${awsfst_err_internal_msg}"
        echo "Invalid Option!!" 1>&2
        exit 1
    fi

    instance_id="$2"; mode="$3"

    if [ "${mode}" == "forcestop" ]
    then
        exec_forcestop=$(aws ec2 stop-instances --instance-ids "${instance_id}" --force ${ec2_cmdopt} --output text 2>&1)
        if [ $? -ne 0 ]
        then
            awscli_err_handle "${exec_forcestop}"
            echo "Command Error. [aws ec2 stop-instances --force]" 1>&2
            chk_stop_protection ${monflag_false}
            exit 1
        fi
    elif [ "${mode}" == "stop" ]
    then
        exec_stop=$(aws ec2 stop-instances --instance-ids "${instance_id}" ${ec2_cmdopt} --output text 2>&1)
        if [ $? -ne 0 ]
        then
            awscli_err_handle "${exec_stop}"
            echo "Command Error. [aws ec2 stop-instances]" 1>&2
            chk_stop_protection ${monflag_false}
            exit 1
        fi
    elif [ "${mode}" == "reboot" ]
    then
        exec_reboot=$(aws ec2 reboot-instances --instance-ids "${instance_id}" ${ec2_cmdopt} --output text 2>&1)
        if [ $? -ne 0 ]
        then
            awscli_err_handle "${exec_reboot}"
            echo "Command Error. [aws ec2 reboot-instances]" 1>&2
            exit 1
        fi
    else
        set_alert_message "${awsfst_err_internal_msg}"
        echo "Invalid Parameter!!" 1>&2
        exit 1
    fi


#==================================
# AWS強制停止状態確認
#==================================
elif [ "${sub_command}" == "check-stop" ]
then

    if [ "$#" -ne 2 ]
    then
        set_alert_message "${awsfst_err_internal_msg}"
        echo "Invalid Option!!" 1>&2
        exit 1
    fi

    instance_id="$2"

    check_instance_state="`aws ec2 describe-instances --instance-ids \"${instance_id}\" \
                --filters \"Name=instance-state-name,Values=stopped\" ${ec2_cmdopt} --output text 2>&1`"
    if [ $? -ne 0 ]
    then
        awscli_err_handle "${check_instance_state}"
        echo "Command Error. [aws ec2 describe-instances(check-stop)]" 1>&2
        exit 1
    elif [ -z "${check_instance_state}" ]
    then
        set_alert_message "${awsfst_err_stopinstance_msg}"
        echo "Instance was not stopped." 1>&2
        exit 1
    fi


#==================================
# AWS強制停止定期確認
#==================================
elif [ "${sub_command}" == "monitor" ]
then

    if [ "$#" -ne 3 ]
    then
        set_alert_message "${awsfst_err_internal_msg}"
        echo "Invalid Option!!" 1>&2
        exit 1
    fi

    instance_id="$2"; mode="$3"
    authmsg_dryrn="DryRunOperation"

    if [ "${mode}" == "stop" ]
    then
        chk_stop_protection ${monflag_true}
        check_dryrun="`aws ec2 stop-instances --instance-ids \"${instance_id}\" --dry-run ${ec2_cmdopt} --output text 2>&1`"
        echo "${check_dryrun}" | grep "${authmsg_dryrn}" > /dev/null 2>&1
        if [ $? -ne 0 ]
        then
            awscli_err_handle "${check_dryrun}"
            echo "Command Error. [aws ec2 stop-instances --dry-run]" 1>&2
            exit 1
        fi
    elif [ "${mode}" == "reboot" ]
    then
        check_dryrun="`aws ec2 reboot-instances --instance-ids \"${instance_id}\" --dry-run ${ec2_cmdopt} --output text 2>&1`"
        echo "${check_dryrun}" | grep "${authmsg_dryrn}" > /dev/null 2>&1
        if [ $? -ne 0 ]
        then
            awscli_err_handle "${check_dryrun}"
            echo "Command Error. [aws ec2 reboot-instances --dry-run]" 1>&2
            exit 1
        fi
    else
        set_alert_message "${awsfst_err_internal_msg}"
        echo "Invalid Parameter!!" 1>&2
        exit 1
    fi

    check_instance_state="`aws ec2 describe-instances --instance-ids \"${instance_id}\" ${ec2_cmdopt} 2>&1`"
    if [ $? -ne 0 ]
    then
        awscli_err_handle "${check_instance_state}"
        echo "Command Error. [aws ec2 describe-instances(monitor)]" 1>&2
        exit 1
    fi


#==================================
# サブコマンド不正
#==================================
else
    set_alert_message "${awsfst_err_internal_msg}"
    echo "Invalid Sub Command!!" 1>&2
    exit 1
fi


#==================================
# 正常終了
#==================================
exit 0
