#!/bin/bash
# shellcheck disable=SC1091,SC2030,SC2031,SC2086,SC2001
#
##
# clpawseip-start.sh
##
# [coding：utf-8]
##


#
# グローバル変数定義
#
declare -g awsCmdLine


#
# モジュール読み込み
#
ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"


#
# コマンドライン作成番号
#
getPriIp=1
getEipInfo=2
eipAlloc=3


#
# 環境変数
#
export AWS_CONFIG_FILE="/root/.aws/config"
export AWS_SHARED_CREDENTIALS_FILE="/root/.aws/credentials"
export AWS_DEFAULT_OUTPUT="text"

export PATH="${PATH}:/usr/local/bin"


#
# スクリプト終了コード (= アラートイベントID)
#
Success=0							# succeed
ErrorAWSFailed=50					# failed in the awscli command
ErrorNotExistAWSCmd=51				# failed to search aws cli path
ErrorGetOcfEnvVariableFailed=52		# failed to get the ocf environment variable
ErrorAWSTimeout=53					# Timeout occurred.
ErrorGetPrimaryPrivateIp=54			# failed to get the primary private ip address.
ErrorCheckEniId=55					# eni id does not exist.
ErrorSetAWSEnvVariableFailed=56		# failed to set the aws cli environment variable
ErrorGetAWSAddingOptionsFailed=57	# failed to get aws cli command line options
ErrorInternal=79					# internal error


#
# アラートログ出力メッセージ
#
# (*1) {0}には、AWS CLIメッセージからエラー原因を抽出して設定
#	  抽出不可の場合は「The AWS CLI command was failed.」
#
declare -A AlertMessageTable=(
	[$Success]="Command succeeded."
	[$ErrorAWSFailed]="The AWS CLI command failed."
	[$ErrorNotExistAWSCmd]="The AWS CLI command was not found."
	[$ErrorGetOcfEnvVariableFailed]="Failed to obtain the setting value."
	[$ErrorAWSTimeout]="Timeout occurred."
	[$ErrorGetPrimaryPrivateIp]="Failed to obtain a primary private IP address."
	[$ErrorCheckEniId]="ENI ID is invalid.(ENI ID=$CLP_OCF_PARAM2)"
	[$ErrorInternal]="Initialize error occurred."
)


#
# アラートログ出力用の共通メモリ最大領域
#
ShmMaxSize=$((128 - 1))


#
# 共通モジュール出力フラグ
#
EnableUtilLogging=0
DisableUtilLogging=1


# ----------------------------------------------------------
#
# 関数定義
#
# ----------------------------------------------------------


#
# [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
# ※未設定の場合は、clpaws_setting.conf を環境変数に設定
#
function SetAWSEnvironmentVariable {
	awsSettingConf="${ScriptRoot}/clpaws_setting.conf"
	clpcloudutil_env_init $EnableUtilLogging "$awsSettingConf"
}


#
# [クラウド: AWS CLI コマンドラインオプション] の取得
#
function GetAWSAddingOptions {
	awsService="ec2"
	clpcloudutil_awscli_cmdopt $DisableUtilLogging $awsService
}


#
# アラートログ出力文字列を共有メモリへ設定する
#
function SetAlertMessage {
	alertMessage=$1
	grpName="${CLP_GROUPNAME}"		# group name
	rscType="awseip"				# resource type
	rscName="${CLP_RESOURCENAME}"	# resource name
	cmdLine="clpshmrmset --descript --rsc -g '$grpName' -t $rscType -n '$rscName' -m '$alertMessage'"
	result=$(eval "$cmdLine" 2>&1)
	shmRmSetExitCode=$?
	if [[ $shmRmSetExitCode -ne 0 ]]; then
		WriteStdOut "[CommandLine] $cmdLine"
		WriteStdErr "The 'clpshmrmset' command failed. ($shmRmSetExitCode)"
		WriteStdErr "$result"
		# 処理継続
	fi
}


#
# AWS CLIの実行コマンドライン設定
#
function CmdLineMake() {
	case $1 in
	"$getPriIp") query="NetworkInterfaces[].PrivateIpAddresses[?Primary==\\\`true\\\`].PrivateIpAddress"
			   awsCmdLine="aws ec2 describe-network-interfaces --network-interface-ids $eniId --query $query" ;;
	"$getEipInfo") query="Addresses[].NetworkInterfaceId"
				 awsCmdLine="aws ec2 describe-addresses --allocation-ids $eipAllocationId --query $query" ;;
	"$eipAlloc") awsCmdLine="aws ec2 associate-address --allow-reassociation --network-interface-id $eniId --allocation-id $eipAllocationId" ;;
	*)
		SetAlertMessage "${AlertMessageTable[$ErrorInternal]}" 
		WriteStdErr "Internal error."
		WriteStdErr "clpawseip start script has failed."
		exit $ErrorInternal ;;
	esac
	if [[ -n $awsAddingOptions ]]; then
		awsCmdLine="$awsCmdLine $awsAddingOptions"
	fi
	WriteStdOut "$awsCmdLine"
}


#
# AWS CLIの実行
#
function CliExec() {
	# ※シグナル指定した上でタイムアウト発生時の終了コードは、128 + SIGNAL
	exitCodeWithTimeout=$((128 + 9))
	# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
	eval "$(eval "timeout -s SIGKILL $awsTimeoutSec $awsCmdLine" \
				2> >(awsResultErr=$(cat); declare -p awsResultErr) \
				1> >(awsResult=$(cat); declare -p awsResult); \
				awsExitCode=$?; declare -p awsExitCode)"
	if [[ $awsExitCode -eq $exitCodeWithTimeout ]]; then
		# AWS CLIタイムアウト
		SetAlertMessage "${AlertMessageTable[$ErrorAWSTimeout]}"
		WriteStdErr "The AWS CLI command timed out. (awsTimeoutSec:$awsTimeoutSec)"
		exit $ErrorAWSTimeout
	fi
	if [[ $awsExitCode -ne 0 ]]; then
		# AWS CLI異常終了
		WriteStdErr "The 'aws' command failed. ($awsExitCode)"
		awsResultErr=$(echo "$awsResultErr" | TrimWhiteSpace)
		WriteStdErr "$awsResultErr"
		if cause=$(ExtractAWSErrorCause "$awsResultErr" "$ShmMaxSize" 2>&1); then
			SetAlertMessage "$cause"
			exit $ErrorAWSFailed
		fi
		SetAlertMessage "${AlertMessageTable[$ErrorAWSFailed]}"
		exit $ErrorAWSFailed
	fi
	WriteStdOut "$awsResult"
}


#
# 標準出力に文字列を出力する
#
function WriteStdOut {
	local message=$1
	# printf "[STDOUT] %04d: %s\n" "${BASH_LINENO[0]}" "$message"
	echo "$message"
}


#
# 標準エラー出力に文字列を出力する
#
function WriteStdErr {
	local message=$1
	# printf "[STDERR] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
	echo "[STDERR] $message" >&2
}




#
# clpawseip　開始
#
WriteStdOut "clpawseip start script start."


#
# AWS CLIインストール確認 (コマンドの存在確認)
#
if ! command -v aws >/dev/null; then
	SetAlertMessage "${AlertMessageTable[$ErrorNotExistAWSCmd]}"
	WriteStdErr "The AWS CLI command was not found."
	exit $ErrorNotExistAWSCmd
fi


#
# [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
# ※未設定の場合は、clpaws_setting.conf を環境変数に設定
#
if ! SetAWSEnvironmentVariable; then
	WriteStdErr "An error occurred while setting AWS environment variables.(error: $ErrorSetAWSEnvVariableFailed)"
	# 処理継続
fi


#
# [クラウド: AWS CLI コマンドラインオプション] の取得
#
if ! awsAddingOptions=$(GetAWSAddingOptions); then
	WriteStdErr "An error occurred while retrieving AWS additional options.(error: $ErrorGetAWSAddingOptionsFailed)"
	# 処理継続
fi


#
# ocf 環境変数取得
#
eipAllocationId=$CLP_OCF_PARAM1		# EIP ALLOCATION ID
eniId=$CLP_OCF_PARAM2				# ENI ID
awsTimeoutSec=$CLP_OCF_PARAM3		# AWS CLI Timeout
if [[ -z "$eipAllocationId"  || -z "$eniId" || -z "$awsTimeoutSec" ]]; then
	SetAlertMessage "${AlertMessageTable[$ErrorGetOcfEnvVariableFailed]}"
	WriteStdErr "failed to get the ocf environment variable."
	WriteStdErr "clpawseip start script has failed."
	exit $ErrorGetOcfEnvVariableFailed
fi


#
# ENI IDの確認
#
CmdLineMake $getPriIp
awsResult=$(CliExec)
exitcode=$?
if [[ $exitcode -ne 0 ]]; then
	exit $exitcode
fi
if [[ -z $awsResult ]]; then
	SetAlertMessage "${AlertMessageTable[$ErrorGetPrimaryPrivateIp]}"
	WriteStdErr "primary ip address does not exist."
	WriteStdErr "clpawseip start script has failed."
	exit $ErrorGetPrimaryPrivateIp
fi
chkeniidFlag=0
osIpList=$(ip -f inet -o addr show|cut -d\  -f 7 | cut -d/ -f 1)
IFS=$'\n'
for osIp in $osIpList
do
	if [[ $awsResult == "$osIp" ]]; then
		WriteStdOut "eni id $eniId exists."
		chkeniidFlag=$(( chkeniidFlag + 1 ))
		break
	fi
done
unset IFS
if [[ $chkeniidFlag -eq 0 ]]; then
	SetAlertMessage "${AlertMessageTable[$ErrorCheckEniId]}"
	WriteStdErr "eni id ${eniId} does not exist."
	WriteStdErr "primary IP address is ${awsResult}."
	osIpList=$(echo ${osIpList} | sed -e "s/\n//g")
	WriteStdErr "ip address assigned to the os is ${osIpList}."
	WriteStdErr "clpawseip start script has failed."
	exit $ErrorCheckEniId
fi


#
# EIP関連の情報を取得する
#
CmdLineMake $getEipInfo
awsResult=$(CliExec)
exitcode=$?
if [[ $exitcode -ne 0 ]]; then
	exit $exitcode
fi
# EIPが既に関連付けされている場合は正常終了
if [[ $awsResult == "$eniId" ]]; then
	WriteStdOut "Elastic IP is already associated."
	WriteStdOut "clpawseip start script has succeeded."
	exit $Success
fi


#
# EIPを関連付け
#
CmdLineMake $eipAlloc
CliExec


#
# end
#
WriteStdOut "clpawseip start script has succeeded."
exit $Success
