#!/bin/bash
#
##
# clpawsdns-stop.sh
##
# [coding：utf-8]
##
# shellcheck disable=SC1091,SC2030,SC2031,SC2086

##
# 関数定義
##


# アラートログ文字列を共有メモリへ設定する
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --rsc -g "${GROUP_NAME}" -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_EXIT_CODE=$?
	if [[ ${SHMRMSET_EXIT_CODE} -ne 0 ]]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_EXIT_CODE})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# 処理継続
	fi
}


# 0以上の整数か判定する
function is_non_negative_number {
	NUM=$1
	if [[ $NUM =~ ^[0-9]+$ && $NUM -ge 0 ]]
	then
		return 0
	else
		return 1
	fi
}


# AWS CLIの引数に指定するjsonコードの作成
function create_jsoncode {
	JSON_CODE=$(cat << EOS
{
	"Comment": "Deleting an A record",
	"Changes": [
		{
			"Action": "DELETE",
			"ResourceRecordSet": {
				"Name": "${RECORD_NAME}",
				"Type": "A",
				"TTL": ${RECORD_TTL},
				"ResourceRecords": [
					{
						"Value": "${RECORD_IP}"
					}
				]
			}
		}
	]
}
EOS
	)
	echo "${JSON_CODE}"
}


# レコードセットの削除に使うjsonファイルの削除
function delete_jsonfile {
	rm -f "${DELETE_JSON_FILE}"
}


##
# 処理開始
##

echo "clpawsdns stop script start."


##
# clpcloudutilの呼び出し
##
SCRIPT_ROOT=$(dirname "$(realpath "$0")")
. "${SCRIPT_ROOT}/../common/clpcloudutil.sh"

# clpcloudutil出力フラグ
ENABLE_UTIL_LOG=0
# DISABLE_UTIL_LOG=1


##
# 返値
##
AWSDNS_SUCCESS_CODE=0					# 成功
# AWSDNS_ERR_AWSCLI_CODE=50				# AWS CLI 失敗
AWSDNS_ERR_AWSCLITO_CODE=51				# AWS CLI タイムアウト
AWSDNS_ERR_SCHAWSCLI_CODE=52			# AWS CLI が存在しない
AWSDNS_ERR_GETOCFENV_CODE=53			# OCF環境変数の取得に失敗
# AWSDNS_ERR_NOTEXIST_REC_CODE=54		# レコードセットが存在しない
# AWSDNS_ERR_NOTMATCH_RECIP_CODE=55		# レコードセットのIPが不正
AWSDNS_ERR_INTERNAL_CODE=79				# 内部エラー


##
# エラーメッセージ
##
# AWSDNS_ERR_AWSCLI_MSG="The AWS CLI command failed."
AWSDNS_ERR_AWSCLITO_MSG="Timeout occurred."
AWSDNS_ERR_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSDNS_ERR_GETOCFENV_MSG="Failed to obtain the setting value."
# AWSDNS_ERR_NOTEXIST_REC_MSG="The resource record set in Amazon Route 53 does not exist."
# AWSDNS_ERR_NOTMATCH_RECIP_MSG="IP address different from the setting is registered in the resource record set of Amazon Route 53."
AWSDNS_ERR_INTERNAL_MSG="Internal error occurred."


##
# シグナル指定した上でタイムアウト発生時の終了コード(128 + SIGNAL)
##
EXIT_CODE_WITH_TO=$((128 + 9))


##
# shmrmsetコマンド関連の初期化
##
if [[ -z ${CLP_RESOURCENAME} ]]
then
	set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
	echo "failed to obtain the resouce name." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_INTERNAL_CODE}
fi
if [[ -z ${CLP_GROUPNAME} ]]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
	# 処理継続 (継続後の処理エラー時はアラート出力不可)
fi
GROUP_NAME="${CLP_GROUPNAME}"			# グループ名
RESOURCE_NAME="${CLP_RESOURCENAME}"		# リソース名
RESOURCE_TYPE="awsdns"					# リソースタイプ
# MAX_SHMRM_SIZE=$((128 - 1))			# エラーメッセージの最大文字数


##
# AWS CLIの設定
##
export PATH=$PATH:/usr/local/bin


##
# AWS CLIの存在確認
##
which aws > /dev/null 2>&1
SCHAWSCLI_EXIT_CODE=$?
if [[ ${SCHAWSCLI_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNS_ERR_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawsdns stop script has failed." 1>&2
	exit ${AWSDNS_ERR_SCHAWSCLI_CODE}
fi


##
# AWS CLIで使用する環境変数の設定
##
clpcloudutil_env_init ${ENABLE_UTIL_LOG} "${SCRIPT_ROOT}/clpaws_setting.conf"
ENV_INIT_EXIT_CODE=$?
if [[ ${ENV_INIT_EXIT_CODE} -ne 0 ]]
then
	echo "Failed to obtain the environment variables." 1>&2
fi


##
# AWS CLIコマンドラインオプションの取得
##
RT53_CMDOPT=$(clpcloudutil_awscli_cmdopt ${ENABLE_UTIL_LOG} route53)
GET_AWSCLI_CMDOPT_EXIT_CODE=$?
if [[ ${GET_AWSCLI_CMDOPT_EXIT_CODE} -ne 0 ]]
then
	echo "Failed to obtain the aws cli command line options." 1>&2
fi


##
# OCF環境変数の取得
##
PARAM_NUM=8
for i in $(seq 1 $PARAM_NUM); do
	CLP_OCF_PARAM_VAR="CLP_OCF_PARAM${i}"
	if [[ -z ${!CLP_OCF_PARAM_VAR} ]]
	then
		set_alert_message "${AWSDNS_ERR_GETOCFENV_MSG}"
		echo "failed to get the ocf environment variable." 1>&2
		echo "clpawsdns stop script has failed." 1>&2
		exit ${AWSDNS_ERR_GETOCFENV_CODE}
	fi
done
HOSTZONE_ID="${CLP_OCF_PARAM1}"					# ホストゾーンID
RECORD_NAME="${CLP_OCF_PARAM2}"					# リソースレコードセット名
RECORD_IP="${CLP_OCF_PARAM3}"					# IPアドレス
RECORD_TTL="${CLP_OCF_PARAM4}"					# TTL(秒)
RECORD_DELETE_FLAG="${CLP_OCF_PARAM5}"			# 非活性時にリソースレコードセットを削除する
AWSCLI_TO="${CLP_OCF_PARAM6}"					# AWS CLI タイムアウト(秒)
# WAIT_INSYNC_FLAG="${CLP_OCF_PARAM7}"			# 権威DNSサーバへの伝搬を待機する
# WAIT_INSYNC_TO="${CLP_OCF_PARAM8}"			# 権威DNSサーバ伝搬タイムアウト (秒)
# 型チェック
for i in "RECORD_TTL" "RECORD_DELETE_FLAG" "AWSCLI_TO"
do
	if ! is_non_negative_number "${!i}"
	then
		set_alert_message "${AWSDNS_ERR_GETOCFENV_MSG}"
		echo "'$i' contains non-negative values. (${!i})" 1>&2
		echo "clpawsdns stop script has failed." 1>&2
		exit ${AWSDNS_ERR_GETOCFENV_CODE}
	fi
done
echo "HOSTZONE_ID: ${HOSTZONE_ID}"
echo "RECORD_NAME: ${RECORD_NAME}"
echo "RECORD_IP: ${RECORD_IP}"
echo "RECORD_TTL: ${RECORD_TTL}"
echo "RECORD_DELETE_FLAG: ${RECORD_DELETE_FLAG}"
echo "AWSCLI_TO: ${AWSCLI_TO}"


##
# リソースレコードセットの削除をスキップ
##
if [[ ${RECORD_DELETE_FLAG} -eq 0 ]]
then
	echo "skip delete resource record set."
	echo "clpawsdns stop script has succeeded."
	exit ${AWSDNS_SUCCESS_CODE}
fi


##
# <インストールパス>/work/awsdsnディレクトリを作成する
##
#パスの作成
WORK_AWSDNS_PATH="${SCRIPT_ROOT}/../../work/awsdns"
#<インストールパス>/work/awsdsnディレクトリが存在しない場合は作成
if [[ ! -d "${WORK_AWSDNS_PATH}" ]]
then
	umask 033
	mkdir "${WORK_AWSDNS_PATH}"
	MKDIR_EXIT_CODE=$?
	if [[ ${MKDIR_EXIT_CODE} -ne 0 ]]
	then
		set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
		echo "failed to make work directory." 1>&2
		echo "clpawsdns stop script has failed." 1>&2
		exit ${AWSDNS_ERR_INTERNAL_CODE}
	fi
	#ディレクトリの作成を待ち合わせ
	DIR_RETRY=10
	DIR_INTERVAL=1
	DIR_CHK_FLAG=0
	for i in $(seq ${DIR_RETRY})
	do
		sleep ${DIR_INTERVAL}
		if [[ -d "${WORK_AWSDNS_PATH}" ]]
		then
			echo "make a work directory succeeded."
			DIR_CHK_FLAG=$(( DIR_CHK_FLAG + 1 ))
			break
		fi
	done
	#<インストールパス>/work/awsdsnディレクトリの作成に失敗
	if [[ ${DIR_CHK_FLAG} -eq 0 ]]
	then
		set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
		echo "failed to make work directory." 1>&2
		echo "clpawsdns stop script has failed." 1>&2
		exit ${AWSDNS_ERR_INTERNAL_CODE}
	fi
fi


##
# レコードセットの削除
##
# レコードセットの削除に使うjsonファイルの作成
DELETE_JSON_FILE="${WORK_AWSDNS_PATH}/delete_record_${RESOURCE_NAME}.json"
# リソースレコードセット名にエスケープコードを含む場合、エスケープコードをエスケープする
if [[ "${RECORD_NAME}" =~ "\\" ]]
then
	RECORD_NAME_ORG="${RECORD_NAME}"
	RECORD_NAME=${RECORD_NAME//\\/\\\\}
fi
DELETE_JSON_CODE=$(create_jsoncode)
echo "${DELETE_JSON_CODE}" > "${DELETE_JSON_FILE}"
# リソースレコードセット名を元に戻す
if [[ "${RECORD_NAME}" =~ "\\" ]]
then
	RECORD_NAME="${RECORD_NAME_ORG}"
fi

# AWS CLIの実行コマンドライン設定
AWS_CMDLINE="aws route53 change-resource-record-sets"
AWS_CMDLINE+=" --hosted-zone-id \"${HOSTZONE_ID}\""
AWS_CMDLINE+=" --change-batch \"file://${DELETE_JSON_FILE}\""
AWS_CMDLINE+=" --output text"
if [[ -n ${RT53_CMDOPT} ]]
then
	AWS_CMDLINE="${AWS_CMDLINE} ${RT53_CMDOPT}"
fi
echo "[CommandLine] ${AWS_CMDLINE}"

# AWS CLIの実行
# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
eval "$(eval "timeout -s SIGKILL ${AWSCLI_TO} ${AWS_CMDLINE}" \
			2> >(DELETE_RECORD_RET_ERR=$(cat); declare -p DELETE_RECORD_RET_ERR) \
			1> >(DELETE_RECORD_RET=$(cat); declare -p DELETE_RECORD_RET); \
			DELETE_RECORD_EXIT_CODE=$?; declare -p DELETE_RECORD_EXIT_CODE )"
# AWS CLIタイムアウト
if [[ ${DELETE_RECORD_EXIT_CODE} -eq ${EXIT_CODE_WITH_TO} ]]
then
	delete_jsonfile
	set_alert_message "${AWSDNS_ERR_AWSCLITO_MSG}"
	echo "the aws cli command timed out. (AWSCLI_TO:${AWSCLI_TO})" 1>&2
	echo "clpawsdns stop script has failed." 1>&2
	exit ${AWSDNS_ERR_AWSCLITO_CODE}
fi
# AWS CLI異常終了
# リソースレコードセットが存在しない場合を考慮して正常終了
if [[ ${DELETE_RECORD_EXIT_CODE} -ne 0 ]]
then
	delete_jsonfile
	echo "${DELETE_RECORD_RET_ERR}" 1>&2
	echo "failed to delete the resource record set." 1>&2
	echo "clpawsdns stop script has succeeded."
	exit ${AWSDNS_SUCCESS_CODE}
else
	echo "delete the resource record set succeeded."
fi


##
# 処理終了
##

# レコードセットの削除に使うjsonファイルの削除
delete_jsonfile

echo "clpawsdns stop script has succeeded."
exit ${AWSDNS_SUCCESS_CODE}
