#!/bin/sh

LANG=C
export LANG
LC_ALL=C
export LC_ALL

# Functions
logwrite () {
  echo `date +"%F %T"` "$*">>"${clp_log_path}" 2>&1
}

collectc () {
  logwrite "$1"

  _dat_file_path=${col_temp_dir}/$1
  shift
  _cmd_str="$*"

  $_cmd_str >>"${_dat_file_path}" 2>&1
}

collectf () {
  if [ "_$1" = "_" ]; then
    return 1
  fi

  _src_path=$1

  if [ ! -f "${_src_path}" ]; then
    logwrite "CopyFile[Src:${_src_path}] - No such file"
    return 1
  fi

  if [ "_$2" = "_" ]; then
    _dst_path=${col_temp_dir}
  else
    _dst_path=$2
  fi

  if [ ! -d "${_dst_path}" ]; then
    mkdir -p "${_dst_path}" >>"${clp_log_path}" 2>&1
  fi

  logwrite "CopyFile[Src:${_src_path}, Dst:${_dst_path}]"
  cp -fp "${_src_path}" "${_dst_path}" >>"${clp_log_path}" 2>&1
}

collectd () {
  if [ "_$1" = "_" ] || [ "_$2" = "_" ]; then
    return 1
  fi

  _src_path=$1
  _dst_path=$2
  _dst_parent_path=$(dirname "${_dst_path}")

  if [ ! -d "${_src_path}" ]; then
    logwrite "CopyDir[Src:${_src_path}] - No such directory"
    return 1
  fi

  if [ ! -d "${_dst_parent_path}" ]; then
    mkdir -p "${_dst_parent_path}" >>"${clp_log_path}" 2>&1
  fi

  logwrite "CopyDir[Src:${_src_path}, Dst:${_dst_path}]"
  cp -af "${_src_path}" "${_dst_path}" >>"${clp_log_path}" 2>&1
}

# Variables
ret_code=1
script_revision='$Revision: 19075 $'
typename=logplus
clp_log_dir=`clplogcf -p 2>/dev/null`
cur_log_file_name=${typename}.log.cur
pre_log_file_name=${typename}.log.pre
host_name=`uname -n 2>/dev/null`
dat_prefix=logplus_${host_name}
rndnum=$(echo $(od -vAn -N2 -tu4 < /dev/urandom))
tmp_prefix=${dat_prefix}_`date +"%Y%m%d_%H%M%S"`_${rndnum}
if [ -d "/tmp/${dat_prefix}" ]; then
  tmp_dir_name=$tmp_prefix
else
  tmp_dir_name=$dat_prefix
fi
col_temp_dir=/tmp/${tmp_dir_name}
cnf_temp_dir=${col_temp_dir}/clpconf
need_logcc=true
out_dir_opt=false
out_dir=""

# Check parameters
for arg in "$@"; do
  if $out_dir_opt ; then
    out_dir="$arg"
    out_dir_opt=false
  else
    [ "_$arg" = "_--nologcc" ] && need_logcc=false
    [ "_$arg" = "_-o" ] && out_dir_opt=true
  fi
done

if [ "${out_dir}" != "" ]; then
  # last "/" delete
  out_dir=${out_dir%/}
  if [ ! -d "${out_dir}" ]; then
    echo Error: Output directory does not exist.
    echo "  "${out_dir}
    exit $ret_code
  fi
fi

# Check admin
if [ "`id -u`" != "0" ]; then
  echo Error: You are not root. Please retry as root.
  exit $ret_code
fi

# Check clp install path
if [ "${clp_log_dir}" != "" ] && [ -d "${clp_log_dir}" ]; then
  clp_inst_dir=`dirname "${clp_log_dir}"`
else
  clp_logcc_symlnk=`which clplogcc 2>/dev/null`
  ret=$?
  if [ $ret -eq 0 ]; then
    clp_bin_dir=$(cd $(dirname $(readlink "${clp_logcc_symlnk}" 2>/dev/null) 2>/dev/null); pwd)
    clp_inst_dir=$(cd $(dirname "${clp_bin_dir}" 2>/dev/null); pwd)
    clp_log_dir=${clp_inst_dir}/log/
  else
    clp_inst_dir=
    clp_log_dir=
  fi
fi
if [ "${clp_inst_dir}" != "" ] && [ -d "${clp_inst_dir}" ]; then
  clp_event_path=${clp_inst_dir}/bin/clpevent
  clp_tmp_dir=${clp_inst_dir}/tmp
else
  clp_event_path=
  clp_tmp_dir=
fi

# Check clp install
if [ ! -f "${clp_event_path}" ]; then
  echo Warning: This server is not a cluster server. Please run on a cluster server.
fi

# Set log path
if [ -d "${clp_log_dir}" ]; then
  pre_log_path=${clp_log_dir}${pre_log_file_name}
  cur_log_path=${clp_log_dir}${cur_log_file_name}
  [ -f "${pre_log_path}" ] && rm -f "${pre_log_path}" >/dev/null 2>&1
  if [ -f "${cur_log_path}" ]; then
    cp -fp "${cur_log_path}" "${pre_log_path}" >/dev/null 2>&1
    rm -f "${cur_log_path}" >/dev/null 2>&1
  fi
  clp_log_path=${cur_log_path}
else
  clp_log_path=/dev/null
fi

# Set archive path
if [ "${out_dir}" != "" ]; then
  tgz_path=${out_dir}/${dat_prefix}.tar.gz
else
  if [ -d "${clp_tmp_dir}" ]; then
    tgz_path=${clp_tmp_dir}/${dat_prefix}.tar.gz
  else
    tgz_path=/tmp/${tmp_prefix}.tar.gz
    if [ -f "${tgz_path}" ]; then
      echo Error: Archive file is already exists. Please retry this tool.
      echo "  "$tgz_path
      logwrite "End[${ret_code}]"
      exit $ret_code
    fi
  fi
fi

# Dump parameters
logwrite "Starting[${script_revision}]"
logwrite "Params:$*"
logwrite "clp_inst_dir:${clp_inst_dir}"
logwrite "clp_log_path:${clp_log_path}"
logwrite "col_temp_dir:${col_temp_dir}"
logwrite "script_path:$(cd $(dirname "$0"); pwd)/$(basename "$0")"
logwrite "out_dir:${out_dir}"
logwrite "tgz_path:${tgz_path}"

# Create working directory
if [ -d "${col_temp_dir}" ]; then
  echo Error: Working directory is already exists. Please retry this tool.
  echo "  "$col_temp_dir
  logwrite "End[${ret_code}]"
  exit $ret_code
fi
mkdir -p "${col_temp_dir}" >>"${clp_log_path}" 2>&1
ret=$?
if [ $ret -ne 0 ]; then
  echo Error: Failed to create working directory. Please retry this tool.
  echo "  "$col_temp_dir
  logwrite "End[${ret_code}]"
  exit $ret_code
fi

# for AWS CLI
export PATH=$PATH:/usr/local/bin

echo Data Collecting...

# information of cluster
if [ "${clp_inst_dir}" != "" ] && [ -d "${clp_inst_dir}" ]; then
  # cluster configuration
  collectf "${clp_inst_dir}/etc/clp.conf" "${cnf_temp_dir}"
  collectd "${clp_inst_dir}/scripts" "${cnf_temp_dir}/scripts"

  # system/cluster/mirror statistics
  collectd "${clp_inst_dir}/perf" "${col_temp_dir}/perf"
  if [ -d "${clp_inst_dir}/log/system" ]; then
    collectd "${clp_inst_dir}/log/system" "${col_temp_dir}/perf/system"
  fi

  # filesystem information of md
  grep dppath "${clp_inst_dir}/etc/clp.conf" 2>/dev/null | sed -e "s/<dppath>\(.*\)<\/dppath>/\1/" | xargs -n1 tune2fs -l >>"${col_temp_dir}/tune2fs_md.txt" 2>&1

  # number of inode using
  for dir in `grep "<historydir>" "${clp_inst_dir}/etc/clp.conf" 2>/dev/null | sed -e 's/\s*<historydir>\(.*\)<\/historydir>.*$/\1/'`
  do
    if [ ! -d "${dir}" ]; then
      continue
    fi
    inode_num=$(cd "${dir}"; ls -1UR 2>/dev/null | grep -v "^\..*:$" | grep -v "^\s*$" | wc -l)
    collectc inode_histdir echo $dir
    collectc inode_histdir echo $inode_num
  done

  # clplogcc command
  logcc_dir=${col_temp_dir}/clplogcc
  mkdir -p "${logcc_dir}" >>"${clp_log_path}" 2>&1
  if [ -d "${logcc_dir}" ] && ${need_logcc}; then
    collectc logcc          clplogcc --local -o "${logcc_dir}"
  fi
fi

# various command
collectc logplusrev.txt     echo "${script_revision}"
collectc lshw               lshw
collectc lshw               lshw -short
collectc lshw               lshw -businfo
collectc dev.ls             ls -lR /dev/
collectc pvs                pvs -av
collectc vgs                vgs -av
collectc lvs                lvs -av
collectc parted             parted -lm
collectc lsblk              lsblk -f
collectc lsblk              lsblk -p
collectc lsblk_rota         lsblk -po NAME,ROTA
collectc blkid              blkid
collectc wdog               lsof /dev/watchdog
collectc tmpport            sysctl net.ipv4.ip_local_port_range
collectc rsvport            sysctl net.ipv4.ip_local_reserved_ports
collectc awscli             aws ec2 describe-availability-zones
collectc getenforce         getenforce
collectc varlog.ls          ls -lR /var/log/
collectc env                env
collectc date.stat          clpstat --date
collectc stat.stat          clpstat --local
collectc rpm.ver            dpkg -l
collectc glibc.ver          dpkg -s libc-bin
collectc libxml2.ver        dpkg -s libxml2
collectc rc_list            ls -l /etc/rc*.d/
collectc ps_th.result       ps -eL auxww
collectc netstat.if         netstat -in
collectc netstat.v4         netstat -rn
collectc netstat.v6         netstat --inet6 -rn
collectc ss                 ss
collectc ss_nap             ss -nap
collectc ip_opt_sl          ip -s l
collectc chkconfig-all.list ls -l /etc/rc*/
collectc chkconfig-all.list initctl list
collectc unit-all.list      ls -l --full-time /lib/systemd/system/
collectc unit-all.list      ls -l --full-time /usr/lib/systemd/system/
collectc systemctl_svcstat  systemctl -t service
collectc systemctl_svcconf  systemctl list-unit-files -t service
collectc chkconfig-allsrv   ls /etc/rc*.d/
# collectc journal.clp        journalctl -u clusterpro_evt
# collectc journal.tail       journalctl -n 100000
collectc vgdisplay          vgdisplay -v
collectc ip_addr            ip addr show
collectc ip_link            ip link show
collectc ip_maddr           ip maddr show
collectc ip_route           ip route show
collectc ip_neigh           ip neigh show
collectc arp                arp -av
collectc arp                arp -anv
# collectc rpcinfo            rpcinfo -p
# collectc iptables           iptables -L
# collectc iptables           echo `service iptables status`
collectc iptables           ufw status verbose
collectc firewalld          service firewalld status
collectc firewalld_active   firewall-cmd --get-active-zones
collectc firewalld_list     firewall-cmd --list-all-zones
collectc multipath          multipath -l
collectc dmidecode          dmidecode
collectc uptime             uptime
collectc snmp.ver           snmpd -v
collectc virt-what          virt-what
collectc virt-what          systemd-detect-virt
collectc blockdev           blockdev --report
collectc nmcli              nmcli device show
collectc nmcli              nmcli connection show
collectc df_inode           df -i
collectc bootctl            bootctl status
collectc uname              uname -a
collectc hostname           uname -n
collectc clpcloud_pftype    "${clp_inst_dir}/cloud/common/clpcloud.sh" get-cloud-info
collectc clpselctrl_check   "${clp_inst_dir}/bin/clpselctrl.sh" --check
collectc semodule           semodule -l
collectc coredumpctl        coredumpctl

ls -1U /sys/class/net 2>/dev/null | xargs -n1 ethtool >>"${col_temp_dir}/ethtool" 2>&1
ls -1U /sys/class/net 2>/dev/null | xargs -n1 ethtool -k >>"${col_temp_dir}/ethtool_k" 2>&1
grep "^path " /etc/kdump.conf 2>/dev/null | sed -e "s/^path //g" | xargs -n1 df >>"${col_temp_dir}/df_kdump" 2>&1

# syslog
# mkdir -p "${col_temp_dir}/var_log" >>"${clp_log_path}" 2>&1
# if [ -d "${col_temp_dir}/var_log" ]; then
#   cp -fp /var/log/messages* "${col_temp_dir}/var_log" >>"${clp_log_path}" 2>&1
# fi

# boot settings
collectf /etc/default/grub
collectf /boot/efi/EFI/redhat/grub.conf
collectf /boot/grub/grub.conf
collectf /boot/grub2/grub.cfg
collectf /boot/efi/EFI/redhat/grub.cfg
collectf /boot/grub/menu.lst
collectf /etc/lilo.conf

# various conf
collectf /proc/sys/kernel/core_pattern
collectf /proc/sys/kernel/core_uses_pid
collectf /etc/rsyslog.conf
collectf /etc/sysctl.conf
collectf /etc/multipath.conf
collectf /etc/snmp/snmpd.conf
collectf /etc/lvm/lvm.conf
collectf /etc/kdump.conf
collectf /etc/sysconfig/kdump

if [ -f "/etc/selinux/config" ]; then
  cp -fp "/etc/selinux/config" "${col_temp_dir}/selinux_config" >>"${clp_log_path}" 2>&1
fi
if [ -f "/usr/lib/tmpfiles.d/systemd.conf" ]; then
  cp -fp "/usr/lib/tmpfiles.d/systemd.conf" "${col_temp_dir}/tmpfiles_systemd.conf" >>"${clp_log_path}" 2>&1
fi

echo Data Archiving...
logwrite "Archiving"
cd /tmp >>"${clp_log_path}" 2>&1
ret=$?
if [ $ret -ne 0 ]; then
  echo Error: Failed to create archive. Please collect following directory and archive manually.
  echo "-> ${col_temp_dir}"
  logwrite "End[${ret_code}]"
  exit $ret_code
fi
tar -czf "${tgz_path}" "${tmp_dir_name}" >>"${clp_log_path}" 2>&1
ret=$?
if [ $ret -ne 0 ]; then
  echo Error: Failed to create archive. Please collect following directory and archive manually.
  echo "-> ${col_temp_dir}"
  logwrite "End[${ret_code}]"
  exit $ret_code
fi

echo Success: Please collect following file.
echo "-> ${tgz_path}"
ret_code=0

# Terminating...
if [ -d "${col_temp_dir}" ]; then
  rm -rf "${col_temp_dir}" >>"${clp_log_path}" 2>&1
fi
logwrite "End[${ret_code}]"
exit $ret_code
