#! /bin/sh

#------- static variables -------#

readonly CMD_BKUP="backup"
readonly CMD_RSTR="restore"
readonly CMD_NAME="${CMD_BKUP}"
readonly SCR_NAME="clp${CMD_NAME}.sh"

readonly OPT_PRE="pre"
readonly OPT_POST="post"
readonly OPT_DOWN="only-shutdown"
readonly OPT_BOOT="only-reboot"
readonly OPT_NODN="no-shutdown"
readonly OPT_NOBT="no-reboot"
readonly OPT_SKIP="skip-copy"
readonly OPT_ONLINE="online"
readonly OPT_FREEZE="--freeze"
readonly OPT_UNFREEZE="--unfreeze"
readonly OPT_HELP="help"
readonly OPT_NONE=""

if   [ "${CMD_NAME}" = "${CMD_BKUP}" ]
then
	MSG_USAGE="${SCR_NAME}\n"
	MSG_USAGE+="\t--${OPT_PRE}\n"
	MSG_USAGE+="\t--${OPT_PRE} --${OPT_NODN}\n"
	MSG_USAGE+="\t--${OPT_PRE} --${OPT_DOWN}\n"
	MSG_USAGE+="\t--${OPT_PRE} --${OPT_ONLINE}\n"
	MSG_USAGE+="\t--${OPT_POST}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_NOBT}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_BOOT}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_ONLINE}\n"
	MSG_USAGE+="\t--${OPT_HELP}"
elif [ "${CMD_NAME}" = "${CMD_RSTR}" ]
then
	MSG_USAGE="${SCR_NAME}\n"
	MSG_USAGE+="\t--${OPT_PRE}\n"
	MSG_USAGE+="\t--${OPT_PRE} --${OPT_NODN}\n"
	MSG_USAGE+="\t--${OPT_PRE} --${OPT_DOWN}\n"
	MSG_USAGE+="\t--${OPT_POST}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_NOBT}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_BOOT}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_SKIP}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_SKIP} --${OPT_NOBT}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_ONLINE}\n"
	MSG_USAGE+="\t--${OPT_POST} --${OPT_ONLINE} --${OPT_SKIP}\n"
	MSG_USAGE+="\t--${OPT_HELP}"
else
	echo "Internal error." 1>&2
	exit 1
fi

readonly MSG_CMD_SUCCESS="${SCR_NAME} : Command succeeded."
readonly MSG_CMD_FAILED="${SCR_NAME} : Command failed."
readonly MSG_ERR_LOG_NOT_FOUND="${SCR_NAME} error : Log directory is not found."
readonly MSG_ERR_INVALID_OPTION="${SCR_NAME} error : Invalid option."
readonly MSG_ERR_BAD_CONFIG="${SCR_NAME} error : Set \"Initial Mirror Construction\" parameter and \"Initial mkfs\" parameter of md/hd resource to off by using Cluster WebUI."
readonly MSG_ERR_NOT_ROOT="${SCR_NAME} error : Log in as root."
readonly MSG_ERR_INTERNAL="${SCR_NAME} : Internal error."
readonly MSG_ACT_FLG_ON="${SCR_NAME} : Beginning backup-mode."
readonly MSG_ACT_FLG_OFF="${SCR_NAME} : Ending backup-mode."
readonly MSG_ACT_SVC_ON="${SCR_NAME} : Changing the setting of cluster services to Auto Startup."
readonly MSG_ACT_SVC_OFF="${SCR_NAME} : Changing the setting of cluster services to Manual Startup."
readonly MSG_ACT_START_AGENT="${SCR_NAME} : Starting Mirror Agent."
readonly MSG_ACT_STOP_AGENT="${SCR_NAME} : Stopping Mirror Agent."
readonly MSG_ACT_START_CLPM="${SCR_NAME} : Starting Cluster Service."
readonly MSG_ACT_STOP_CLPM="${SCR_NAME} : Stopping Cluster Service."
readonly MSG_ACT_SHUTDOWN="${SCR_NAME} : Shutting down..."
readonly MSG_ACT_REBOOT="${SCR_NAME} : Rebooting..."
readonly MSG_ACT_SUSPEND="${SCR_NAME} : Suspending Cluster Service."
readonly MSG_ACT_RESUME="${SCR_NAME} : Resuming Cluster Service."
readonly MSG_RSC_FROZEN="has been frozen."
readonly MSG_RSC_FROZEN_ALREADY="is already frozen."
readonly MSG_RSC_UNFROZEN="has been unfrozen."
readonly MSG_RSC_UNFROZEN_ALREADY="is not frozen."

readonly MSG_RET_VAL="  ret ="
readonly MSG_RET_CHK="result of checking config is"

readonly RSC_MD="md"
readonly RSC_HD="hd"
readonly FLG_ON="on"
readonly FLG_OFF="off"
readonly SVC_CORE_ON="--enable core"
readonly SVC_CORE_OFF="--disable core"
readonly GRP_AUTO_ON="enable"
readonly GRP_AUTO_OFF="disable"
readonly DWN_POWER_OFF=""
readonly DWN_REBOOT="-r"
readonly CL_START="-s"
readonly CL_STOP="-t"
readonly CL_NODESTAT="--nodestat"
readonly CL_SUSPEND="--suspend"
readonly CL_FORCE="--force"
readonly CL_RESUME="--resume"
readonly CL_API_TO=3600
readonly CFSET_INTERNAL="--internal-use"
readonly CFSET_ADD="--add"
readonly CFSET_DEL="--del"
readonly CFSET_CLSPARAM="clsparam"

readonly RET_CMD_SUCCESS=0
readonly RET_CMD_FAILED=1

readonly RET_RSC_DEFINED=0
readonly RET_RSC_NG_PARAM=1
readonly RET_RSC_NOT_ROOT=2
readonly RET_RSC_NOT_FOUND=11
readonly RET_RSC_ERROR=255

readonly RET_CLNODE_OFFLINE=0
readonly RET_CLNODE_ONLINE=1
readonly RET_CLNODE_SUSPEND=2

readonly RET_CFSET_NOTEXIST=9

readonly RET_RSC_DEFINED_BOTH=${RET_RSC_DEFINED}
readonly RET_RSC_DEFINED_ONLY_HD=${RET_RSC_NOT_FOUND}
readonly RET_RSC_DEFINED_ONLY_MD=`expr ${RET_RSC_NOT_FOUND} "*" 2`
readonly RET_RSC_DEFINED_NONE=`expr ${RET_RSC_DEFINED_ONLY_MD} + ${RET_RSC_DEFINED_ONLY_HD}`

readonly XML_RC_AUTOSTART="rc/autostart/group/disableinternal"

#------- variables -------#
# opt_nam : pre/post
# opt_skp : skip-copy
# opt_oly : only-shutdown / only-reboot
# opt_not : no-shutdown   / no-reboot
# opt_online : online

opt_cnt_nam=0
opt_cnt_skp=0
opt_cnt_oly=0
opt_cnt_not=0
opt_cnt_online=0
opt_nam="${OPT_NONE}"
opt_skp="${OPT_NONE}"
opt_oly="${OPT_NONE}"
opt_not="${OPT_NONE}"
opt_online="${OPT_NONE}"

log_pre="/dev/null"
log_cur="/dev/null"

msg_cmd_err=""
ret=${RET_CMD_SUCCESS}


# goto MAIN_ROUTINE



####################################################################
#                                                                  #
# Sub Routines                                                     #
#                                                                  #
####################################################################


####################################################################
# name   : PRINT_USAGE
# arg    : (none)
# local  : (none)
# in     : MSG_USAGE
# out    : (none)
# return : (only 0)
# 
# description:
#  Prints usage.
####################################################################
PRINT_USAGE()
{
	echo ""
	echo -e "  ${MSG_USAGE}"
	echo ""
	return 0
}


#####################################################################
# name   : LOG_START
# arg    : (none)
# local  : ret_logcf, logpath, add_skp, add_oly, add_not
# in     : CMD_NAME,  opt_nam, opt_skp, opt_oly, opt_not
# out    : log_cur, log_pre
# return : 
#  0 : Succeeded.
#  1 : Could not find log directory.
# 
# description:
#  Finds log-file-path like as {InstallDir}/"log"
#  and Decides log_cur and log_pre
#  and Writes 1st message to the log_cur file.
# note:
#  - 'local' command changes the value of '$?'.
####################################################################
LOG_START()
{
	local logpath=""
	local ret_logcf=0

	#------- search log dir -------
	logpath=`clplogcf -p`
	ret_logcf=$?
	if [ ${ret_logcf} -ne 0 ]
	then
		logpath="/opt/nec/clusterpro/log/"
	fi
	if [ -d "${logpath}" ]
	then
		log_cur="${logpath}${CMD_NAME}.${opt_nam}.cur"
		log_pre="${logpath}${CMD_NAME}.${opt_nam}.pre"
	else
		echo "${MSG_ERR_LOG_NOT_FOUND}" 1>&2
		return 1
	fi
	
	#------- save previous log file -------
	if [ -e "${log_cur}" ]
	then
		mv -f "${log_cur}" "${log_pre}" > /dev/null 2>&1
	fi

	#------- write 1st log message -------
	local add_oln=" "
	local add_skp=" "
	local add_oly=" "
	local add_not=" "
	if [ "${opt_online}" != "${OPT_NONE}" ]
	then
		add_oln=" --${opt_online}"
	fi
	if [ "${opt_skp}" != "${OPT_NONE}" ]
	then
		add_skp=" --${opt_skp}"
	fi
	if [ "${opt_oly}" != "${OPT_NONE}" ]
	then
		add_oly=" --${opt_oly}"
	fi
	if [ "${opt_not}" != "${OPT_NONE}" ]
	then
		add_not=" --${opt_not}"
	fi
	
	echo "`date '+%Y/%m/%d %H:%M:%S'` ${SCR_NAME} --${opt_nam}${add_oln}${add_skp}${add_oly}${add_not} START." > "${log_cur}"
	
	return 0
}


#####################################################################
# name   : LOG_WRITE
# arg#1  : Message to be recorded.
# local  : msg
# in     : log_cur
# out    : (none)
# return : (only 0)
# 
# description:
#  Records message specified by arg#1. Do not forget "" on calling.
####################################################################
LOG_WRITE()
{
	local msg="$1"
	echo "`date '+%Y/%m/%d %H:%M:%S'` ${msg}" >> "${log_cur}"
	return 0
}


#####################################################################
# name   : LOG_END
# arg#1  : Exit code to be recorded.
# local  : msg
# in     : (SCR_NAME,) log_cur
# out    : (none)
# return : (only 0)
# 
# description:
#  Records end message and exit code.
####################################################################
LOG_END()
{
	local msg="${SCR_NAME} END. exit code=$1."
	echo "`date '+%Y/%m/%d %H:%M:%S'` ${msg}" >> "${log_cur}"
	return 0
}


#####################################################################
# name   : CHK_CFG_VAL
# arg#1  : Resource type. ("md" or "hd")
# arg#2  : Resource name.
# arg#3  : Parameter sub path. (ex: "parameters/fullcopy")
# arg#4  : Expected config value to be set. (ex: "0")
# local  : rsc_typ, rsc_nam, sub_pth, exp_val,
#          cmd_cfget, ret_cfget, cfg_val
# in     : (none)
# out    : (none)
# return : 
#  RET_RSC_DEFINED  : Value of the parameter is correct.
#  RET_RSC_NG_PARAM : Value of the parameter is not the expected one.
#  RET_RSC_ERROR    : Value of the parameter did not be gotten.
#  
# description:
#  Checks if the value of config parameter is expected value or not.
#  
# note:
#  - 'local' command changes the value of '$?'.
#    If you want to get both STDOUT string of a command and its exit
#    code at once, you should not write it in the local command.
#    (ex: 'local ret_str=`clpcfget` ; ret_val=$? ; echo $ret_val'
#     Output indicates exit code of not the clpcfget but the local.)
####################################################################
CHK_CFG_VAL()
{
	local rsc_typ="$1"
	local rsc_nam="$2"
	local sub_pth="$3"
	local exp_val="$4"
	
	local cfg_val=""
	local ret_cfget=0

	local cmd_cfget="clpcfget -g /root/resource/${rsc_typ}@${rsc_nam}/${sub_pth} -p ${rsc_typ}"
	LOG_WRITE "${cmd_cfget}"
	
	#------- get value -------
	cfg_val=`${cmd_cfget} 2>/dev/null`
	ret_cfget=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."
	
	#------- check exit code -------
	if [ ${ret_cfget} -ne ${RET_RSC_DEFINED} ]
	then
		return ${RET_RSC_ERROR}
	fi
	
	#------- check value -------
	LOG_WRITE "val=${cfg_val}"
	LOG_WRITE "req=${exp_val}"
	if [ "${cfg_val}" = "${exp_val}" ]
	then
		return ${RET_RSC_DEFINED}
	fi
	return ${RET_RSC_NG_PARAM}
}


#####################################################################
# name   : CHK_CFG_VAL_1
# arg#1  : Resource type. ("md" or "hd")
# local  : rsc_typ, cmd_cfget, ret_cfget,
#          rsc_lst, rsc_nam, ret_chk_cfg,
#          PAR_COPY, PAR_MKFS, VAL_MUST
# in     : (none)
# out    : (none)
# return : 
#  RET_RSC_NOT_FOUND : Resource does not defined.
#  RET_RSC_DEFINED   : Resource exists and its parameter is correct.
#  RET_RSC_NG_PARAM  : Parameter of resource must be changed.
#  RET_RSC_NOT_ROOT  : User is not root.
#  RET_RSC_ERROR     : Parameter did not be gotten.
# 
# description:
#  Checks if resources of the specified type exist or not,
#  and checks their init-copy/mkfs parameters are set to off.
####################################################################
CHK_CFG_VAL_1()
{
	local rsc_typ="$1"
	
	local rsc_lst=""
	local ret_cfget=0
	
	local cmd_cfget="clpcfget -e /root/resource/${rsc_typ}"
	LOG_WRITE "${cmd_cfget}"

	#------- get resource list and exit code of clpcfget command -------
	rsc_lst=`${cmd_cfget} 2>/dev/null`
	ret_cfget=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."

	#------- check exit code -------
	case ${ret_cfget} in
	${RET_RSC_DEFINED} )
		local rsc_nam=""
		local ret_chk_cfg=0
		local flg_rsc="NOTFOUND"
		local readonly PAR_COPY="parameters/fullcopy"
		local readonly PAR_MKFS="parameters/mkfs"
		local readonly VAL_MUST="0"
		
		#------- check parameters about each resource -------
		for rsc_nam in ${rsc_lst}
		do
			flg_rsc="FOUND"
			CHK_CFG_VAL "${rsc_typ}" "${rsc_nam}" "${PAR_COPY}" "${VAL_MUST}"
			ret_chk_cfg=$?
			if [ ${ret_chk_cfg} -ne ${RET_RSC_DEFINED} ]
			then
				return ${ret_chk_cfg}
			fi
			
			if [ "${rsc_typ}" = "${RSC_MD}" ]
			then
				CHK_CFG_VAL "${rsc_typ}" "${rsc_nam}" "${PAR_MKFS}" "${VAL_MUST}"
				ret_chk_cfg=$?
				if [ ${ret_chk_cfg} -ne ${RET_RSC_DEFINED} ]
				then
					return ${ret_chk_cfg}
				fi
			fi
		done
		
		if [ "${flg_rsc}" = "NOTFOUND" ]
		then
			return ${RET_RSC_ERROR}
		fi
		
		return ${RET_RSC_DEFINED}
		;;
	${RET_RSC_NOT_FOUND} | ${RET_RSC_NOT_ROOT} )
		return ${ret_cfget}
		;;
	esac

	return ${RET_RSC_ERROR}
}


#####################################################################
# name   : CHK_CFG_VAL_2
# arg    : (none)
# local  : status_md, status_hd, ret_chk_cfg
# in     : (none)
# out    : (none)
# return : 
#  RET_RSC_NG_PARAM        : Parameter of resource must be changed.
#  RET_RSC_DEFINED_BOTH    : Both md and hd resources are defined.
#  RET_RSC_DEFINED_ONLY_MD : Only md resource(s) are defined.
#  RET_RSC_DEFINED_ONLY_HD : Only hd resource(s) are defined.
#  RET_RSC_DEFINED_NONE    : Neither of them are defined.
#  RET_RSC_NOT_ROOT        : User is not root.
#  RET_RSC_ERROR           : Parameter did not be gotten.
# 
# description:
#  Checks if user must change init-copy/mkfs parameter of config or not,
#  and gets which (md/hd) command should be called.
####################################################################
CHK_CFG_VAL_2()
{
	local status_md=${RET_RSC_NOT_FOUND}
	local status_hd=${RET_RSC_NOT_FOUND}
	local ret_chk_cfg=0
	
	#------- check md resources -------
	CHK_CFG_VAL_1 ${RSC_MD}
	ret_chk_cfg=$?
	
	#------- check exit code -------
	case ${ret_chk_cfg} in
	${RET_RSC_DEFINED} | ${RET_RSC_NOT_FOUND} )
		status_md=${ret_chk_cfg}
		;;
	${RET_RSC_NG_PARAM} | ${RET_RSC_NOT_ROOT} )
		return ${ret_chk_cfg}
		;;
	* )
		return ${RET_RSC_ERROR}
		;;
	esac

	#------- check hd resources -------
	CHK_CFG_VAL_1 ${RSC_HD}
	ret_chk_cfg=$?
	
	#------- check exit code -------
	case ${ret_chk_cfg} in
	${RET_RSC_DEFINED} | ${RET_RSC_NOT_FOUND} )
		status_hd=${ret_chk_cfg}
		;;
	${RET_RSC_NG_PARAM} | ${RET_RSC_NOT_ROOT} )
		return ${ret_chk_cfg}
		;;
	* )
		return ${RET_RSC_ERROR}
		;;
	esac

	#------- return with result -------
	ret_chk_cfg=`expr ${status_hd} "*" 2 + ${status_md}`
	return ${ret_chk_cfg}
}


#####################################################################
# name   : CHK_RSC_DEF
# arg#1  : Resource type. ("md" or "hd")
# local  : rsc_typ, cmd_cfget, ret_cfget
# in     : (none)
# out    : (none)
# return : 
#  RET_RSC_DEFINED   : Resource of the specified type exists.
#  RET_RSC_NOT_FOUND : Resource of the specified type is not found.
#  RET_RSC_NOT_ROOT  : User is not root.
#  RET_RSC_ERROR     : Parameter did not be gotten.
# 
# description:
#  Checks if resources of the specified type exist or not,
####################################################################
CHK_RSC_DEF()
{
	local rsc_typ="$1"
	local ret_cfget=0
	local cmd_cfget="clpcfget -e /root/resource/${rsc_typ}"
	LOG_WRITE "${cmd_cfget}"

	#------- get only exit code of clpcfget command -------
	${cmd_cfget} > /dev/null 2>&1
	ret_cfget=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."

	#------- check exit code -------
	case ${ret_cfget} in
	${RET_RSC_DEFINED} | ${RET_RSC_NOT_FOUND} | ${RET_RSC_NOT_ROOT} )
		return ${ret_cfget}
		;;
	esac
	return ${RET_RSC_ERROR}
}


#####################################################################
# name   : CHG_FLG
# arg#1  : Resource type. ("md" or "hd")
# arg#2  : Flag.          ("on" or "off")
# local  : rsc, flg, cmd_line, ret_cmd, ret_chk_rsc
# in     : (MSG_*)
# out    : msg_cmd_err
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed. msg_cmd_err includes error message.
# 
# description:
#  Changes backup flag.
####################################################################
CHG_FLG()
{
	local rsc="$1"
	local flg="$2"
	local ret_chk_rsc=0
	local cmd_line=""
	local ret_cmd=0
	msg_cmd_err=""
	
	#------- check if resources exist or not -------
	CHK_RSC_DEF ${rsc}
	ret_chk_rsc=$?
	
	#------- check return code -------
	case ${ret_chk_rsc} in
	${RET_RSC_DEFINED} )
		cmd_line="clp${rsc}ctrl --backup-${flg}"
		LOG_WRITE "${cmd_line}"
		
		#------- change backup flag -------
		${cmd_line}
		ret_cmd=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
		
		if [ ${ret_cmd} -ne 0 ];
		then
			msg_cmd_err=${MSG_CMD_FAILED}
			return      ${RET_CMD_FAILED}
		fi
		return ${RET_CMD_SUCCESS}
		;;
	${RET_RSC_NOT_FOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	${RET_RSC_NOT_ROOT} )
		msg_cmd_err=${MSG_ERR_NOT_ROOT}
		return      ${RET_CMD_FAILED}
		;;
	esac

	msg_cmd_err=${MSG_ERR_INTERNAL}
	return      ${RET_CMD_FAILED}
}


#####################################################################
# name   : CLR_CP
# arg#1  : Resource type. ("md" or "hd")
# local  : rsc, cmd_line, ret_cmd,
#          RET_INIT_SUCCESS, RET_INIT_NOT_FND
# in     : (none)
# out    : (none)
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Clears Cluster Partitions.
#  (Only for Linux version.)
####################################################################
CLR_CP()
{
	local rsc="$1"
	local readonly RET_INIT_SUCCESS=0
	local readonly RET_INIT_NOT_FND=100
	local cmd_line="clp${rsc}init --create force"
	LOG_WRITE "${cmd_line}"
	
	${cmd_line}
	local ret_cmd=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
	
	case ${ret_cmd} in
	${RET_INIT_SUCCESS} | ${RET_INIT_NOT_FND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	
	return ${RET_CMD_FAILED}
}


####################################################################
# name   : SVC_CTRL
# arg    : clpsvcctrl options ("--enable core" or "--disable core")
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : clpsvcctrl.sh executed normally.
#  1      : An error about clpsvcctrl.sh option occurred.
#  Others : An error occurred.
# 
# description:
#  Sets the cluster core service to auto-start or manual-start
####################################################################
SVC_CTRL()
{
	local opt="$@"
	local cmd_line="clpsvcctrl.sh ${opt}"
	LOG_WRITE "${cmd_line}"
	
	#------- execute clpsvcctrl.sh -------
	${cmd_line}
	local ret_cmd=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
	return ${ret_cmd}
}


####################################################################
# name   : SV_SHUTDOWN
# arg    : clpdown option ("" or "-r")
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : Succeeded.
#  Others : Failed.
# 
# description:
#  Power-off or re-boot this server by using clpdown.
####################################################################
SV_SHUTDOWN()
{
	local opt="$1"
	local cmd_line="clpdown ${opt}"
	LOG_WRITE "${cmd_line}"
	
	#------- execute clpdown -------
	${cmd_line}
	local ret_cmd=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
	return ${ret_cmd}
}

####################################################################
# name   : CL_MDAGENT
# arg#1  : clpcl option ("-s" or "-t")
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : Succeeded.
#  Others : Failed.
# 
# description:
#  Starts or stops the Mirror Agent by using clpcl.
#  (Only for Linux version.)
####################################################################
CL_MDAGENT()
{
	local opt="$1"
	local cmd_line="clpcl ${opt} --md"
	LOG_WRITE "${cmd_line}"
	
	#------- execute clpcl -------
	${cmd_line}
	local ret_cmd=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
	return ${ret_cmd}
}


####################################################################
# name   : CL_PM
# arg#1  : clpcl option ("-s" or "-t")
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : Succeeded.
#  Others : Failed.
# 
# description:
#  Starts or stops the Cluster Service (PM) by using clpcl.
####################################################################
CL_PM()
{
	local opt="$1"
	local cmd_line="clpcl ${opt}"
	
	#------- wait if stopping -------
	if [ ${opt} = ${CL_STOP} ]
	then
		cmd_line="clpcl ${opt} -w --apito ${CL_API_TO}"
	fi
	
	LOG_WRITE "${cmd_line}"
	
	#------- execute clpcl -------
	${cmd_line}
	local ret_cmd=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
	return ${ret_cmd}
}

####################################################################
# name   : CL_CLUSTER
# arg#1  : clpcl option ("--resume" or "--suspend")
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : Succeeded.
#  Others : Failed.
# 
# description:
#  Resume or Suspend the cluster by using clpcl.
####################################################################
CL_CLUSTER()
{
	local opt="$1"
	local cmd_line=""
	local ret_chk_stat=0
	local ret_cmd=0

	#------- check status -------
	cmd_line="clpcl ${CL_NODESTAT}"
	LOG_WRITE "${cmd_line}"
	${cmd_line}
	ret_chk_stat=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_chk_stat}."

	if [ "${opt}" = "${CL_RESUME}" ]
	then
		if [ ${ret_chk_stat} -eq ${RET_CLNODE_SUSPEND} ]
		then
			#------- resume -------
			cmd_line="clpcl ${CL_RESUME}"
			LOG_WRITE "${cmd_line}"
			${cmd_line}
			ret_cmd=$?
			LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
			if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
			then
				# Only this server resumes 
				#      in case other servers are down and unable to resume.
				cmd_line="clpcl ${CL_RESUME} --local"
				LOG_WRITE "${cmd_line}"
				${cmd_line}
				ret_cmd=$?
				LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
			fi
		fi
	elif [ "${opt}" = "${CL_SUSPEND}" ]
	then
		if [ ${ret_chk_stat} -eq ${RET_CLNODE_ONLINE} ]
		then
			#------- suspend -------
			cmd_line="clpcl ${CL_SUSPEND} ${CL_FORCE}"
			LOG_WRITE "${cmd_line}"
			${cmd_line}
			ret_cmd=$?
			LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."
		fi
	else
		echo "${MSG_ERR_INTERNAL}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	return ${ret_cmd}
}

####################################################################
# name   : SET_GROUP_AUTOSTART
# arg#1  : --enable or --disable
# local  : opt, cmd_line, ret_cmd
# in     : (none)
# out    : (none)
# return : 
#  0      : Succeeded.
#  Others : Failed.
# 
# description:
#  Enable or disable autostart of all groups by using clpcfset.
####################################################################
SET_GROUP_AUTOSTART()
{
	local opt="$1"
	local ret_cmd=0
	local err_cmd=""
	local inspath="/opt/nec/clusterpro"
	if [ "${opt}" = "${GRP_AUTO_OFF}" ]
	then
		local cmd_line="clpcfset ${CFSET_INTERNAL} ${CFSET_ADD} ${CFSET_CLSPARAM} ${XML_RC_AUTOSTART} 1"
	elif [ "${opt}" = "${GRP_AUTO_ON}" ]
	then
		local cmd_line="clpcfset ${CFSET_INTERNAL} ${CFSET_DEL} ${CFSET_CLSPARAM} ${XML_RC_AUTOSTART}"
	else
		return ${RET_CMD_FAILED}
	fi

	#------- change directory -------
	pushd ${inspath}/etc > /dev/null

	#------- set autostart of all groups -------
	LOG_WRITE "${cmd_line}"
	err_cmd=`${cmd_line} 2>&1`
	ret_cmd=$?
	LOG_WRITE "  ${err_cmd}"
	LOG_WRITE "${MSG_RET_VAL} ${ret_cmd}."

	#------- return directory -------
	popd > /dev/null
	return ${ret_cmd}
}

#####################################################################
# name   : CHG_FLG_AND_FSFREEZE
# arg#1  : Resource type. ("md" or "hd")
# arg#2  : Flag.          ("on" or "off")
# in     : (MSG_*)
# out    : msg_cmd_err
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed. msg_cmd_err includes error message.
# 
# description:
#  Changes backup flag and execute fsfreeze.
#####################################################################
CHG_FLG_AND_FSFREEZE()
{
	local rsc_typ="$1"
	local flg_backup="$2"

	local rsc_lst=""
	local rsc_nam=""
	local svr_nam=`echo $HOSTNAME | cut -d'.' -f1`
	local cmd_cfget=""
	local ret_cfget=0
	local cmd_mdctrl=""
	local ret_mdctrl=0
	local ret_cut=0
	local ret_sed=0
	local mnt_pnt=""
	local mnt_chk=""
	local flg_mnt=""
	local err_fsfreeze=""
	local ret_fsfreeze=0
	local flg_fsfreeze=""
	local opt_fsfreeze=""

	#------- decide fsfreeze option -------
	if   [ "${flg_backup}" = "${FLG_ON}" ]
	then
		opt_fsfreeze=${OPT_FREEZE}
	elif [ "${flg_backup}" = "${FLG_OFF}" ]
	then
		opt_fsfreeze=${OPT_UNFREEZE}
	else
		msg_cmd_err=${MSG_ERR_INTERNAL}
		return      ${RET_CMD_FAILED}
	fi

	#------- get resource list and exit code of clpcfget command -------
	cmd_cfget="clpcfget -e /root/resource/${rsc_typ}"
	LOG_WRITE "${cmd_cfget}"
	rsc_lst=`${cmd_cfget} 2>/dev/null`
	ret_cfget=$?
	LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."

	#------- check exit code -------
	case ${ret_cfget} in
	${RET_RSC_DEFINED} )
		break
		;;
	${RET_RSC_NOT_FOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	${RET_RSC_NOT_ROOT} )
		msg_cmd_err=${MSG_ERR_NOT_ROOT}
		return      ${RET_CMD_FAILED}
		;;
	* )
		msg_cmd_err=${MSG_ERR_INTERNAL}
		return      ${RET_CMD_FAILED}
		;;
	esac

	#------- backup flag OFF if unfreezing -------
	if [ "${flg_backup}" = "${FLG_OFF}" ]
	then
		cmd_mdctrl="clp${rsc_typ}ctrl --backup-off"
		LOG_WRITE "${cmd_mdctrl}"
		${cmd_mdctrl}
		ret_mdctrl=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_mdctrl}."
		
		if [ ${ret_mdctrl} -ne 0 ];
		then
			msg_cmd_err=${MSG_CMD_FAILED}
			return      ${RET_CMD_FAILED}
		fi
	fi

	#------- about each resource -------
	for rsc_nam in ${rsc_lst}
	do
		LOG_WRITE "${rsc_nam}"

		######## get mount point ########
		#------- get individual server setting if exist -------
		cmd_cfget="clpcfget -g /root/resource/${rsc_typ}@${rsc_nam}/server@${svr_nam}/parameters/mount/point -p ${rsc_typ}"
		LOG_WRITE "${cmd_cfget}"
		mnt_pnt=`${cmd_cfget} 2>/dev/null`
		ret_cfget=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."
		
		#------- check exit code -------
		if [ ${ret_cfget} -ne ${RET_RSC_DEFINED} ]
		then
			#------- get resource setting -------
			cmd_cfget="clpcfget -g /root/resource/${rsc_typ}@${rsc_nam}/parameters/mount/point -p ${rsc_typ}"
			LOG_WRITE "${cmd_cfget}"
			mnt_pnt=`${cmd_cfget} 2>/dev/null`
			ret_cfget=$?
			LOG_WRITE "${MSG_RET_VAL} ${ret_cfget}."

			#------- check exit code -------
			if [ ${ret_cfget} -ne ${RET_RSC_DEFINED} ]
			then
				msg_cmd_err=${MSG_ERR_INTERNAL}
				return      ${RET_CMD_FAILED}
			fi
		fi
		LOG_WRITE "  mount point : ${mnt_pnt}"
		
		######## check mount point is mounted ########
		# Note : A command line that includes quotations and space character #
		#        MUST execute directly!  Do NOT set it into variable!        #
		# Note : A space character included in a mount point is printed with #
		#        octal number code in /proc/mounts.                          #

		#------- mount point to search -------#
		LOG_WRITE "echo \"${mnt_pnt}\" | /bin/sed 's/ /\\\\040/g'"
		mnt_chk=`echo "${mnt_pnt}" | /bin/sed 's/ /\\\\040/g' 2>/dev/null`
		ret_sed=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_sed}."
		LOG_WRITE "  mount check : ${mnt_chk}"

		#------- check mount point is mounted -------#
		LOG_WRITE "/bin/cut -d' ' -f 2 /proc/mounts"
		mount_points=`/bin/cut -d' ' -f 2 /proc/mounts 2>/dev/null`
		ret_cut=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_cut}."

		flg_mnt="NotMounted"
		for mount_point in ${mount_points}
		do
			LOG_WRITE "    picked : ${mount_point}"
			if [ "${mount_point}" = "${mnt_chk}" ]
			then
				flg_mnt="Mounted"
				LOG_WRITE "  ${mnt_pnt} is mounted."
				break
			fi
		done
		if [ "${flg_mnt}" = "NotMounted" ]
		then
			LOG_WRITE "  ${mnt_pnt} is NOT mounted."
			continue
		fi

		######## execute fsfreeze ########
		flg_fsfreeze="ERROR"
		LOG_WRITE "LC_ALL=C /sbin/fsfreeze ${opt_fsfreeze} \"${mnt_pnt}\" 2>&1"
		err_fsfreeze=`LC_ALL=C /sbin/fsfreeze ${opt_fsfreeze} "${mnt_pnt}" 2>&1`
		ret_fsfreeze=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_fsfreeze}."
		
		if [ ${ret_fsfreeze} -ne 0 ]
		then
			#------- check English (LANG=C) error message -------
			LOG_WRITE "  ${err_fsfreeze}"
			case "${err_fsfreeze}" in
			*"Device or resource busy"*)
				# EBUSY 16 #
				if [ "${opt_fsfreeze}" = "${OPT_FREEZE}" ]
				then
					flg_fsfreeze="FROZEN"
					echo      "  ${rsc_nam} ${MSG_RSC_FROZEN_ALREADY}"
					LOG_WRITE "  ${rsc_nam} ${MSG_RSC_FROZEN_ALREADY}"
				fi
				;;
			*"Invalid argument"*)
				# EINVAL 22 #
				if [ "${opt_fsfreeze}" != "${OPT_FREEZE}" ]
				then
					flg_fsfreeze="UNFROZEN"
					echo      "  ${rsc_nam} ${MSG_RSC_UNFROZEN_ALREADY}"
					LOG_WRITE "  ${rsc_nam} ${MSG_RSC_UNFROZEN_ALREADY}"
				fi
				;;
			esac
		else
			#------- fsfreeze is executed successfully -------
			if   [ "${opt_fsfreeze}" = "${OPT_FREEZE}" ]
			then
				flg_fsfreeze="FROZEN"
				echo      "  ${rsc_nam} ${MSG_RSC_FROZEN}"
				LOG_WRITE "  ${rsc_nam} ${MSG_RSC_FROZEN}"
			elif [ "${opt_fsfreeze}" = "${OPT_UNFREEZE}" ]
			then
				flg_fsfreeze="UNFROZEN"
				echo      "  ${rsc_nam} ${MSG_RSC_UNFROZEN}"
				LOG_WRITE "  ${rsc_nam} ${MSG_RSC_UNFROZEN}"
			else
				msg_cmd_err=${MSG_ERR_INTERNAL}
				return      ${RET_CMD_FAILED}
			fi
		fi
		
		if [ "${flg_fsfreeze}" = "ERROR" ]
		then
			#------- fsfreeze failed -------
			echo "${err_fsfreeze}" 1>&2
			msg_cmd_err=${MSG_CMD_FAILED}
			return      ${RET_CMD_FAILED}
		fi
	done
	
	#------- backup flag ON if freezing -------
	if [ "${flg_backup}" = "${FLG_ON}" ]
	then
		cmd_mdctrl="clp${rsc_typ}ctrl --backup-on"
		LOG_WRITE "${cmd_mdctrl}"
		${cmd_mdctrl}
		ret_mdctrl=$?
		LOG_WRITE "${MSG_RET_VAL} ${ret_mdctrl}."
		
		if [ ${ret_mdctrl} -ne 0 ];
		then
			msg_cmd_err=${MSG_CMD_FAILED}
			return      ${RET_CMD_FAILED}
		fi
	fi
	
	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : EXECUTE_PRE_BKUP
# arg    : (none)
# local  : ret_chg_flg
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Executes the process for pre-backup.
####################################################################
EXECUTE_PRE_BKUP()
{
	local ret_chg_flg=0
	
	#------- set backup flags to ON -------
	echo "${MSG_ACT_FLG_ON}"
	
	CHG_FLG ${RSC_MD} ${FLG_ON}
	ret_chg_flg=$?
	if [ ${ret_chg_flg} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${ret_chg_flg}
	fi
	
	CHG_FLG ${RSC_HD} ${FLG_ON}
	ret_chg_flg=$?
	if [ ${ret_chg_flg} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${ret_chg_flg}
	fi

	#------- set service manual-start and shutdown server -------
	echo "${MSG_ACT_SVC_OFF}"
	SVC_CTRL "${SVC_CORE_OFF}"

	if [ "${opt_not}" = "${OPT_NODN}" ]
	then
		echo "${MSG_ACT_STOP_CLPM}"
		CL_PM "${CL_STOP}"
		
		echo "${MSG_ACT_STOP_AGENT}"
		CL_MDAGENT "${CL_STOP}"
		
		LOG_WRITE "sync"
		sync
	else
		echo "${MSG_ACT_SHUTDOWN}"
		SV_SHUTDOWN "${DWN_POWER_OFF}"
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}


####################################################################
# name   : EXECUTE_POST_BKUP
# arg    : (none)
# local  : ret_chg_flg, ret_mdagent
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Executes the process for post-backup.
####################################################################
EXECUTE_POST_BKUP()
{
	local ret_chg_flg=0
	local ret_mdagent=0
	
	#------- set backup flags to OFF -------
	echo "${MSG_ACT_START_AGENT}"
	
	CL_MDAGENT "${CL_START}"
	ret_mdagent=$?
	if [ ${ret_mdagent} -ne 0 ]
	then
		echo  "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	echo "${MSG_ACT_FLG_OFF}"
	
	CHG_FLG ${RSC_MD} ${FLG_OFF}
	ret_chg_flg=$?
	if [ ${ret_chg_flg} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${ret_chg_flg}
	fi
	
	CHG_FLG ${RSC_HD} ${FLG_OFF}
	ret_chg_flg=$?
	if [ ${ret_chg_flg} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${ret_chg_flg}
	fi

	#------- set service autorun and reboot server -------
	echo "${MSG_ACT_SVC_ON}"
	SVC_CTRL "${SVC_CORE_ON}"

	if [ "${opt_not}" = "${OPT_NOBT}" ]
	then
		echo "${MSG_ACT_START_CLPM}"
		CL_PM "${CL_START}"
	else
		echo "${MSG_ACT_STOP_AGENT}"
		CL_MDAGENT "${CL_STOP}"
		
		echo "${MSG_ACT_REBOOT}"
		SV_SHUTDOWN "${DWN_REBOOT}"
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}


####################################################################
# name   : EXECUTE_PRE_RSTR
# name   : EXECUTE_ONLY_SHUTDOWN
# arg    : (none)
# local  : (none)
# in     : (none)
# out    : (none)
# return : RET_CMD_SUCCESS
# 
# description:
#  Executes the process for pre-restore.
#  And for the only-shutdown option.
####################################################################
EXECUTE_PRE_RSTR()
{
	#------- set service manual-start and shutdown server -------
	echo "${MSG_ACT_SVC_OFF}"
	SVC_CTRL "${SVC_CORE_OFF}"

	if [ "${opt_not}" = "${OPT_NODN}" ]
	then
		echo "${MSG_ACT_STOP_CLPM}"
		CL_PM "${CL_STOP}"
		
		echo "${MSG_ACT_STOP_AGENT}"
		CL_MDAGENT "${CL_STOP}"
	else
		echo "${MSG_ACT_SHUTDOWN}"
		SV_SHUTDOWN "${DWN_POWER_OFF}"
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}

EXECUTE_ONLY_SHUTDOWN()
{
	EXECUTE_PRE_RSTR
	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : EXECUTE_PRE_BKUP_ONLINE
# arg#1  : (none)
# local  : ret_cmd
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Executes the process for pre-online.
####################################################################
EXECUTE_PRE_BKUP_ONLINE()
{
	local ret_cmd=0

	#------- disable autostart of all groups -------
	SET_GROUP_AUTOSTART ${GRP_AUTO_OFF}
	ret_cmd=$?
	if [ ${ret_cmd} -eq  ${RET_RSC_NOT_ROOT} ]
	then
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		return ${RET_CMD_FAILED}
	elif [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- set service manual-start -------
	echo "${MSG_ACT_SVC_OFF}"
	SVC_CTRL "${SVC_CORE_OFF}"
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- flash the above settings to the system disk -------
	LOG_WRITE "sync"
	sync

	#------- suspend the cluster -------
	echo "${MSG_ACT_SUSPEND}"
	CL_CLUSTER "${CL_SUSPEND}"
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		SVC_CTRL "${SVC_CORE_ON}"
		SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- set fsfreeze -------
	echo "${MSG_ACT_FLG_ON}"
	#------- set fsfreeze on (md) -------
	CHG_FLG_AND_FSFREEZE ${RSC_MD} ${FLG_ON}
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		CHG_FLG_AND_FSFREEZE ${RSC_MD} ${FLG_OFF}
		CL_CLUSTER "${CL_RESUME}"
		SVC_CTRL "${SVC_CORE_ON}"
		SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
		echo  "${msg_cmd_err}" 1>&2
		return ${RET_CMD_FAILED}
	fi
	
	#------- set fsfreeze on (hd) -------
	CHG_FLG_AND_FSFREEZE ${RSC_HD} ${FLG_ON}
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		CHG_FLG_AND_FSFREEZE ${RSC_HD} ${FLG_OFF}
		CHG_FLG_AND_FSFREEZE ${RSC_MD} ${FLG_OFF}
		CL_CLUSTER "${CL_RESUME}"
		SVC_CTRL "${SVC_CORE_ON}"
		SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
		echo  "${msg_cmd_err}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}


####################################################################
# name   : EXECUTE_POST_RSTR
# name   : EXECUTE_ONLY_REBOOT
# arg    : (none)
# local  : (none)
# in     : (none)
# out    : (none)
# return : RET_CMD_SUCCESS
#
# description:
#  Executes the process for post-restore.
#  And for the only-reboot option.
####################################################################
EXECUTE_POST_RSTR()
{
	#------- set service autorun and reboot server -------
	echo "${MSG_ACT_SVC_ON}"
	SVC_CTRL "${SVC_CORE_ON}"

	if [ "${opt_not}" = "${OPT_NOBT}" ]
	then
		echo "${MSG_ACT_START_AGENT}"
		CL_MDAGENT "${CL_START}"
		
		echo "${MSG_ACT_START_CLPM}"
		CL_PM "${CL_START}"
	else
		echo "${MSG_ACT_REBOOT}"
		SV_SHUTDOWN "${DWN_REBOOT}"
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}

EXECUTE_ONLY_REBOOT()
{
	EXECUTE_POST_RSTR
	return ${RET_CMD_SUCCESS}
}


####################################################################
# name   : EXECUTE_POST_RSTR_WITHOUT_COPY
# arg    : (none)
# local  : ret_chk_cfg, ret_clr_cp
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Executes the process for post-same-image-restore.
####################################################################
EXECUTE_POST_RSTR_WITHOUT_COPY()
{
	local ret_chk_cfg=0
	local ret_clr_cp=0
	
#	echo "${MSG_ACT_FLG_OFF}"
	
	#------- check if config is correct or not -------
	CHK_CFG_VAL_2
	ret_chk_cfg=$?
	LOG_WRITE "${MSG_RET_CHK} ${ret_chk_cfg}."

	case ${ret_chk_cfg} in
	${RET_RSC_DEFINED_BOTH} | ${RET_RSC_DEFINED_ONLY_MD} | ${RET_RSC_DEFINED_ONLY_HD} | ${RET_RSC_DEFINED_NONE} )
		;;
	${RET_RSC_NG_PARAM} )
		echo  "${MSG_ERR_BAD_CONFIG}" 1>&2
		return ${RET_CMD_FAILED}
		;;
	${RET_RSC_NOT_ROOT} )
		echo  "${MSG_ERR_NOT_ROOT}" 1>&2
		return ${RET_CMD_FAILED}
		;;
	* )
		echo  "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
		;;
	esac
	
	#------- clear CP of md resources -------
	case ${ret_chk_cfg} in
	${RET_RSC_DEFINED_BOTH} | ${RET_RSC_DEFINED_ONLY_MD} )
		CLR_CP ${RSC_MD}
		ret_clr_cp=$?
		if [ ${ret_clr_cp} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${MSG_CMD_FAILED}" 1>&2
			return ${RET_CMD_FAILED}
		fi
		;;
	esac
	
	#------- clear CP of hd resources -------
	case ${ret_chk_cfg} in
	${RET_RSC_DEFINED_BOTH} | ${RET_RSC_DEFINED_ONLY_HD} )
		CLR_CP ${RSC_HD}
		ret_clr_cp=$?
		if [ ${ret_clr_cp} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${MSG_CMD_FAILED}" 1>&2
			return ${RET_CMD_FAILED}
		fi
		;;
	esac

	#------- set service autorun and reboot server -------
	echo "${MSG_ACT_SVC_ON}"
	SVC_CTRL "${SVC_CORE_ON}"

	if [ "${opt_not}" = "${OPT_NOBT}" ]
	then
		echo "${MSG_ACT_START_AGENT}"
		CL_MDAGENT "${CL_START}"
		
		echo "${MSG_ACT_START_CLPM}"
		CL_PM "${CL_START}"
	else
		echo "${MSG_ACT_REBOOT}"
		SV_SHUTDOWN "${DWN_REBOOT}"
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : EXECUTE_POST_BKUP_ONLINE
# arg    : (none)
# local  : ret_cmd
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
#
# description:
#  Executes the process for post-restore-online.
####################################################################
EXECUTE_POST_BKUP_ONLINE()
{
	local ret_cmd=0

	#------- enable autostart of all groups -------
	SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
	ret_cmd=$?
	if [ ${ret_cmd} -eq ${RET_RSC_NOT_ROOT} ]
	then
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		return ${RET_CMD_FAILED}
	elif [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ] && [ ${ret_cmd} -ne ${RET_CFSET_NOTEXIST} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- set service autorun -------
	echo "${MSG_ACT_SVC_ON}"
	SVC_CTRL "${SVC_CORE_ON}"
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- fsfreeze off -------
	echo "${MSG_ACT_FLG_OFF}"
	#------- set fsfreeze off (md) -------
	CHG_FLG_AND_FSFREEZE ${RSC_MD} ${FLG_OFF}
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${RET_CMD_FAILED}
	fi
	
	#------- set fsfreeze off (hd) -------
	CHG_FLG_AND_FSFREEZE ${RSC_HD} ${FLG_OFF}
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${msg_cmd_err}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- resume the cluster -------
	echo "${MSG_ACT_RESUME}"
	CL_CLUSTER "${CL_RESUME}"
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi
	
	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : EXECUTE_POST_RSTR_ONLINE
# arg    : clpmdctrl option ("skip-copy" or "")
# local  : ret_cmd, opt
# in     : (none)
# out    : (none)
# return : Value for exit code of this script.
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
#
# description:
#  Executes the process for post-restore-online.
####################################################################
EXECUTE_POST_RSTR_ONLINE()
{
	local ret_cmd=0
	local opt="$1"

	#------- enable autostart of all groups -------
	SET_GROUP_AUTOSTART ${GRP_AUTO_ON}
	ret_cmd=$?
	if [ ${ret_cmd} -eq ${RET_RSC_NOT_ROOT} ]
	then
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		return ${RET_CMD_FAILED}
	elif [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ] && [ ${ret_cmd} -ne ${RET_CFSET_NOTEXIST} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- set service autorun -------
	echo "${MSG_ACT_SVC_ON}"
	SVC_CTRL "${SVC_CORE_ON}"
	ret_cmd=$?
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "${MSG_CMD_FAILED}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- set backup flags -------
	if [ "${opt}" = "${OPT_NONE}" ]
	then
		#------- on -------
		echo "${MSG_ACT_FLG_ON}"

		CHG_FLG ${RSC_MD} ${FLG_ON}
		ret_cmd=$?
		if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${msg_cmd_err}" 1>&2
			return ${RET_CMD_FAILED}
		fi

		CHG_FLG ${RSC_HD} ${FLG_ON}
		ret_cmd=$?
		if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${msg_cmd_err}" 1>&2
			return ${RET_CMD_FAILED}
		fi
	elif [ "${opt}" = "${OPT_SKIP}" ]
	then
		#------- off -------
		echo "${MSG_ACT_FLG_OFF}"

		CHG_FLG ${RSC_MD} ${FLG_OFF}
		ret_cmd=$?
		if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${msg_cmd_err}" 1>&2
			return ${RET_CMD_FAILED}
		fi

		CHG_FLG ${RSC_HD} ${FLG_OFF}
		ret_cmd=$?
		if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
		then
			echo  "${msg_cmd_err}" 1>&2
			return ${RET_CMD_FAILED}
		fi
	fi

	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : CHECK_OPTIONS
# arg    : (none)
# local  : (none)
# in     : opt_cnt_*, opt_*, CMD_NAME
# out    : (none)
# return : Validation of options.
#  RET_CMD_SUCCESS : OK.
#  RET_CMD_FAILED  : NG.
# 
# description:
#  Checks options.
####################################################################
CHECK_OPTIONS()
{

	if [ ${opt_cnt_nam} -ne 1 ]
	then
		return ${RET_CMD_FAILED}
	fi
	if [ ${opt_cnt_skp} -gt 1 ]
	then
		return ${RET_CMD_FAILED}
	fi
	if [ ${opt_cnt_oly} -gt 1 ]
	then
		return ${RET_CMD_FAILED}
	fi
	if [ ${opt_cnt_not} -gt 1 ]
	then
		return ${RET_CMD_FAILED}
	fi
	if [ ${opt_cnt_online} -gt 1 ]
	then
		return ${RET_CMD_FAILED}
	fi

	if [ "${opt_online}" = "${OPT_ONLINE}" ]
	then
		if [ "${opt_nam}" = "${OPT_PRE}" ]
		then
			if [ "${CMD_NAME}" = "${CMD_RSTR}" ]
			then
				return ${RET_CMD_FAILED}
			fi
		fi
		if [ "${opt_oly}" != "${OPT_NONE}" ]
		then
			return ${RET_CMD_FAILED}
		fi
		if [ "${opt_not}" != "${OPT_NONE}" ]
		then
			return ${RET_CMD_FAILED}
		fi
	fi
	if [ "${opt_skp}" = "${OPT_SKIP}" ]
	then
		if [ "${CMD_NAME}" != "${CMD_RSTR}" ]
		then
			return ${RET_CMD_FAILED}
		fi
		if [ "${opt_nam}" != "${OPT_POST}" ]
		then
			return ${RET_CMD_FAILED}
		fi
		if [ "${opt_oly}" != "${OPT_NONE}" ]
		then
			return ${RET_CMD_FAILED}
		fi
	fi
	if [ "${opt_nam}" = "${OPT_PRE}" ]
	then
		if [ "${opt_oly}" = "${OPT_BOOT}" ]
		then
			return ${RET_CMD_FAILED}
		fi
		if [ "${opt_not}" = "${OPT_NOBT}" ]
		then
			return ${RET_CMD_FAILED}
		fi
	fi
	if [ "${opt_nam}" = "${OPT_POST}" ]
	then
		if [ "${opt_oly}" = "${OPT_DOWN}" ]
		then
			return ${RET_CMD_FAILED}
		fi
		if [ "${opt_not}" = "${OPT_NODN}" ]
		then
			return ${RET_CMD_FAILED}
		fi
	fi

	return ${RET_CMD_SUCCESS}
}


####################################################################
#                                                                  #
# Main Routine                                                     #
#                                                                  #
####################################################################
#:MAIN_ROUTINE

#------- option check 1 -------#

opt_cnt_nam=0
opt_cnt_skp=0
opt_cnt_oly=0
opt_cnt_not=0
opt_cnt_online=0

for i in $@
do
	case "$i" in
	"--${OPT_PRE}" )
		opt_nam="${OPT_PRE}"
		opt_cnt_nam=`expr ${opt_cnt_nam} + 1`
		;;
	"--${OPT_POST}" )
		opt_nam="${OPT_POST}"
		opt_cnt_nam=`expr ${opt_cnt_nam} + 1`
		;;
	"--${OPT_SKIP}" )
		opt_skp="${OPT_SKIP}"
		opt_cnt_skp=`expr ${opt_cnt_skp} + 1`
		;;
	"--${OPT_DOWN}" )
		opt_oly="${OPT_DOWN}"
		opt_cnt_oly=`expr ${opt_cnt_oly} + 1`
		;;
	"--${OPT_BOOT}" )
		opt_oly="${OPT_BOOT}"
		opt_cnt_oly=`expr ${opt_cnt_oly} + 1`
		;;
	"--${OPT_NODN}" )
		opt_not="${OPT_NODN}"
		opt_cnt_not=`expr ${opt_cnt_not} + 1`
		;;
	"--${OPT_NOBT}" )
		opt_not="${OPT_NOBT}"
		opt_cnt_not=`expr ${opt_cnt_not} + 1`
		;;
	"--${OPT_ONLINE}" )
		opt_online="${OPT_ONLINE}"
		opt_cnt_online=`expr ${opt_cnt_online} + 1`
		;;
	"--${OPT_HELP}" )
		PRINT_USAGE
		exit ${RET_CMD_SUCCESS}
		;;
	* )
		echo "${MSG_ERR_INVALID_OPTION}" 1>&2
		PRINT_USAGE
		exit ${RET_CMD_FAILED}
		;;
	esac
done

#------- option check 2 -------#

CHECK_OPTIONS
ret_opt=$?
if [ ${ret_opt} -ne ${RET_CMD_SUCCESS} ]
then
	echo "${MSG_ERR_INVALID_OPTION}" 1>&2
	PRINT_USAGE
	exit ${RET_CMD_FAILED}
fi

#------- execute -------#

if [ "${opt_nam}" = "${OPT_PRE}" ]
then
	if [ "${opt_online}" != "${OPT_NONE}" ]
	then
		#------- online -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_PRE_BKUP_ONLINE
		ret=$?
	elif [ "${opt_oly}" != "${OPT_NONE}" ]
	then
		#------- only-shutdown -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_ONLY_SHUTDOWN
		ret=$?
	else
		#------- normal -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_PRE_BKUP
		ret=$?
	fi
elif [ "${opt_nam}" = "${OPT_POST}" ]
then
	if [ "${opt_online}" != "${OPT_NONE}" ]
	then
		#------- online -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_POST_BKUP_ONLINE ${opt_skp}
		ret=$?
	elif [ "${opt_skp}" != "${OPT_NONE}" ]
	then
		#------- skip-copy -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_POST_RSTR_WITHOUT_COPY
		ret=$?
	elif [ "${opt_oly}" != "${OPT_NONE}" ]
	then
		#------- only-reboot -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_ONLY_REBOOT
		ret=$?
	else
		#------- normal -------#
		LOG_START || exit ${RET_CMD_FAILED}
		EXECUTE_POST_BKUP
		ret=$?
	fi
else
	echo "${MSG_ERR_INTERNAL}" 1>&2
	PRINT_USAGE
	exit ${RET_CMD_FAILED}
fi

# ret=$?
LOG_END ${ret}
exit    ${ret}
