#!/bin/bash
# shellcheck disable=SC1091,SC2030,SC2031
# ==========================================================
# clpocdns-start.sh
# ------------------
# [Oracle Cloud DNS活性スクリプト]
# Oracle Cloud DNSゾーンにAレコードを作成します
# ==========================================================

#
# デバッグ
#
DebugMode=false


#
# モジュール読み込み
#
ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"


#
# 環境変数
#
export OCI_CLI_SUPPRESS_FILE_PERMISSIONS_WARNING="True"


#
# スクリプト終了コード
#
Success=0                           # [正常](0)
ErrorOciFailed=50                   # [異常] OCI CLI失敗(50)
ErrorOciTimeout=51                  # [異常] OCI CLIタイムアウト(51)
ErrorNotExistOCICmd=52              # [異常] OCI CLI未インストール(52)
ErrorInternalError=79               # [異常] 内部エラー(79)


#
# アラート出力文字列
#
AlertOciFailed="The OCI CLI command failed. (%1)"
AlertOciTimeout="Timeout occurred."
AlertNotExistOCICmd="The OCI CLI is not found."
AlertInternalError="Internal error occurred."


#
# OCI CLIエラー原因不明
#
ErrorCauseUnknown="Internal error"


#
# 上位で設定する環境変数: CLP_OCF_PARAM の数
#
ClpOcfParamCnt=9


#
# 外部コマンド正常終了
#
ExCmdSuccess=0


#
# アラートログ出力用の共通メモリ最大領域
#
ShmMaxSize=$((128 - 1))


#
# リソースタイプ
#
RscType="ocdns"


#
# マルチリージョンでのリソースレコードの操作範囲
#
# OperationOnlyActiveServerRegion=0       # 現用系サーバが所属するリージョンのみ
OperationAllServerRegion=1              # クラスタサーバが所属する全てのリージョン


#
# タイムアウト発生時の終了コード(128 + SIGNAL)
#
ExitCodeWithTimeout=$((128 + 9))


# ----------------------------------------------------------
#
# 関数定義
#
# ----------------------------------------------------------

#
# プロキシ情報の設定
#
function SetProxy {
    unset HTTPS_PROXY
    useProxyHTTP=1
    proxyXmlPath="/root/server@${actServerName}/proxy"
    proxyScheme=$(GetClpConfValue "-g" "${proxyXmlPath}/scheme") || return
    WriteStdOut "proxyScheme: $proxyScheme"
    if [[ $proxyScheme -eq $useProxyHTTP ]]; then
        proxyServer=$(GetClpConfValue "-g" "${proxyXmlPath}/server") || return
        proxyPort=$(GetClpConfValue "-g" "${proxyXmlPath}/port") || return
        proxy="http://${proxyServer}:${proxyPort}"
        export HTTPS_PROXY=$proxy
        WriteStdOut "proxy: $HTTPS_PROXY"
    fi
}


#
# アラートログ出力文字列を共有メモリへ設定する
#
function SetAlertMessage {
    alertMessage=$1
    shmRmCmdLine="clpshmrmset --descript --rsc -g '$grpName' -t $RscType -n '$rscName' -m '$alertMessage'"
    result=$(eval "$shmRmCmdLine" 2>&1)
    shmRmSetExitCode=$?
    if [[ $shmRmSetExitCode -ne $ExCmdSuccess ]]; then
        WriteStdOut "[CommandLine] $shmRmCmdLine"
        WriteStdErr "The 'clpshmrmset' command failed. ($shmRmSetExitCode)"
        WriteStdErr "$result"
        # 処理継続
    fi
}


#
# クラスタ構成情報 (clp.conf) から値を取得
#
function GetClpConfValue {
    optType=$1
    xmlPath=$2
    server=$3
    cfGetCmdLine="clpcfget $optType $xmlPath"
    if [[ -n $server ]]; then
        cfGetCmdLine+=" -s $server"
    fi
    result=$(eval "$cfGetCmdLine" 2>&1)
    cfGetExitCode=$?
    if [[ $cfGetExitCode -ne $ExCmdSuccess ]]; then
        WriteStdErr "The 'clpcfget' command failed. ($cfGetExitCode: [CommandLine] $cfGetCmdLine)"
        WriteStdErr "$result"
        return 1
    fi
    echo "$result"
}


#
# 標準出力に文字列を出力する
#
function WriteStdOut {
    local message=$1
    if ! $DebugMode; then
        # printf "%04d: %s\n" "${BASH_LINENO[0]}" "$message"
        echo "$message"
    else
        # printf "[STDOUT] %04d: %s\n" "${BASH_LINENO[0]}" "$message"
        echo "[STDOUT] $message"
    fi
}


#
# 標準エラー出力に文字列を出力する
#
function WriteStdErr {
    local message=$1
    if ! $DebugMode; then
        # printf "%04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
        echo "$message" >&2
    else
        # printf "[STDERR] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
        echo "[STDERR] $message" >&2
    fi
}


#
# コンソールに文字列を出力する(デバッグ用)
#
function WriteDebug {
    local message=$1
    if ! $DebugMode; then
        :
    else
        # printf "[ DEBUG] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >/dev/tty
        echo "[ DEBUG] $message" >/dev/tty
    fi
}


#
# 0以上の整数か判定する
#
function IsNonNegativeNumber {
    number=$1
    if [[ $number =~ ^[0-9]+$ && $number -ge 0 ]]; then
        return 0
    else
        return 1
    fi
}


# ----------------------------------------------------------
#
# 処理開始
#
# ----------------------------------------------------------
WriteStdOut "Start the activation process for the Oracle Cloud DNS resource."

#
# 上位モジュールで設定した環境変数: CLP_* の確認
#
for i in "CLP_GROUPNAME" "CLP_RESOURCENAME"; do
    if [[ -z ${!i} ]]; then
        WriteStdErr "The environment variable '$i' has not been set."
        SetAlertMessage "$AlertInternalError"
        exit $ErrorInternalError
    fi
done
actServerName=$(uname -n | cut -d '.' -f 1)
grpName=$CLP_GROUPNAME
rscName=$CLP_RESOURCENAME
WriteDebug "actServerName: $actServerName"
WriteStdOut "grpName: $grpName"
WriteStdOut "rscName: $rscName"

for i in $(seq 1 $ClpOcfParamCnt); do
    CLP_OCF_PARAM_VAR="CLP_OCF_PARAM${i}"
    if [[ -z ${!CLP_OCF_PARAM_VAR} ]]; then
        WriteStdErr "The environment variable '$CLP_OCF_PARAM_VAR' has not been set."
        SetAlertMessage "$AlertInternalError"
        exit $ErrorInternalError
    fi
done
actServerRegion=$CLP_OCF_PARAM1
domainFqdn=$CLP_OCF_PARAM2
actServerZoneOcid=$CLP_OCF_PARAM3
actServerIpAddress=$CLP_OCF_PARAM4
ttlSec=$CLP_OCF_PARAM5
# deleteRecord=$CLP_OCF_PARAM6
useProxySetting=$CLP_OCF_PARAM7
operationScope=$CLP_OCF_PARAM8
ociTimeoutSec=$CLP_OCF_PARAM9
for i in "ttlSec" "useProxySetting" \
         "operationScope" "ociTimeoutSec"; do
    if ! IsNonNegativeNumber "${!i}"; then
        WriteStdErr "'$i' contains non-negative values. (${!i})"
        SetAlertMessage "$AlertInternalError"
        exit $ErrorInternalError
    fi
done
WriteStdOut "actServerRegion: $actServerRegion"
WriteStdOut "domainFqdn: $domainFqdn"
WriteStdOut "actServerZoneOcid: $actServerZoneOcid"
WriteStdOut "actServerIpAddress: $actServerIpAddress"
WriteStdOut "ttlSec: $ttlSec"
# WriteStdOut "deleteRecord: $deleteRecord"
WriteStdOut "useProxySetting: $useProxySetting"
WriteStdOut "operationScope: $operationScope"
WriteStdOut "ociTimeoutSec: $ociTimeoutSec"


#
# OCI CLIインストール確認 (コマンドの存在確認)
#
if ! command -v oci >/dev/null; then
    WriteStdErr "The OCI CLI command was not found."
    SetAlertMessage "$AlertNotExistOCICmd"
    exit $ErrorNotExistOCICmd
fi


#
# プロキシ情報の設定
#
if [[ $useProxySetting -eq 1 ]]; then
    SetProxy
fi


#
# '<インストールディレクトリ>/work/ocdns' の作成
#
umask 033
installDir=$(cd "$(dirname "$0")/../.." && pwd)
workDir="${installDir}/work/${RscType}"
mkdir -p "$workDir"


#
# OCI CLIのオプションに指定するJSONファイルの作成
# '<インストールディレクトリ>/work/ocdns/<リソース名>.json'
#
jsonFileName="${rscName}.json"
jsonFilePath="${workDir}/${jsonFileName}"
cat << EOF > "$jsonFilePath"
[
    {
        "domain": "$domainFqdn",
        "rdata": "$actServerIpAddress",
        "rtype": "A",
        "ttl": "$ttlSec"
    }
]
EOF


#
# OCI CLIの実行コマンドライン設定
#
ociCmdLine="oci dns record rrset update"
ociCmdLine+=" --domain \"$domainFqdn\""
ociCmdLine+=" --zone-name-or-id \"$actServerZoneOcid\""
ociCmdLine+=" --rtype A"
ociCmdLine+=" --items \"file://$jsonFilePath\""
ociCmdLine+=" --region \"$actServerRegion\""
ociCmdLine+=" --force"
if $DebugMode; then
    ociCmdLine+=" --debug"
fi
WriteStdOut "[CommandLine] $ociCmdLine"


#
# OCI CLIの実行
#
# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
eval "$(eval "timeout -s SIGKILL $ociTimeoutSec $ociCmdLine" \
            2> >(ociResultErr=$(cat); declare -p ociResultErr) \
            1> >(ociResult=$(cat); declare -p ociResult); \
            ociExitCode=$?; declare -p ociExitCode )"
if [[ $ociExitCode -eq $ExitCodeWithTimeout ]]; then
    # OCI CLIタイムアウト
    WriteStdErr "The OCI CLI command timed out."
    SetAlertMessage "$AlertOciTimeout"
    rm -f "$jsonFilePath"
    exit $ErrorOciTimeout
fi
if [[ $ociExitCode -ne $ExCmdSuccess ]]; then
    # OCI CLI異常終了
    WriteStdErr "The OCI CLI command failed. ($ociExitCode)"
    ociResultErr=$(echo "$ociResultErr" | TrimWhiteSpace)
    WriteStdErr "$ociResultErr"
    if ! cause=$(ExtractOCIErrorCause "$ociResultErr" "$ShmMaxSize"); then
        cause=$ErrorCauseUnknown
    fi
    SetAlertMessage "${AlertOciFailed/\%1/$cause}"
    rm -f "$jsonFilePath"
    exit $ErrorOciFailed
fi
WriteDebug "$ociResult"
WriteStdOut "Created an A record in the DNS Zone. region: $actServerRegion"


#
# [クラスタサーバが所属する全てのリージョン] が設定されている場合は、
# 待機系サーバが所属するリージョンのDNSに対してもAレコードの作成(更新)を行う
# ※なお、作成(更新)に失敗してもステータスは正常
#
declare -A otherServers
if [[ $operationScope -eq $OperationAllServerRegion ]]; then
    # サーバ毎に設定しているリージョン・ゾーンOCIDを取得
    allServersXmlPath="/root/server"
    rscBaseXmlPath="/root/resource/${RscType}@${rscName}"
    allServers=$(GetClpConfValue "-e" "$allServersXmlPath") && {
        for server in $allServers; do
            regionXmlPath="$rscBaseXmlPath/parameters/region"
            region=$(GetClpConfValue "-g" "$regionXmlPath" "$server") || continue
            zoneOcidXmlPath="$rscBaseXmlPath/parameters/zoneid"
            zoneOcid=$(GetClpConfValue "-g" "$zoneOcidXmlPath" "$server") || continue
            otherServers["$region"]=$zoneOcid
        done

        # 現用系サーバが所属するリージョンのDNSに対しては既に設定済みのため除外
        unset "otherServers[$actServerRegion]"

        for otherRegion in "${!otherServers[@]}"; do
            otherZoneOcid="${otherServers[$otherRegion]}"
            WriteStdOut "otherRegion: $otherRegion"
            WriteStdOut "otherZoneOcid: $otherZoneOcid"

            #
            # OCI CLIの実行コマンドライン設定
            #
            ociCmdLine="oci dns record rrset update"
            ociCmdLine+=" --domain \"$domainFqdn\""
            ociCmdLine+=" --zone-name-or-id \"$otherZoneOcid\""
            ociCmdLine+=" --rtype A"
            ociCmdLine+=" --items \"file://$jsonFilePath\""
            ociCmdLine+=" --region \"$otherRegion\""
            ociCmdLine+=" --force"
            if $DebugMode; then
                ociCmdLine+=" --debug"
            fi
            WriteStdOut "[CommandLine] $ociCmdLine"

            eval "$(eval "timeout -s SIGKILL $ociTimeoutSec $ociCmdLine" \
                        2> >(ociResultErr=$(cat); declare -p ociResultErr) \
                        1> >(ociResult=$(cat); declare -p ociResult); \
                        ociExitCode=$?; declare -p ociExitCode )"
            if [[ $ociExitCode -eq $ExitCodeWithTimeout ]]; then
                # OCI CLIタイムアウト
                WriteStdErr "The OCI CLI command timed out."
                WriteStdErr "Skip processing for DNS of the region to which Server $server belongs."
                continue
            fi
            if [[ $ociExitCode -ne $ExCmdSuccess ]]; then
                # OCI CLI異常終了
                WriteStdErr "The OCI CLI command failed. ($ociExitCode)"
                ociResultErr=$(echo "$ociResultErr" | TrimWhiteSpace)
                WriteStdErr "$ociResultErr"
                WriteStdErr "Skip processing for DNS of the region to which Server $server belongs."
                continue
            fi
            WriteDebug "$ociResult"
            WriteStdOut "Created an A record in the DNS Zone. region: $otherRegion"
        done
    }
fi
rm -f "$jsonFilePath"


# ----------------------------------------------------------
#
# 処理終了(正常)
#
# ----------------------------------------------------------
WriteStdOut "Succeeded in activating the Oracle Cloud DNS resource."
exit $Success
