#!/bin/bash
#
##
# clpgcdns-stop.sh
##
# stop the clpgcdns
##


##
# function
##


# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --rsc -g "${GROUP_NAME}" -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# gcloud cli error handling
function gcloudcli_err_handle {
	ERR_DETAIL=$1
	GCLOUDCLI_DETAIL=$2
	echo "${GCLOUDCLI_DETAIL}"
	GCLOUDCLI_DETAIL=$(ExtractGCPErrorCause "${GCLOUDCLI_DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		GCLOUDCLI_DETAIL="${GCDNS_ERR_GCLODCLI_MSG}"
	fi
	ALTMSG="${ERR_DETAIL}(${GCLOUDCLI_DETAIL})"
	ALTMSG=$(Trancate "${ALTMSG}" ${MAX_SHMRM_SIZE})
	set_alert_message "${ALTMSG}"
}


# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# stop the clpgcdns
##

echo "clpgcdns stop script start."


##
# ret code
##
GCDNS_SUCCESS_CODE=0						# succeed
GCDNS_ERR_GET_VALUE_CODE=50					# failed to get the setting value
GCDNS_ERR_GET_RECORD_CODE=51				# get dns record has failed
GCDNS_ERR_TRN_START_CODE=52					# start transaction has failed
GCDNS_ERR_REMOVE_RECORD_CODE=53				# remove dns record has failed
# GCDNS_ERR_ADD_RECORD_CODE=54				# add dns record has failed
GCDNS_ERR_TRN_EXEC_CODE=55					# transaction execute has failed
GCDNS_ERR_PARAM_CODE=56						# invalid parameter
GCDNS_ERR_SCH_GCLOUDCLI_CODE=57				# failed to search gcloud cli path


##
# resultmsg
##
GCDNS_ERR_GCLODCLI_MSG="The gcloud CLI command failed."
GCDNS_ERR_GET_VALUE_MSG="Failed to obtain the setting value."
GCDNS_ERR_GET_RECORD_MSG="Failed to obtain the record set."
GCDNS_ERR_TRN_START_MSG="Failed to start the transaction."
GCDNS_ERR_REMOVE_RECORD_MSG="Failed to delete the record set."
# GCDNS_ERR_ADD_RECORD_MSG="Failed to add the record set."
GCDNS_ERR_TRN_EXEC_MSG="Failed to execute the transaction."
GCDNS_ERR_PARAM_MSG="Detected an invalid parameter."
GCDNS_ERR_SCH_GCLOUDCLI_MSG="The gcloud CLI command is not found."


##
# set the value to use for the shmrmset command
##
GROUP_NAME="${CLP_GROUPNAME}"				# group name
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="gcdns"						# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${GROUP_NAME}" ] || [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# gcloud cli path search
##
which gcloud > /dev/null 2>&1
SCH_GCLOUDCLI_EXIT_CODE=$?
if [[ ${SCH_GCLOUDCLI_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${GCDNS_ERR_SCH_GCLOUDCLI_MSG}"
	echo "failed to search gcloud cli path." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_SCH_GCLOUDCLI_CODE}
fi


##
# set value
##
ZONE_NAME="${CLP_OCF_PARAM1}"				# zone name (ex. "test-zone")
RECORD_NAME="${CLP_OCF_PARAM2}"				# dns record (ex. "test-cluster.example.com")
RECORD_TYPE="${CLP_OCF_PARAM3}"				# record type (hidden settings) (ex. "A")
RECORD_DELETE_FLAG="${CLP_OCF_PARAM6}"		# delete records when inactive flag

RETRY=10									# number of transaction execution result confirmations
INTERVAL=1									# transaction execution result confirmation interval

if [ -z "$ZONE_NAME" ] || [ -z "$RECORD_NAME" ] || [ -z "$RECORD_TYPE" ] || [ -z "$RECORD_DELETE_FLAG" ]
then
	set_alert_message "${GCDNS_ERR_GET_VALUE_MSG}"
	echo "failed to get the setting value." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_GET_VALUE_CODE}
fi

##
# get dns record
##
CURRENT="`gcloud dns record-sets list -z=\"${ZONE_NAME}\" --name=\"${RECORD_NAME}\" --type=\"${RECORD_TYPE}\" \
	--format=\"table[no-heading](name,type,ttl, rrdatas.list():label=DATA)\" 2>&1`"
if [ $? -ne 0 ]
then
	gcloudcli_err_handle "${GCDNS_ERR_GET_RECORD_MSG}" "${CURRENT}"
	echo "get dns record has failed." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_GET_RECORD_CODE}
elif [ -z "$CURRENT" ]
then
	echo "there is no designated dns record."
	echo "clpgcdns stop script has succeeded."
	exit ${GCDNS_SUCCESS_CODE}
else
	CURRENT_NAME="`echo ${CURRENT} | awk '{print $1}'`"
	CURRENT_TYPE="`echo ${CURRENT} | awk '{print $2}'`"
	CURRENT_TTL="`echo ${CURRENT} | awk '{print $3}'`"
	CURRENT_IP="`echo ${CURRENT} | awk '{print $4}'`"

	echo "get dns record has succeeded (name=${CURRENT_NAME} type=${CURRENT_TYPE} ttl=${CURRENT_TTL} ip=${CURRENT_IP})."
fi



##
# delete records when inactive flag
##
if [ $RECORD_DELETE_FLAG -eq 0 ]
then
	echo "clpgcdns stop script has succeeded."
	exit ${GCDNS_SUCCESS_CODE}
elif [ $RECORD_DELETE_FLAG -eq 1 ]
then
	: #do nothing
else
	set_alert_message "${GCDNS_ERR_PARAM_MSG}"
	echo "invalid parameter." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_PARAM_CODE}
fi


##
# abort transaction
##
gcloud dns record-sets transaction abort -z="${ZONE_NAME}" > /dev/null 2>&1


##
#
# unegister dns record
#
##


##
# start transaction
##
TRN_STR=$(gcloud dns record-sets transaction start -z="${ZONE_NAME}" 2>&1)
if [ $? -ne 0 ]
then
	gcloud dns record-sets transaction abort -z="${ZONE_NAME}" > /dev/null 2>&1
	gcloudcli_err_handle "${GCDNS_ERR_TRN_START_MSG}" "${TRN_STR}"
	echo "start transaction has failed." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_TRN_START_CODE}
fi
echo "${TRN_STR}"


##
# remove dns record
##
RM_RECORD=$(gcloud dns record-sets transaction remove -z="${ZONE_NAME}" --name="${CURRENT_NAME}" \
	--ttl="${CURRENT_TTL}" --type="${CURRENT_TYPE}" "${CURRENT_IP}" 2>&1)
if [ $? -ne 0 ]
then
	gcloud dns record-sets transaction abort -z="${ZONE_NAME}" > /dev/null 2>&1
	gcloudcli_err_handle "${GCDNS_ERR_REMOVE_RECORD_MSG}" "${RM_RECORD}"
	echo "remove dns record has failed." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_REMOVE_RECORD_CODE}
fi
echo "${RM_RECORD}"


##
# execute transaction
##
CHANGE_ID="`gcloud dns record-sets transaction execute -z=\"${ZONE_NAME}\" --format=\"table[no-heading](id)\" 2>&1`"
if [ $? -ne 0 ]
then
	gcloud dns record-sets transaction abort -z="${ZONE_NAME}" > /dev/null 2>&1
	gcloudcli_err_handle "${GCDNS_ERR_TRN_EXEC_MSG}" "${CHANGE_ID}"
	echo "transaction execute has failed." 1>&2
	echo "clpgcdns stop script has failed." 1>&2
	exit ${GCDNS_ERR_TRN_EXEC_CODE}
fi
echo "${CHANGE_ID}"
CHANGE_ID=$(echo "${CHANGE_ID}" | tail -n 1)


##
# confirmation of transaction execution result
##
for i in `seq ${RETRY}`
do
	sleep ${INTERVAL}

	STATUS="`gcloud dns record-sets changes describe \"${CHANGE_ID}\" -z=\"${ZONE_NAME}\" --format=\"table[no-heading](status)\"`"

	if [ $? -ne 0 ]
	then
		echo "check the transaction execution result has failed." 1>&2
		echo "clpgcdns stop script has succeeded."
		exit ${GCDNS_SUCCESS_CODE}
	elif [ "$STATUS" = "done" ]
	then
		echo "check the transaction execution result has succeeded."
		break
	fi
done

if [ "$STATUS" != "done" ]
then
	echo "transaction execution status is ${STATUS}." 1>&2
fi


##
# end
##
echo "clpgcdns stop script has succeeded."

exit ${GCDNS_SUCCESS_CODE}

