#!/bin/bash
#
##
# clpawsvip-watch.sh
##
# [coding：utf-8]
##


##
# function
##

# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --mon -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# get the code according to the [Action when AWS CLI command failed to receive response]
function shift_status_on_awscli_err {
	# (Note:)
	#	[回復動作を実行しない(警告を表示しない)] 場合は、
	#	上位モジュールへ「正常」で返却するため
	#	CLI実行権限不足などのケースを含めた、すべてのケースでアラート通知しません
	EXIT_ERR=$1
	EXIT_WARN=$2
	case ${MON_MODE} in
		"${NO_RECOVERY_NO_WARN}") echo ${AWSVIPW_SUCCESS_CODE} ;;
		"${NO_RECOVERY_WARN}") echo "${EXIT_WARN}" ;;
		"${RECOVERY}") echo "${EXIT_ERR}" ;;
		*)
			echo "unknown MON_MODE(${MON_MODE})" 1>&2
			echo "${EXIT_ERR}" ;;
	esac
}


# aws cli error handling
function awscli_err_handle {
	DETAIL=$1
	AWSCLI_ALTMSG=$(ExtractAWSErrorCause "${DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		AWSCLI_ALTMSG="${AWSVIPW_ERR_WARN_AWSCLI_MSG}"
	fi
	set_alert_message "${AWSCLI_ALTMSG}"
}


# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# monitoring the clpawsvip
##

echo "clpawsvip watch script start."


##
# ret code
##
# success
AWSVIPW_SUCCESS_CODE=0					# succeed
# error
AWSVIPW_ERR_AWSCLI_CODE=50				# failed in the awscli command
AWSVIPW_ERR_CHKRT_CODE=56				# failed to check route table
AWSVIPW_ERR_CHKVIP_CODE=57				# failed to check virtual ip address
# warning
WARN_OFFSET=100															# warning offset value
AWSVIPW_WARN_AWSCLI_CODE=$((AWSVIPW_ERR_AWSCLI_CODE + WARN_OFFSET))		# failed in the awscli command(150)
AWSVIPW_WARN_SCHAWSCLI_CODE=151											# failed to search aws cli path
AWSVIPW_WARN_GETOCFENV_CODE=152											# failed to get the ocf environment variable
AWSVIPW_WARN_GETVIP_CODE=153											# get virtual ip has failed
AWSVIPW_WARN_GETVPCID_CODE=154											# get vpc id has failed
AWSVIPW_WARN_GETENIID_CODE=155											# get eni id has failed
AWSVIPW_WARN_SETENV_CODE=158											# failed to set the aws cli environment variable
AWSVIPW_WARN_GETCMDOPT_CODE=159											# failed to get aws cli command line options


##
# resultmsg
##
# error & warning
AWSVIPW_ERR_WARN_AWSCLI_MSG="The AWS CLI command failed."
# error
AWSVIPW_ERR_CHKRT_MSG="" #The routing for VIP ${VIRTUAL_IP} was changed.
AWSVIPW_ERR_CHKVIP_MSG="" #The VIP address ${VIRTUAL_IP} does not exist.
# warning
AWSVIPW_WARN_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSVIPW_WARN_GETOCFENV_MSG="Failed to obtain the setting value."
AWSVIPW_WARN_GETVIP_MSG="Failed to obtain the virtual IP address."
AWSVIPW_WARN_GETVPCID_MSG="Failed to obtain the VPC ID."
AWSVIPW_WARN_GETENIID_MSG="Failed to obtain the ENI ID."
AWSVIPW_WARN_SETENV_MSG="Failed to obtain the environment variables."
AWSVIPW_WARN_GETCMDOPT_MSG="Failed to obtain the AWS CLI command line options."


##
# Action when AWS CLI command failed to receive response
##
NO_RECOVERY_NO_WARN=0
NO_RECOVERY_WARN=1
RECOVERY=2


##
# set the value to use for the shmrmset command
##
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="awsvipw"					# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# awscli path
##
export PATH=$PATH:/usr/local/bin


##
# awscli path search
##
which aws > /dev/null 2>&1

if [ $? -ne 0 ]
then
	set_alert_message "${AWSVIPW_WARN_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_SCHAWSCLI_CODE}
fi


##
# set environment variables for use with the aws cli
##
clpcloudutil_env_init "${clpcloudutil_false}" "${CURRENT}/clpaws_setting.conf"
ENV_INIT_EXIT_CODE=$?
if [[ ${ENV_INIT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSVIPW_WARN_SETENV_MSG}"
	echo "Failed to obtain the environment variables." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_SETENV_CODE}
fi


##
# get aws cli command line options
##
EC2_CMDOPT="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_false}\" ec2`"
GET_AWSCLI_CMDOPT_EXIT_CODE=$?
if [[ ${GET_AWSCLI_CMDOPT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSVIPW_WARN_GETCMDOPT_MSG}"
	echo "Failed to obtain the aws cli command line options." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_GETCMDOPT_CODE}
fi


##
# set ocf environment variable
##
TARGET_RSC="${CLP_OCF_PARAM1}"			# resource name
MON_MODE="${CLP_OCF_PARAM2}"			# action when aws cli command failed to receive response
PING_TIMEOUT="${CLP_OCF_PARAM3}"		# ping timeout

if [ -z "${TARGET_RSC}" ] || [ -z "${MON_MODE}" ] || [ -z "${PING_TIMEOUT}" ]
then
	set_alert_message "${AWSVIPW_WARN_GETOCFENV_MSG}"
	echo "failed to get the ocf environment variable." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_GETOCFENV_CODE}
fi


##
# set server name
##
SRV_FULLNAME="`uname -n`"				# get hostname (use output file path)
SRV_NAME="`echo ${SRV_FULLNAME%%.*}`"	# delete domain name


##
# get target resource config
##
VIRTUAL_IP="`clpcfget -g /root/resource/awsvip@\"${TARGET_RSC}\"/parameters/ip -p awsvip`"
if [ $? -ne 0 ]
then
	set_alert_message "${AWSVIPW_WARN_GETVIP_MSG}"
	echo "${VIRTUAL_IP}" | head -n 1 1>&2
	echo "get virtual ip has failed." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_GETVIP_CODE}
fi

VPC_ID="`clpcfget -g /root/resource/awsvip@\"${TARGET_RSC}\"/parameters/vpcid -p awsvip -s \"${SRV_NAME}\"`"
if [ $? -ne 0 ]
then
	set_alert_message "${AWSVIPW_WARN_GETVPCID_MSG}"
	echo "${VPC_ID}" | head -n 1 1>&2
	echo "get vpc id has failed." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_GETVPCID_CODE}
fi

ENI_ID="`clpcfget -g /root/resource/awsvip@\"${TARGET_RSC}\"/parameters/eniid -p awsvip -s \"${SRV_NAME}\"`"
if [ $? -ne 0 ]
then
	set_alert_message "${AWSVIPW_WARN_GETENIID_MSG}"
	echo "${ENI_ID}" | head -n 1 1>&2
	echo "get eni id has failed." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_WARN_GETENIID_CODE}
fi

echo "get config has succeeded. (vip=${VIRTUAL_IP} vpcid=${VPC_ID} eniid=${ENI_ID})"

##
# check virtual ip exists
##
clpping -c ${PING_TIMEOUT} "${VIRTUAL_IP}" > /dev/null 2>&1
if [ $? -ne 0 ]
then
	AWSVIPW_ERR_CHKVIP_MSG="The VIP address \"${VIRTUAL_IP}\" does not exist."
	set_alert_message "${AWSVIPW_ERR_CHKVIP_MSG}"
	echo "failed to check virtual ip address." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_ERR_CHKVIP_CODE}
fi

echo "Ping succeeded."

##
# check route table
##
CIDR_BLOCK="/32"
VIP_CIDR_BLOCK="$VIRTUAL_IP$CIDR_BLOCK"

CHG_RT_IDLIST=$(aws ec2 describe-route-tables \
	--filters "Name=vpc-id,Values=${VPC_ID}" "Name=route.destination-cidr-block,Values=${VIP_CIDR_BLOCK}" \
		--query "RouteTables[?contains(Routes[?DestinationCidrBlock=='${VIP_CIDR_BLOCK}'].NetworkInterfaceId,'${ENI_ID}')==\`false\`].RouteTableId" ${EC2_CMDOPT} --output json 2>&1)
if [ $? -ne 0 ]
then
	echo "${CHG_RT_IDLIST}" 1>&2
	echo "failed to check route table." 1>&2
	# AWS CLI コマンド応答取得失敗時動作の設定によって終了ステータスを変更する。
	# 設定が「回復動作を実行しない(警告を表示しない)」の場合は正常終了
	AWSCLI_FAILED_CODE=$(shift_status_on_awscli_err ${AWSVIPW_ERR_AWSCLI_CODE} ${AWSVIPW_WARN_AWSCLI_CODE})
	if [ "${AWSCLI_FAILED_CODE}" -eq ${NO_RECOVERY_NO_WARN} ]
	then
		echo "clpawsvip watch script has succeeded."
	else
		awscli_err_handle "${CHG_RT_IDLIST}"
		echo "clpawsvip watch script has failed." 1>&2
	fi
	exit "${AWSCLI_FAILED_CODE}"
elif [ "${CHG_RT_IDLIST}" != "[]" ]
then
	AWSVIPW_ERR_CHKRT_MSG="The routing for VIP \"${VIRTUAL_IP}\" was changed."
	set_alert_message "${AWSVIPW_ERR_CHKRT_MSG}"
	echo "vip ${VIRTUAL_IP} route of ${CHG_RT_IDLIST} is changed." 1>&2
	echo "clpawsvip watch script has failed." 1>&2
	exit ${AWSVIPW_ERR_CHKRT_CODE}
fi

echo "check route table has succeeded."


##
# end
##
echo "clpawsvip watch script has succeeded."

exit ${AWSVIPW_SUCCESS_CODE}
