#!/bin/bash
#
##
# clpawssip-watch.sh
##
# [coding：utf-8]
##


##
# function
##


# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --mon -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# get the code according to the [Action when AWS CLI command failed to receive response]
function shift_status_on_awscli_err {
	# (Note:)
	#	[回復動作を実行しない(警告を表示しない)] 場合は、
	#	上位モジュールへ「正常」で返却するため
	#	CLI実行権限不足などのケースを含めた、すべてのケースでアラート通知しません
	EXIT_ERR=$1
	EXIT_WARN=$2
	case ${MON_MODE} in
		"${NO_RECOVERY_NO_WARN}") echo ${AWSSIPW_SUCCESS_CODE} ;;
		"${NO_RECOVERY_WARN}") echo "${EXIT_WARN}" ;;
		"${RECOVERY}") echo "${EXIT_ERR}" ;;
		*)
			echo "unknown MON_MODE(${MON_MODE})" 1>&2
			echo "${EXIT_ERR}" ;;
	esac
}


# aws cli error handling
function awscli_err_handle {
	ERR_DETAIL=$1
	AWSCLI_DETAIL=$2
	AWSCLI_DETAIL=$(ExtractAWSErrorCause "${AWSCLI_DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		AWSCLI_DETAIL="${AWSSIPW_ERR_WARN_AWSCLI_MSG}"
	fi
	ALTMSG="${ERR_DETAIL}(${AWSCLI_DETAIL})"
	ALTMSG=$(Trancate "${ALTMSG}" ${MAX_SHMRM_SIZE})
	set_alert_message "${ALTMSG}"
}


# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# monitoring the clpawssip
##

echo "clpawssip watch script start."


##
# ret code
##
# success
AWSSIPW_SUCCESS_CODE=0					# succeed
# error
AWSSIPW_ERR_AWSCHKIP_CODE=54			# failed to check ip address for aws
AWSSIPW_ERR_OSCHKIP_CODE=55				# failed to check ip address for os
# warning
WARN_OFFSET=100																	# warning offset value
AWSSIPW_WARN_SCHAWSCLI_CODE=150													# failed to search aws cli path
AWSSIPW_WARN_GETOCFENV_CODE=151													# failed to get the ocf environment variable
AWSSIPW_WARN_GETSECIP_CODE=152													# get secondary ip has failed
AWSSIPW_WARN_GETENIID_CODE=153													# get eni id has failed
AWSSIPW_WARN_AWSCHKIP_CODE=$((AWSSIPW_ERR_AWSCHKIP_CODE + WARN_OFFSET))			# failed to check ip address for aws(154)
AWSSIPW_WARN_SETENV_CODE=156													# failed to set the aws cli environment variable
AWSSIPW_WARN_GETCMDOPT_CODE=157													# failed to get aws cli command line options



##
# resultmsg
##
# error & warning
AWSSIPW_ERR_WARN_AWSCLI_MSG="The AWS CLI command failed."
AWSSIPW_ERR_WARN_AWSCHKIP_MSG="Failed to process checking the secondary IP address on the AWS side."
# error
AWSSIPW_ERR_OSCHKIP_MSG="Failed to process checking the secondary IP address on the OS side."
# warning
AWSSIPW_WARN_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSSIPW_WARN_GETOCFENV_MSG="Failed to obtain the setting value."
AWSSIPW_WARN_GETSECIP_MSG="Failed to obtain the secondary IP address."
AWSSIPW_WARN_GETENIID_MSG="Failed to obtain the ENI ID."
AWSSIPW_WARN_SETENV_MSG="Failed to obtain the environment variables."
AWSSIPW_WARN_GETCMDOPT_MSG="Failed to obtain the AWS CLI command line options."


##
# Action when AWS CLI command failed to receive response
##
NO_RECOVERY_NO_WARN=0
NO_RECOVERY_WARN=1
RECOVERY=2


##
# set the value to use for the shmrmset command
##
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="awssipw"						# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# awscli path
##
export PATH=$PATH:/usr/local/bin


##
# awscli path search
##
which aws > /dev/null 2>&1

if [ $? -ne 0 ]
then
	set_alert_message "${AWSSIPW_WARN_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_SCHAWSCLI_CODE}
fi


##
# set environment variables for use with the aws cli
##
clpcloudutil_env_init "${clpcloudutil_false}" "${CURRENT}/clpaws_setting.conf"
ENV_INIT_EXIT_CODE=$?
if [[ ${ENV_INIT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSSIPW_WARN_SETENV_MSG}"
	echo "Failed to obtain the environment variables." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_SETENV_CODE}
fi


##
# get aws cli command line options
##
EC2_CMDOPT="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_false}\" ec2`"
GET_AWSCLI_CMDOPT_EXIT_CODE=$?
if [[ ${GET_AWSCLI_CMDOPT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSSIPW_WARN_GETCMDOPT_MSG}"
	echo "Failed to obtain the aws cli command line options." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_GETCMDOPT_CODE}
fi


##
# set ocf environment variable
##
TARGET_RSC="${CLP_OCF_PARAM1}"		# resource name
MON_MODE="${CLP_OCF_PARAM2}"			# action when aws cli command failed to receive response
PING_TIMEOUT="${CLP_OCF_PARAM3}"		# ping timeout

if [ -z "${TARGET_RSC}" ] || [ -z "${MON_MODE}" ] || [ -z "${PING_TIMEOUT}" ]
then
	set_alert_message "${AWSSIPW_WARN_GETOCFENV_MSG}"
	echo "failed to get the ocf environment variable." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_GETOCFENV_CODE}
fi


##
# set server name
##
SRV_FULLNAME="`uname -n`"				# get hostname (use output file path)
SRV_NAME="`echo ${SRV_FULLNAME%%.*}`"	# delete domain name


##
# get target resource config
##
SECONDARY_IP="`clpcfget -g /root/resource/awssip@\"${TARGET_RSC}\"/parameters/ip -p awssip`"
if [ $? -ne 0 ]
then
	set_alert_message "${AWSSIPW_WARN_GETSECIP_MSG}"
	echo "${SECONDARY_IP}" | head -n 1 1>&2
	echo "get secondary ip has failed." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_GETSECIP_CODE}
fi

ENI_ID="`clpcfget -g /root/resource/awssip@\"${TARGET_RSC}\"/parameters/eniid -p awssip -s \"${SRV_NAME}\"`"
if [ $? -ne 0 ]
then
	set_alert_message "${AWSSIPW_WARN_GETENIID_MSG}"
	echo "${ENI_ID}" | head -n 1 1>&2
	echo "get eni id has failed." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_WARN_GETENIID_CODE}
fi

echo "get config has succeeded (secip=${SECONDARY_IP} eniid=${ENI_ID})."


##
# determine whether the secondary IP address is ipv4 or ipv6
##
if [ `echo "${SECONDARY_IP}" | grep :` ]
then
	SECONDARYIP_TYPE="IPv6"
	echo "secondary ip address is ipv6."
else
	SECONDARYIP_TYPE="IPv4"
	echo "secondary ip address is ipv4."
fi


##
# check secondary ip exists in aws
##
if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
then
	AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
							--filters \"Name=ipv6-addresses.ipv6-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text 2>&1`"
elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
then
	AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
							--filters \"Name=addresses.private-ip-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text 2>&1`"
fi

if [ $? -ne 0 ]
then
	echo "${AWS_CHECK_IP}" 1>&2
	echo "failed to check the existence of ip address of aws." 1>&2
	# AWS CLI コマンド応答取得失敗時動作の設定によって終了ステータスを変更する。
	# 設定が「回復動作を実行しない(警告を表示しない)」の場合は正常終了
	AWSCLI_FAILED_CODE=$(shift_status_on_awscli_err ${AWSSIPW_ERR_AWSCHKIP_CODE} ${AWSSIPW_WARN_AWSCHKIP_CODE})
	if [ "${AWSCLI_FAILED_CODE}" -eq ${NO_RECOVERY_NO_WARN} ]
	then
		echo "clpawssip watch script has failed." 1>&2
	else
		awscli_err_handle "${AWSSIPW_ERR_WARN_AWSCHKIP_MSG}" "${AWS_CHECK_IP}"
		echo "clpawssip watch script has failed." 1>&2
	fi
	exit "${AWSCLI_FAILED_CODE}"
elif [ -z "${AWS_CHECK_IP}" ]
then
	set_alert_message "${AWSSIPW_ERR_WARN_AWSCHKIP_MSG}(The secondary IP address does not exist.)"
	echo "failed to check the existence of ip address of aws." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_ERR_AWSCHKIP_CODE}
fi


##
# check secondary ip exists in os
##
clpping -c ${PING_TIMEOUT} "${SECONDARY_IP}" > /dev/null 2>&1
if [ $? -ne 0 ]
then
	set_alert_message "${AWSSIPW_ERR_OSCHKIP_MSG}"
	echo "failed to check ip address for os." 1>&2
	echo "clpawssip watch script has failed." 1>&2
	exit ${AWSSIPW_ERR_OSCHKIP_CODE}
fi


##
# end
##
echo "clpawssip watch script has succeeded."

exit ${AWSSIPW_SUCCESS_CODE}

