#!/bin/bash
#
##
# clpawssip-start.sh
##
# [coding：utf-8]
##


##
# function
##


# set alert log strings to shared memory
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --rsc -g "${GROUP_NAME}" -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_RET=$?
	if [ ${SHMRMSET_RET} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_RET})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# aws cli error handling
function awscli_err_handle {
	ERR_DETAIL=$1
	AWSCLI_DETAIL=$2
	echo "${AWSCLI_DETAIL}"
	AWSCLI_DETAIL=$(ExtractAWSErrorCause "${AWSCLI_DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [ ${ERRCAUSE_RET} -ne 0 ]
	then
		AWSCLI_DETAIL="${AWSSIP_ERR_AWSCLI_MSG}"
	fi
	ALTMSG="${ERR_DETAIL}(${AWSCLI_DETAIL})"
	ALTMSG=$(Trancate "${ALTMSG}" ${MAX_SHMRM_SIZE})
	set_alert_message "${ALTMSG}"
}


# unassign secondary ip addresses
function unassign_sip(){
	if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
	then
		aws ec2 unassign-ipv6-addresses --network-interface-id "${ENI_ID}" --ipv6-addresses "${SECONDARY_IP}" ${EC2_CMDOPT} > /dev/null 2>&1
	elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
	then
		aws ec2 unassign-private-ip-addresses --network-interface-id "${ENI_ID}" --private-ip-addresses "${SECONDARY_IP}" ${EC2_CMDOPT} > /dev/null 2>&1
	fi
}


# clpcloudutil calls
CURRENT=$(cd $(dirname $0);pwd)
. "${CURRENT}/../common/clpcloudutil.sh"


##
# start the clpawssip
##

echo "clpawssip start script start."


##
# ret code
##
AWSSIP_SUCCESS_CODE=0					# succeed
AWSSIP_ERR_SCHAWSCLI_CODE=50			# failed to search aws cli path
AWSSIP_ERR_GETOCFENV_CODE=51			# failed to get the ocf environment variable
AWSSIP_ERR_AWSASSIGNIP_CODE=52			# failed to assign ip address for aws
# AWSSIP_ERR_AWSUNASSIGNIP_CODE=53		# failed to unassign ip address for aws
AWSSIP_ERR_AWSCHKIP_CODE=54				# failed to check ip address for aws
AWSSIP_ERR_OSASSIGNIP_CODE=55			# failed to assign ip address for os
# AWSSIP_ERR_OSUNASSIGNIP_CODE=56		# failed to unassign ip address for os
AWSSIP_ERR_OSCHKIP_CODE=57				# failed to check ip address for os
AWSSIP_ERR_GETMAC_CODE=58				# failed to get mac address
AWSSIP_ERR_GETNICNAME_CODE=59			# failed to get nic name
AWSSIP_ERR_GETSUBNET_CODE=60			# failed to get subnet id
AWSSIP_ERR_GETCIDR_CODE=61				# failed to get cidrblock
AWSSIP_ERR_GETPRIIP_CODE=62				# failed to get the primary private ip address.
AWSSIP_ERR_CHKENIID_CODE=63				# eni id does not exist.


##
# resultmsg
##
AWSSIP_ERR_AWSCLI_MSG="The AWS CLI command failed."
AWSSIP_ERR_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSSIP_ERR_GETOCFENV_MSG="Failed to obtain the setting value."
AWSSIP_ERR_AWSASSIGNIP_MSG="Failed to assign the secondary IP address on the AWS side."
# AWSSIP_ERR_AWSUNASSIGNIP_MSG="Failed to release the assigned secondary IP address on the AWS side."
AWSSIP_ERR_AWSCHKIP_MSG="Failed to process checking the secondary IP address on the AWS side."
AWSSIP_ERR_OSASSIGNIP_MSG="Failed to assign the secondary IP address on the OS side."
# AWSSIP_ERR_OSUNASSIGNIP_MSG="Failed to release the assigned secondary IP address on the OS side."
AWSSIP_ERR_OSCHKIP_MSG="Failed to process checking the secondary IP address on the OS side."
AWSSIP_ERR_GETMAC_MSG="Failed to obtain a MAC address."
AWSSIP_ERR_GETNICNAME_MSG="Failed to obtain a NIC name."
AWSSIP_ERR_GETSUBNET_MSG="Failed to obtain a subnet ID."
AWSSIP_ERR_GETCIDR_MSG="Failed to obtain a CIDR block."
AWSSIP_ERR_GETPRIIP_MSG="Failed to obtain a primary private IP address."
AWSSIP_ERR_CHKENIID_MSG=""	#ENI ID is invalid.(ENI ID=%1)


##
# set the value to use for the shmrmset command
##
GROUP_NAME="${CLP_GROUPNAME}"				# group name
RESOURCE_NAME="${CLP_RESOURCENAME}"		# resource name
RESOURCE_TYPE="awssip"						# resource type
MAX_SHMRM_SIZE=$((128 - 1))				# shmrmset command max string size

if [ -z "${GROUP_NAME}" ] || [ -z "${RESOURCE_NAME}" ]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
fi


##
# awscli path
##
export PATH=$PATH:/usr/local/bin


##
# awscli path search
##
which aws > /dev/null 2>&1

if [ $? -ne 0 ]
then
	set_alert_message "${AWSSIP_ERR_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_SCHAWSCLI_CODE}
fi


##
# set environment variables for use with the aws cli
##
clpcloudutil_env_init "${clpcloudutil_true}" "${CURRENT}/clpaws_setting.conf"


##
# get aws cli command line options
##
EC2_CMDOPT="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" ec2`"


##
# set ocf environment variable
##
SECONDARY_IP="${CLP_OCF_PARAM1}"		# secondary ip address
ENI_ID="${CLP_OCF_PARAM2}"			# ENI ID
PING_TIMEOUT="${CLP_OCF_PARAM3}"		# ping timeout

if [ -z "${SECONDARY_IP}" ] || [ -z "${ENI_ID}" ] || [ -z "${PING_TIMEOUT}" ]
then
	set_alert_message "${AWSSIP_ERR_GETOCFENV_MSG}"
	echo "failed to get the ocf environment variable." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETOCFENV_CODE}
fi


##
# check ENI ID
##
PRIMARY_IP=$(aws ec2 describe-network-interfaces --network-interface-ids "${ENI_ID}" \
	--query "NetworkInterfaces[].PrivateIpAddresses[?Primary==\`true\`].PrivateIpAddress" --output text ${EC2_CMDOPT} 2>&1)
GETPRIIP_RET=$?
if [ ${GETPRIIP_RET} -ne 0 ]
then
	awscli_err_handle "${AWSSIP_ERR_GETPRIIP_MSG}" "${PRIMARY_IP}"
	echo "failed to get the primary private ip address." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETPRIIP_CODE}
elif [ -z "${PRIMARY_IP}" ]
then
	set_alert_message "${AWSSIP_ERR_GETPRIIP_MSG}(Internal error)"
	echo "primary ip address does not exist." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETPRIIP_CODE}
fi

CHKENIID_FLAG=0
OS_IP_LIST=$(ip -f inet -o addr show|cut -d\  -f 7 | cut -d/ -f 1)

IFS=$'\n'
for OS_IP in ${OS_IP_LIST}
do
	if [ "${PRIMARY_IP}" == "${OS_IP}" ]
	then
		echo "eni id ${ENI_ID} exists."
		CHKENIID_FLAG=$(( CHKENIID_FLAG + 1 ))
		break
	fi
done
unset IFS

if [ ${CHKENIID_FLAG} -eq 0 ]
then
	AWSSIP_ERR_CHKENIID_MSG="ENI ID is invalid.(ENI ID=\"${ENI_ID}\")"
	set_alert_message "${AWSSIP_ERR_CHKENIID_MSG}"
	echo "eni id ${ENI_ID} does not exist." 1>&2
	echo "primary IP address is ${PRIMARY_IP}." 1>&2
	OS_IP_LIST=$(echo ${OS_IP_LIST} | sed -e "s/\n//g")
	echo "ip address assigned to the os is ${OS_IP_LIST}." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_CHKENIID_CODE}
fi


##
# determine whether the secondary IP address is ipv4 or ipv6
##
if [ `echo "${SECONDARY_IP}" | grep :` ]
then
	SECONDARYIP_TYPE="IPv6"
	echo "secondary ip address is ipv6."
else
	SECONDARYIP_TYPE="IPv4"
	echo "secondary ip address is ipv4."
fi


##
# check secondary ip exists in aws
##
if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
then
	AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
							--filters \"Name=ipv6-addresses.ipv6-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text`"
elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
then
	AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
						--filters \"Name=addresses.private-ip-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text`"
fi

if [ $? -eq 0 ] && [ -n "${AWS_CHECK_IP}" ]
then
	AWS_SECONDARYIP="exists"
else
	AWS_SECONDARYIP="not exists"
fi


##
#
# execute when the secondary ip does not exist on the aws side
#
##
if [ "${AWS_SECONDARYIP}" == "exists" ]
then
	echo "skip the process because the secondary ip address of aws exists."
else


##
# assign a secondary ip to aws
##
	if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
	then
		AWS_ASSIGN_IP=$(aws ec2 assign-ipv6-addresses --network-interface-id "${ENI_ID}" --ipv6-addresses "${SECONDARY_IP}" ${EC2_CMDOPT} --output text 2>&1)
	elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
	then
		AWS_ASSIGN_IP=$(aws ec2 assign-private-ip-addresses --allow-reassignment --network-interface-id "${ENI_ID}" --private-ip-addresses "${SECONDARY_IP}" ${EC2_CMDOPT} --output text 2>&1)
	fi

	if [ $? -ne 0 ]
	then
		awscli_err_handle "${AWSSIP_ERR_AWSASSIGNIP_MSG}" "${AWS_ASSIGN_IP}"
		echo "failed to assign ip address for aws." 1>&2
		echo "clpawssip start script has failed." 1>&2
		exit ${AWSSIP_ERR_AWSASSIGNIP_CODE}
	fi


##
# check secondary ip exists in aws
##
	if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
	then
		AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
								--filters \"Name=ipv6-addresses.ipv6-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text 2>&1`"
	elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
	then
		AWS_CHECK_IP="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
							--filters \"Name=addresses.private-ip-address,Values=\"${SECONDARY_IP}\"\" ${EC2_CMDOPT} --output text 2>&1`"
	fi

	if [ $? -ne 0 ]
	then
		awscli_err_handle "${AWSSIP_ERR_AWSCHKIP_MSG}" "${AWS_CHECK_IP}"
		echo "failed to check the existence of ip address of aws." 1>&2
		echo "clpawssip start script has failed." 1>&2
		exit ${AWSSIP_ERR_AWSCHKIP_CODE}
	elif [ -z "${AWS_CHECK_IP}" ]
	then
		set_alert_message "${AWSSIP_ERR_AWSCHKIP_MSG}(The secondary IP address does not exist.)"
		echo "failed to check the existence of ip address of aws." 1>&2
		echo "clpawssip start script has failed." 1>&2
		exit ${AWSSIP_ERR_AWSCHKIP_CODE}
	fi
fi


##
# check secondary ip exists in os
##
clpping -c ${PING_TIMEOUT} "${SECONDARY_IP}" > /dev/null 2>&1
if [ $? -eq 0 ]
then
	OS_SECONDARYIP="exists"
else
	OS_SECONDARYIP="not exists"
fi


##
# If a secondary IP exists on both the AWS side and the OS side in the pre-check, it terminates normally
##
if [ "${AWS_SECONDARYIP}" == "exists" ] && [ "${OS_SECONDARYIP}" == "exists" ]
then
	echo "skip the process because the secondary ip address of os exists."
	echo "secondary ip address is exists."
	echo "clpawssip start script has succeeded."
	exit ${AWSSIP_SUCCESS_CODE}
fi


##
#
# execute when the secondary ip does not exist on the os side
#
##
if [ "${OS_SECONDARYIP}" == "exists" ]
then
	echo "skip the process because the secondary ip address of os exists."
	echo "clpawssip start script has succeeded."
	exit ${AWSSIP_SUCCESS_CODE}
fi


##
# nic name acquisition process
##
MAC_ADDRESS="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
				--query \"NetworkInterfaces[].[MacAddress]\" ${EC2_CMDOPT} --output text 2>&1`"
if [ $? -ne 0 ]
then
	unassign_sip
	awscli_err_handle "${AWSSIP_ERR_GETMAC_MSG}" "${MAC_ADDRESS}"
	echo "failed to get mac address." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETMAC_CODE}
fi

NIC_NAME="`ip link | awk '/^[0-9]/ { iface=$2 } /link/ { mac=$2 } /brd/ { print iface, mac }' | grep \"${MAC_ADDRESS}\" | awk '{ sub(\":\", \"\") }  { print $1 }'`"
if [ $? -ne 0 ] || [ -z "${NIC_NAME}" ]
then
	unassign_sip
	set_alert_message "${AWSSIP_ERR_GETNICNAME_MSG}"
	echo "failed to get nic name (mac=${MAC_ADDRESS})." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETNICNAME_CODE}
fi

echo "mac=${MAC_ADDRESS} nic=${NIC_NAME}"


##
# prefix acquisition process
##
SUBNET_ID="`aws ec2 describe-network-interfaces --network-interface-ids \"${ENI_ID}\" \
				--query \"NetworkInterfaces[].[SubnetId]\" ${EC2_CMDOPT} --output text 2>&1`"
if [ $? -ne 0 ]
then
	unassign_sip
	awscli_err_handle "${AWSSIP_ERR_GETSUBNET_MSG}" "${SUBNET_ID}"
	echo "failed to get subnet id." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETSUBNET_CODE}
fi


if [ "${SECONDARYIP_TYPE}" == "IPv6" ]
then
	CIDR="`aws ec2 describe-subnets --subnet-ids \"${SUBNET_ID}\" \
				--query \"Subnets[].Ipv6CidrBlockAssociationSet[].[Ipv6CidrBlock]\" ${EC2_CMDOPT} --output text 2>&1`"
elif [ "${SECONDARYIP_TYPE}" == "IPv4" ]
then
	CIDR="`aws ec2 describe-subnets --subnet-ids \"${SUBNET_ID}\" \
				--query \"Subnets[].[CidrBlock]\" ${EC2_CMDOPT} --output text 2>&1`"
fi

if [ $? -ne 0 ]
then
	unassign_sip
	awscli_err_handle "${AWSSIP_ERR_GETCIDR_MSG}" "${CIDR}"
	echo "failed to get the cidrblock of ip address (subnet=${SUBNET_ID})." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_GETCIDR_CODE}
fi


PREFIX="`echo ${CIDR##*/}`"

echo "subnet=${SUBNET_ID} cidr=${CIDR} prefix=${PREFIX}"


##
# assign a secondary ip to os
##
ip address add "${SECONDARY_IP}"/"${PREFIX}" dev "${NIC_NAME}"
if [ $? -ne 0 ]
then
	unassign_sip
	set_alert_message "${AWSSIP_ERR_OSASSIGNIP_MSG}"
	echo "failed to assign ip address for os." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_OSASSIGNIP_CODE}
fi


##
# check secondary ip exists in os
##
clpping -c ${PING_TIMEOUT} "${SECONDARY_IP}" > /dev/null 2>&1
if [ $? -ne 0 ]
then
	unassign_sip
	set_alert_message "${AWSSIP_ERR_OSCHKIP_MSG}"
	echo "failed to check ip address for os." 1>&2
	echo "clpawssip start script has failed." 1>&2
	exit ${AWSSIP_ERR_OSCHKIP_CODE}
fi


##
# end
##
echo "clpawssip start script has succeeded."

exit ${AWSSIP_SUCCESS_CODE}

