#!/bin/bash
# shellcheck disable=SC1091,SC2030,SC2031,SC2086
#
##
# clpawseip-watch.sh
##
# [coding：utf-8]
##


#
# グローバル変数定義
#
declare -g awsCmdLine


#
# モジュール読み込み
#
ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"


#
# 環境変数
#
export AWS_CONFIG_FILE="/root/.aws/config"
export AWS_SHARED_CREDENTIALS_FILE="/root/.aws/credentials"
export AWS_DEFAULT_OUTPUT="text"

export PATH="${PATH}:/usr/local/bin"


#
# スクリプト終了コード (= アラートイベントID)
#
# エラー
Success=0										# succeed
ErrorAWSFailed=50								# failed in the awscli command
ErrorAWSTimeout=51 								# Timeout occurred.
ErrorCheckEip=52								# failed to search Elastic IP
# 警告
WarnOffset=100
WarnAWSFailed=$((ErrorAWSFailed + WarnOffset))
WarnAWSTimeout=$((ErrorAWSTimeout + WarnOffset))
WarnNotExistAWSCmd=153							# failed to search aws cli path
WarnGetOcfEnvVariableFailed=154					# failed to get the ocf environment variable
WarnSetAWSCliEnvVariableFailed=155				# failed to set the aws cli environment variable
WarnGetAWSAddingOptionsFailed=156				# failed to get aws cli command line options
#WarnInternal = 179								# internal error


#
# 標準出力に文字列を出力する
#
function WriteStdOut {
	local message=$1
	# printf "[STDOUT] %04d: %s\n" "${BASH_LINENO[0]}" "$message"
	echo "$message"
}


#
# 標準エラー出力に文字列を出力する
#
function WriteStdErr {
	local message=$1
	# printf "[STDERR] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
	echo "[STDERR] $message" >&2
}


#
# アラートログ出力文字列を共有メモリへ設定する
#
function SetAlertMessage {
	alertMessage=$1
	monType="awseipw"
	monName=$CLP_RESOURCENAME
	cmdLine="clpshmrmset --descript --mon -t $monType -n '$monName' -m '$alertMessage'"
	result=$(eval "$cmdLine" 2>&1)
	shmRmSetExitCode=$?
	if [[ $shmRmSetExitCode -ne 0 ]]; then
		WriteStdOut "[CommandLine] $cmdLine"
		WriteStdErr "The 'clpshmrmset' command failed. ($shmRmSetExitCode)"
		WriteStdErr "$result"
		# 処理継続
	fi
}


#
# eipAllocationIdを取得
#
eipAllocationId="$(clpcfget -g /root/resource/awseip@"${TargetResource}"/parameters/allocid -p awseip)"
if [[ -z $eipAllocationId ]]; then
	SetAlertMessage "Failed to obtain the setting value."
	WriteStdErr "failed to get Elastic IP address."
	WriteStdErr "clpawseip watch script has failed."
	exit $WarnGetOcfEnvVariableFailed
fi


#
# アラートログ出力メッセージ
#
# (*1) {0}には、AWS CLIメッセージからエラー原因を抽出して設定
#	  抽出不可の場合は「The AWS CLI command was failed.」
#
declare -A AlertMessageTable=(
	[$Success]="Command succeeded."
	[$ErrorAWSFailed]="The AWS CLI command failed."
	[$ErrorAWSTimeout]="Timeout occurred."
	[$ErrorCheckEip]="The EIP address does not exist.(ENI ALLOCATION ID=$eipAllocationId)"
	[$WarnAWSFailed]="The AWS CLI command failed."
	[$WarnAWSTimeout]="Timeout occurred."
	[$WarnNotExistAWSCmd]="The AWS CLI command was not found."
	[$WarnGetOcfEnvVariableFailed]="Failed to obtain the setting value."
	[$WarnSetAWSCliEnvVariableFailed]="Failed to obtain the environment variables."
	[$WarnGetAWSAddingOptionsFailed]="Failed to obtain the AWS CLI command line options."
	# [$ErrorInternal]="Initialize error occurred."
)


#
# アラートログ出力用の共通メモリ最大領域
#
ShmMaxSize=$((128 - 1))


#
# 共通モジュール出力フラグ
#
EnableUtilLogging=0
DisableUtilLogging=1


#
# AWS CLI コマンド応答取得失敗時動作
#
NoRecoveryNoWarn=0
NoRecoveryWarn=1
Recovery=2


#
# [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
# ※未設定の場合は、clpaws_setting.conf を環境変数に設定
#
function SetAWSEnvironmentVariable {
	awsSettingConf="${ScriptRoot}/clpaws_setting.conf"
	clpcloudutil_env_init $EnableUtilLogging "$awsSettingConf"
}


#
# [クラウド: AWS CLI コマンドラインオプション] の取得
#
function GetAWSAddingOptions {
	awsService="ec2"
	clpcloudutil_awscli_cmdopt $DisableUtilLogging $awsService
}


#
# [AWS CLI コマンド応答取得失敗時動作] に応じた終了コードを取得する
#
function ShiftStatusOnAWSError {
	# (Note:)
	#	[回復動作を実行しない(警告を表示しない)] 場合は、
	#	上位モジュールへ「正常」で返却するため
	#	CLI実行権限不足などのケースを含めた、すべてのケースでアラート通知しません
	exitError=$1
	exitWarn=$2
	case $recoveryActOnAwsErr in
		"$NoRecoveryNoWarn") exitCode=$Success ;;
		"$NoRecoveryWarn") exitCode=$exitWarn ;;
		"$Recovery") exitCode=$exitError ;;
		*)
			WriteStdErr "Unknown recoveryActOnAwsErr: '$recoveryActOnAwsErr'"
			exitCode=$exitError ;;
	esac
	echo "$exitCode"
}


function CmdLineMake() {
	# AWS CLIの実行コマンドライン設定
	query="Addresses[].NetworkInterfaceId"
	awsCmdLine="aws ec2 describe-addresses --allocation-ids $eipAllocationId --query $query"
	if [[ -n $awsAddingOptions ]]; then
		awsCmdLine="$awsCmdLine $awsAddingOptions"
	fi
	WriteStdOut "$awsCmdLine"
}


#
# AWS CLIの実行
#
function CliExec() {
	# ※シグナル指定した上でタイムアウト発生時の終了コードは、128 + SIGNAL
	exitCodeWithTimeout=$((128 + 9))
	# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
	eval "$(eval "timeout -s SIGKILL $awsTimeoutSec $awsCmdLine" \
				2> >(awsResultErr=$(cat); declare -p awsResultErr) \
				1> >(awsResult=$(cat); declare -p awsResult); \
				awsExitCode=$?; declare -p awsExitCode)"
	if [[ $awsExitCode -eq $exitCodeWithTimeout ]]; then
		# AWS CLIタイムアウト
		ShiftStatusOnAWSError $ErrorAWSTimeout $WarnAWSTimeout
		if [[ $exitCode -eq $Success ]]; then
			exit $Success
		elif [[ $exitCode -eq $WarnAWSTimeout ]]; then
			SetAlertMessage "${AlertMessageTable[$WarnAWSTimeout]}"
			WriteStdErr "The AWS CLI command timed out. (awsTimeoutSec:$awsTimeoutSec)"
			exit $WarnAWSTimeout
		else
			SetAlertMessage "${AlertMessageTable[$ErrorAWSTimeout]}"
			WriteStdErr "The AWS CLI command timed out. (awsTimeoutSec:$awsTimeoutSec)"
			exit $ErrorAWSTimeout
		fi
	fi
	if [[ $awsExitCode -ne 0 ]]; then
		# AWS CLI異常終了
		WriteStdErr "The 'aws' command failed. ($awsExitCode)"
		awsResultErr=$(echo "$awsResultErr" | TrimWhiteSpace)
		WriteStdErr "$awsResultErr"
		if cause=$(ExtractAWSErrorCause "$awsResultErr" "$ShmMaxSize" 2>&1); then
			ShiftStatusOnAWSError $ErrorAWSFailed $WarnAWSFailed
			if [[ $exitCode -eq $Success ]]; then
				exit $Success
			elif [[ $exitCode -eq $WarnAWSFailed ]]; then
				SetAlertMessage "$cause"
				exit $WarnAWSFailed
			else
				SetAlertMessage "$cause"
				exit $ErrorAWSFailed
			fi
		fi
		ShiftStatusOnAWSError $ErrorAWSFailed $WarnAWSFailed
		if [[ $exitCode -eq $Success ]]; then
			exit $Success
		elif [[ $exitCode -eq $WarnAWSFailed ]]; then
			SetAlertMessage "${AlertMessageTable[$WarnAWSFailed]}"
			exit $WarnAWSFailed
		else
			SetAlertMessage "${AlertMessageTable[$ErrorAWSFailed]}"
			exit $ErrorAWSFailed
		fi
	fi
	WriteStdOut "$awsResult"
}




#
# clpawseip　開始
#
WriteStdOut "clpawseip watch script start."


#
# AWS CLIインストール確認 (コマンドの存在確認)
#
if ! command -v aws >/dev/null; then
	SetAlertMessage "${AlertMessageTable[$WarnNotExistAWSCmd]}"
	WriteStdErr "The AWS CLI command was not found."
	exit $WarnNotExistAWSCmd
fi


#
# [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
# ※未設定の場合は、clpaws_setting.conf を環境変数に設定
#
if ! SetAWSEnvironmentVariable; then
	SetAlertMessage "${AlertMessageTable[$WarnSetAWSCliEnvVariableFailed]}"
	WriteStdErr "An error occurred while setting AWS environment variables."
	exit $WarnSetAWSCliEnvVariableFailed
fi


#
# [クラウド: AWS CLI コマンドラインオプション] の取得
#
if ! awsAddingOptions=$(GetAWSAddingOptions); then
	SetAlertMessage "${AlertMessageTable[$WarnGetAWSAddingOptionsFailed]}"
	WriteStdErr "An error occurred while retrieving AWS additional options."
	exit $WarnGetAWSAddingOptionsFailed
fi


#
# ocf 環境変数取得
#
TargetResource=$CLP_OCF_PARAM1			# ターゲットリソース名
recoveryActOnAwsErr=$CLP_OCF_PARAM2		# AWS CLI コマンド応答取得失敗時動作
monTimeoutSec=$CLP_OCF_PARAM3			# 監視タイムアウト
execScriptMargin=$CLP_OCF_PARAM4		# モニタリソースタイムアウトマージン
monTimeoutMargin=$CLP_OCF_PARAM5		# AWS CLI タイムアウトマージン
if [[ -z "$TargetResource" || -z "$recoveryActOnAwsErr" || -z "$monTimeoutSec" || \
	-z "$execScriptMargin" || -z "$monTimeoutMargin" ]]; then
	SetAlertMessage "${AlertMessageTable[$WarnGetOcfEnvVariableFailed]}"
	WriteStdErr "failed to get the ocf environment variable."
	WriteStdErr "clpawseip watch script has failed."
	exit $WarnGetOcfEnvVariableFailed
fi


#
# サーバ名取得
#
srvFullName=$(uname -n)
srvName=${srvFullName%%.*}


#
# eniidを取得
#
eniId="$(clpcfget -g /root/resource/awseip@"${TargetResource}"/parameters/eniid -p awseip -s "${srvName}")"
if [[ -z $eniId ]]; then
	SetAlertMessage "${AlertMessageTable[$WarnGetOcfEnvVariableFailed]}"
	WriteStdErr "failed to get interfaceID."
	WriteStdErr "clpawseip watch script has failed."
	exit $WarnGetOcfEnvVariableFailed
fi


#
# タイムアウト倍率の取得
#
defaultToRatio=1
toRatioResult=$(clptoratio -s 2>&1)
toRatioExitCode=$?
if [[ $toRatioExitCode -eq 0 ]]; then
	# 出力結果(present toratio : <倍率>)から<倍率>のみを抽出
	toRatio=$(echo "$toRatioResult" | awk -F':' '{print $2}' | TrimWhiteSpace)
	if [[ ! $toRatio =~ ^[0-9]+$ ]]; then
		WriteStdErr "toRatio is not a number. ($toRatio)"
		WriteStdOut "Use the default timeout ratio. ($defaultToRatio)"
		toRatio=$defaultToRatio
	fi
else
	WriteStdErr "The 'clptoratio' comand failed. ($toRatioExitCode)"
	WriteStdErr "$toRatioResult"
	WriteStdOut "Use the default timeout ratio. ($defaultToRatio)"
	toRatio=$defaultToRatio
fi


#
# AWS CLIタイムアウトの取得
#
lowerLimitSec=3
awsTimeoutSec=$((monTimeoutSec * toRatio - monTimeoutMargin - execScriptMargin))
if [[ $awsTimeoutSec -lt $lowerLimitSec ]]; then
	awsTimeoutSec=$lowerLimitSec
fi
WriteStdOut "awsTimeoutSec: $awsTimeoutSec (ration:$toRatio)"


#
# EIP関連の情報を取得する
#
CmdLineMake
awsResult=$(CliExec)
exitcode=$?
if [[ $exitcode -ne 0 ]]; then
	exit $exitcode
fi
# EIPが既に関連付けされている場合は正常終了
if [[ $awsResult == "$eniId" ]]; then
	WriteStdOut "clpawseip watch script has succeeded."
	exit $Success
fi


# ここまで来たらエラー処理
SetAlertMessage "${AlertMessageTable[$ErrorCheckEip]}"
WriteStdErr "The EIP address does not exist."
WriteStdErr "clpawseip watch script has failed."
exit $ErrorCheckEip
