#!/bin/bash
#
##
# clpawsdns-watch.sh
##
# [coding：utf-8]
##
# shellcheck disable=SC1091,SC2030,SC2031,SC2086,SC2116


##
# 関数定義
##

# アラートログ文字列を共有メモリへ設定する
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --mon -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_EXIT_CODE=$?
	if [ ${SHMRMSET_EXIT_CODE} -ne 0 ]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_EXIT_CODE})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# Processing Continuation
	fi
}


# [AWS CLI コマンド応答取得失敗時動作] に応じたコードを取得する
function shift_status_on_awscli_err {
	# (Note:)
	#	[回復動作を実行しない(警告を表示しない)] 場合は、
	#	上位モジュールへ「正常」で返却するため
	#	CLI実行権限不足などのケースを含めた、すべてのケースでアラート通知しません
	EXIT_ERR=$1
	EXIT_WARN=$2
	case ${MON_MODE} in
		"${NO_RECOVERY_NO_WARN}") echo ${AWSDNSW_SUCCESS_CODE} ;;
		"${NO_RECOVERY_WARN}") echo "${EXIT_WARN}" ;;
		"${RECOVERY}") echo "${EXIT_ERR}" ;;
		*)
			echo "unknown MON_MODE(${MON_MODE})" 1>&2
			echo "${EXIT_ERR}" ;;
	esac
}


# AWS CLI エラー処理
function awscli_err_handle {
	DETAIL=$1
	AWSCLI_ALTMSG=$(ExtractAWSErrorCause "${DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_RET=$?
	if [[ ${ERRCAUSE_RET} -ne 0 ]]
	then
		AWSCLI_ALTMSG="${AWSDNSW_ERR_WARN_AWSCLI_MSG}"
	fi
	set_alert_message "${AWSCLI_ALTMSG}"
}


# 0以上の整数か判定する
function is_non_negative_number {
	NUM=$1
	if [[ $NUM =~ ^[0-9]+$ && $NUM -ge 0 ]]
	then
		return 0
	else
		return 1
	fi
}


# 名前解決確認処理
function check_dns {
	#名前解決を実行
	HOSTCMD_RET=$(host "${RECORD_NAME}")
	HOSTCMD_EXIT_CODE=$?
	if [[ ${HOSTCMD_EXIT_CODE} -ne 0 ]]
	then
		set_alert_message "${AWSDNSW_ERR_CHKDNS_MSG}"
		echo "${HOSTCMD_RET}" 1>&2
		echo "failed to name resolution.RECORD_NAME:(${RECORD_NAME})" 1>&2
		echo "clpawsdns watch script has failed." 1>&2
		exit ${AWSDNSW_ERR_CHKDNS_CODE}
	fi
	# 名前解決で取得したIPアドレスの確認
	RESOLVED_IP=$(echo ${HOSTCMD_RET} | awk '{print $4}')
	if [[ "${RESOLVED_IP}" != "${RECORD_IP}" ]]
	then
		set_alert_message "${AWSDNSW_ERR_NOTMATCH_CHKDNSIP_MSG}"
		echo "unexpected ip address which is resolved the record set name ${RECORD_NAME} from the DNS resolver. (RESOLVED_IP:${RESOLVED_IP})" 1>&2
		echo "clpawsdns watch script has failed." 1>&2
		exit ${AWSDNSW_ERR_NOTMATCH_CHKDNSIP_CODE}
	fi
	echo "check dns succeeded."
}


##
# 処理開始
##

echo "clpawsdns watch script start."


##
# clpcloudutilの呼び出し
##
SCRIPT_ROOT=$(dirname "$(realpath "$0")")
. "${SCRIPT_ROOT}/../common/clpcloudutil.sh"

# clpcloudutil出力フラグ
#ENABLE_UTIL_LOG=0
DISABLE_UTIL_LOG=1


##
# 返値
##
#成功
AWSDNSW_SUCCESS_CODE=0						# 成功
#異常
AWSDNSW_ERR_AWSCLI_CODE=50					# AWS CLI 失敗
AWSDNSW_ERR_AWSCLITO_CODE=51				# AWS CLI タイムアウト
AWSDNSW_ERR_NOTEXIST_REC_CODE=52			# レコードセットが存在しない
AWSDNSW_ERR_NOTMATCH_RECIP_CODE=53			# レコードセットのIPアドレスが不正
AWSDNSW_ERR_CHKDNS_CODE=54					# 名前解決 失敗
AWSDNSW_ERR_NOTMATCH_CHKDNSIP_CODE=55		# 名前解決で取得したIPアドレスが不正
#警告
WARN_OFFSET=100																# 警告オフセット値
AWSDNSW_WARN_AWSCLI_CODE=$((AWSDNSW_ERR_AWSCLI_CODE + WARN_OFFSET))			# AWS CLI 失敗(150)
AWSDNSW_WARN_AWSCLITO_CODE=$((AWSDNSW_ERR_AWSCLITO_CODE + WARN_OFFSET))		# AWS CLI タイムアウト(151)
AWSDNSW_WARN_SCHAWSCLI_CODE=156												# AWS CLI が存在しない
AWSDNSW_WARN_GETOCFENV_CODE=157												# OCF環境変数の取得に失敗
AWSDNSW_WARN_SETENV_CODE=158												# 環境変数の取得に失敗
AWSDNSW_WARN_GETCMDOPT_CODE=159												# AWS CLI コマンドラインオプションの取得に失敗
AWSDNSW_WARN_GET_HOSTZONE_ID_CODE=160										# ホストゾーンIDの取得に失敗
AWSDNSW_WARN_GET_RECORD_NAME_CODE=161										# リソースレコードセット名の取得に失敗
AWSDNSW_WARN_GET_RECORD_IP_CODE=162											# IPアドレスの取得に失敗
AWSDNSW_WARN_SCH_HOSTCMD_CODE=163											# host コマンドが存在しない


##
# エラーメッセージ
##
#異常&警告
AWSDNSW_ERR_WARN_AWSCLI_MSG="The AWS CLI command failed."
AWSDNSW_ERR_WARN_AWSCLITO_MSG="Timeout occurred."
#異常
AWSDNSW_ERR_NOTEXIST_REC_MSG="The resource record set in Amazon Route 53 does not exist."
AWSDNSW_ERR_NOTMATCH_RECIP_MSG="IP address different from the setting is registered in the resource record set of Amazon Route 53."
AWSDNSW_ERR_CHKDNS_MSG="Failed to resolve domain name."
AWSDNSW_ERR_NOTMATCH_CHKDNSIP_MSG="IP address which is resolved domain name from the DNS resolver is different from the setting."
#警告
AWSDNSW_WARN_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSDNSW_WARN_GETOCFENV_MSG="Failed to obtain the setting value."
AWSDNSW_WARN_SETENV_MSG="Failed to obtain the environment variables."
AWSDNSW_WARN_GETCMDOPT_MSG="Failed to obtain the AWS CLI command line options."
AWSDNSW_WARN_GET_HOSTZONE_ID_MSG="Failed to obtain the Hosted Zone ID."
AWSDNSW_WARN_GET_RECORD_NAME_MSG="Failed to obtain the Resource Record Set Name."
AWSDNSW_WARN_GET_RECORD_IP_MSG="Failed to obtain the IP Address."
AWSDNSW_WARN_SCH_HOSTCMD_MSG="The host command is not found. Monitoring based on name resolution will be skipped."

##
# シグナル指定した上でタイムアウト発生時の終了コード(128 + SIGNAL)
##
EXIT_CODE_WITH_TO=$((128 + 9))


##
# AWS CLI コマンド応答取得失敗時動作
##
NO_RECOVERY_NO_WARN=0
NO_RECOVERY_WARN=1
RECOVERY=2


##
# shmrmsetコマンド関連の初期化
##
if [[ -z ${CLP_RESOURCENAME} ]]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
	# 処理継続 (継続後の処理エラー時はアラート出力不可)
fi
RESOURCE_NAME="${CLP_RESOURCENAME}"		# リソース名
RESOURCE_TYPE="awsdnsw"					# リソースタイプ
MAX_SHMRM_SIZE=$((128 - 1))				# エラーメッセージの最大文字数


##
# AWS CLIの設定
##
export PATH=$PATH:/usr/local/bin


##
# AWS CLIの存在確認
##
which aws > /dev/null 2>&1
SCHAWSCLI_EXIT_CODE=$?
if [[ ${SCHAWSCLI_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNSW_WARN_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_SCHAWSCLI_CODE}
fi


##
# AWS CLIで使用する環境変数の設定
##
clpcloudutil_env_init ${DISABLE_UTIL_LOG} "${SCRIPT_ROOT}/clpaws_setting.conf"
ENV_INIT_EXIT_CODE=$?
if [[ ${ENV_INIT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNSW_WARN_SETENV_MSG}"
	echo "Failed to obtain the environment variables." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_SETENV_CODE}
fi


##
# AWS CLIコマンドラインオプションの取得
##
RT53_CMDOPT=$(clpcloudutil_awscli_cmdopt ${DISABLE_UTIL_LOG} route53)
GET_AWSCLI_CMDOPT_EXIT_CODE=$?
if [[ ${GET_AWSCLI_CMDOPT_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNSW_WARN_GETCMDOPT_MSG}"
	echo "Failed to obtain the aws cli command line options." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_GETCMDOPT_CODE}
fi


##
# OCF環境変数の取得
##
PARAM_NUM=7
for i in $(seq 1 $PARAM_NUM); do
	CLP_OCF_PARAM_VAR="CLP_OCF_PARAM${i}"
	if [[ -z ${!CLP_OCF_PARAM_VAR} ]]
	then
		set_alert_message "${AWSDNSW_WARN_GETOCFENV_MSG}"
		echo "failed to get the ocf environment variable." 1>&2
		echo "clpawsdns watch script has failed." 1>&2
		exit ${AWSDNSW_WARN_GETOCFENV_CODE}
	fi
done
TARGET_RSC="${CLP_OCF_PARAM1}"			# 対象リソース
MON_RECORD_FLAG="${CLP_OCF_PARAM2}"		# リソースレコードセットを監視する
MON_MODE="${CLP_OCF_PARAM3}"			# AWS CLI コマンド応答取得失敗時動作
CHKDNS_FLAG="${CLP_OCF_PARAM4}"			# 名前解決確認をする
MON_TO="${CLP_OCF_PARAM5}"				# 監視タイムアウト
MON_MARGIN="${CLP_OCF_PARAM6}"			# モニタリソースタイムアウトマージン
AWSCLI_MARGIN="${CLP_OCF_PARAM7}"		# AWS CLI タイムアウトマージン
# 型チェック
for i in "MON_RECORD_FLAG" "MON_MODE" "CHKDNS_FLAG" \
			"MON_TO" "MON_MARGIN" "AWSCLI_MARGIN"
do
	if ! is_non_negative_number "${!i}"
	then
		set_alert_message "${AWSDNSW_WARN_GETOCFENV_MSG}"
		echo "'$i' contains non-negative values. (${!i})" 1>&2
		echo "clpawsdns watch script has failed." 1>&2
		exit ${AWSDNSW_WARN_GETOCFENV_CODE}
	fi
done
echo "TARGET_RSC: ${TARGET_RSC}, MON_RECORD_FLAG: ${MON_RECORD_FLAG}, MON_MODE: ${MON_MODE}, CHKDNS_FLAG: ${CHKDNS_FLAG}"


##
# リソースレコードセットの監視をスキップ
##
if [ "${MON_RECORD_FLAG}" -eq 0 ]
then
	echo "skip monitor resource record set."
	echo "clpawsdns watch script has succeeded."
	exit ${AWSDNSW_SUCCESS_CODE}
fi


##
# サーバ名の取得
##
SRV_FULLNAME=$(uname -n)
# ドメイン名の削除
SRV_NAME=${SRV_FULLNAME%%.*}


##
# 対象リソースの設定値取得
##
HOSTZONE_ID=$(clpcfget -g /root/resource/awsdns@"${TARGET_RSC}"/parameters/hostedzoneid -p awsdns)
HOSTZONE_ID_EXIT_CODE=$?
if [ ${HOSTZONE_ID_EXIT_CODE} -ne 0 ]
then
	set_alert_message "${AWSDNSW_WARN_GET_HOSTZONE_ID_MSG}"
	echo "${HOSTZONE_ID}" | head -n 1 1>&2
	echo "get hostedzone id has failed." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_GET_HOSTZONE_ID_CODE}
fi

RECORD_NAME=$(clpcfget -g /root/resource/awsdns@"${TARGET_RSC}"/parameters/recordset -p awsdns)
RECORD_NAME_EXIT_CODE=$?
if [ ${RECORD_NAME_EXIT_CODE} -ne 0 ]
then
	set_alert_message "${AWSDNSW_WARN_GET_RECORD_NAME_MSG}"
	echo "${RECORD_NAME}" | head -n 1 1>&2
	echo "get resource record set name failed." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_GET_RECORD_NAME_CODE}
fi

RECORD_IP=$(clpcfget -g /root/resource/awsdns@"${TARGET_RSC}"/parameters/ip -p awsdns -s "${SRV_NAME}")
RECORD_IP_EXIT_CODE=$?
if [ ${RECORD_IP_EXIT_CODE} -ne 0 ]
then
	set_alert_message "${AWSDNSW_WARN_GET_RECORD_IP_MSG}"
	echo "${RECORD_IP}" | head -n 1 1>&2
	echo "get hostedzone id has failed." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_GET_RECORD_IP_CODE}
fi
echo "RECORD_IP: ${RECORD_IP}"


#
# タイムアウト倍率の取得
#
DEFALT_TO_RATIO=1
TO_RATIO_RET=$(clptoratio -s 2>&1)
TO_RATIO_EXIT_CODE=$?
if [[ $TO_RATIO_EXIT_CODE -eq 0 ]]
then
	# 出力結果(present toratio : <倍率>)から<倍率>のみを抽出
	TO_RATIO=$(echo "$TO_RATIO_RET" | awk -F':' '{print $2}' | TrimWhiteSpace)
	if [[ ! $TO_RATIO =~ ^[0-9]+$ ]]
	then
		echo "TO_RATIO is not a number. ($TO_RATIO)" 1>&2
		echo "use the default timeout ratio. ($DEFALT_TO_RATIO)"
		TO_RATIO=$DEFALT_TO_RATIO
	fi
else
	echo "an error has occurred with the 'clptoratio' command. (${TO_RATIO_EXIT_CODE})" 1>&2
	echo "${TO_RATIO_RET}" | head -n 1 1>&2
	echo "use the default timeout ratio. ($DEFALT_TO_RATIO)"
	TO_RATIO=$DEFALT_TO_RATIO
fi


#
# AWS CLIタイムアウトの取得
#
LOWER_LIMIT=3
AWSCLI_TO=$((MON_TO * TO_RATIO - MON_MARGIN - AWSCLI_MARGIN))
if [[ $AWSCLI_TO -lt $LOWER_LIMIT ]]
then
	AWSCLI_TO=$LOWER_LIMIT
fi
echo "MON_TO: ${MON_TO}, TO_RATIO: ${TO_RATIO}, AWSCLI_TO: ${AWSCLI_TO}"


##
# リソースレコードセットの存在確認
##
# AWS CLIの実行コマンドライン設定
AWS_CMDLINE="aws route53 list-resource-record-sets"
AWS_CMDLINE+=" --hosted-zone-id \"${HOSTZONE_ID}\""
AWS_CMDLINE+=" --query \"ResourceRecordSets[?Name=='${RECORD_NAME}']\""
AWS_CMDLINE+=" --output text"
if [[ -n ${RT53_CMDOPT} ]]
then
	AWS_CMDLINE="${AWS_CMDLINE} ${RT53_CMDOPT}"
fi
echo "[CommandLine] ${AWS_CMDLINE}"

# AWS CLIの実行
# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
eval "$(eval "timeout -s SIGKILL ${AWSCLI_TO} ${AWS_CMDLINE}" \
			2> >(LIST_RECORD_RET_ERR=$(cat); declare -p LIST_RECORD_RET_ERR) \
			1> >(LIST_RECORD_RET=$(cat); declare -p LIST_RECORD_RET); \
			LIST_RECORD_EXIT_CODE=$?; declare -p LIST_RECORD_EXIT_CODE )"
# AWS CLIタイムアウト
if [[ ${LIST_RECORD_EXIT_CODE} -eq ${EXIT_CODE_WITH_TO} ]]
then
	echo "the aws cli command timed out. (AWSCLI_TO:${AWSCLI_TO})" 1>&2
	# AWS CLI コマンド応答取得失敗時動作の設定によって終了ステータスを変更する。
	# 設定が「回復動作を実行しない(警告を表示しない)」の場合は正常終了
	AWSCLI_TO_CODE=$(shift_status_on_awscli_err ${AWSDNSW_ERR_AWSCLITO_CODE} ${AWSDNSW_WARN_AWSCLITO_CODE})
	if [ "${AWSCLI_TO_CODE}" -eq ${AWSDNSW_SUCCESS_CODE} ]
	then
		echo "clpawsdns watch script has succeeded."
	else
		set_alert_message "${AWSDNSW_ERR_WARN_AWSCLITO_MSG}"
		echo "clpawsdns watch script has failed." 1>&2
	fi
	exit "${AWSCLI_TO_CODE}"
fi
# AWS CLI異常終了
if [[ ${LIST_RECORD_EXIT_CODE} -ne 0 ]]
then
	echo "${LIST_RECORD_RET_ERR}" 1>&2
	echo "failed to check resource record set." 1>&2
	# AWS CLI コマンド応答取得失敗時動作の設定によって終了ステータスを変更する。
	# 設定が「回復動作を実行しない(警告を表示しない)」の場合は正常終了
	AWSCLI_FAILED_CODE=$(shift_status_on_awscli_err ${AWSDNSW_ERR_AWSCLI_CODE} ${AWSDNSW_WARN_AWSCLI_CODE})
	if [ "${AWSCLI_FAILED_CODE}" -eq ${AWSDNSW_SUCCESS_CODE} ]
	then
		echo "clpawsdns watch script has succeeded."
	else
		awscli_err_handle "${LIST_RECORD_RET_ERR}"
		echo "clpawsdns watch script has failed." 1>&2
	fi
	exit "${AWSCLI_FAILED_CODE}"
fi

# レコードセットの存在確認
if [[ -z "${LIST_RECORD_RET}" ]]
then
	set_alert_message "${AWSDNSW_ERR_NOTEXIST_REC_MSG}"
	echo "the resource record set ${RECORD_NAME} in Amazon Route 53 does not exist." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_ERR_NOTEXIST_REC_CODE}
fi

# レコードセットのIPアドレス確認
LIST_RECORD_IP=$(echo ${LIST_RECORD_RET} | awk '{print $5}')
if [[ "${LIST_RECORD_IP}" != "${RECORD_IP}" ]]
then
	set_alert_message "${AWSDNSW_ERR_NOTMATCH_RECIP_MSG}"
	echo "unexpected IP address of the resource record set ${RECORD_NAME} in Amazon Route 53. (registered: ${LIST_RECORD_IP})" 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_ERR_NOTMATCH_RECIP_CODE}
fi

echo "check resource record set succeeded."


##
# host コマンドの存在確認
##
which host > /dev/null 2>&1
SCH_HOSTCMD_EXIT_CODE=$?
if [[ ${SCH_HOSTCMD_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNSW_WARN_SCH_HOSTCMD_MSG}"
	echo "failed to search host command path." 1>&2
	echo "clpawsdns watch script has failed." 1>&2
	exit ${AWSDNSW_WARN_SCH_HOSTCMD_CODE}
fi


##
# 名前解決確認処理判定
##
if [[ ${CHKDNS_FLAG} -ne 0 ]]
then
	#名前解決確認処理開始
	check_dns
fi


##
# 処理終了
##
echo "clpawsdns watch script has succeeded."
exit ${AWSDNSW_SUCCESS_CODE}
