#!/bin/bash
#
##
# clpawsdns-start.sh
##
# [coding：utf-8]
##
# shellcheck disable=SC1091,SC2030,SC2031,SC2086


##
# 関数定義
##

# アラートログ文字列を共有メモリへ設定する
function set_alert_message {
	DETAIL=$1
	RESULT=$(clpshmrmset --descript --rsc -g "${GROUP_NAME}" -t "${RESOURCE_TYPE}" -n "${RESOURCE_NAME}" -m "${DETAIL}" 2>&1)
	SHMRMSET_EXIT_CODE=$?
	if [[ ${SHMRMSET_EXIT_CODE} -ne 0 ]]
	then
		echo "an error has occurred with the 'clpshmrmset' command. (${SHMRMSET_EXIT_CODE})" 1>&2
		echo "${RESULT}" | sed -n 2p 1>&2
		# 処理継続
	fi
}


# AWS CLI エラー処理
function awscli_err_handle {
	DETAIL=$1
	echo "${DETAIL}" 1>&2
	AWSCLI_ALTMSG=$(ExtractAWSErrorCause "${DETAIL}" ${MAX_SHMRM_SIZE})
	ERRCAUSE_EXIT_CODE=$?
	if [[ ${ERRCAUSE_EXIT_CODE} -ne 0 ]]
	then
		AWSCLI_ALTMSG="${AWSDNS_ERR_AWSCLI_MSG}"
	fi
	set_alert_message "${AWSCLI_ALTMSG}"
}


# 0以上の整数か判定する
function is_non_negative_number {
	NUM=$1
	if [[ $NUM =~ ^[0-9]+$ && $NUM -ge 0 ]]
	then
		return 0
	else
		return 1
	fi
}


# AWS CLIの引数に指定するjsonコードの作成
function create_jsoncode {
	JSON_CODE=$(cat << EOS
{
	"Comment": "Modifying an A record",
	"Changes": [
		{
			"Action": "UPSERT",
			"ResourceRecordSet": {
				"Name": "${RECORD_NAME}",
				"Type": "A",
				"TTL": ${RECORD_TTL},
				"ResourceRecords": [
					{
						"Value": "${RECORD_IP}"
					}
				]
			}
		}
	]
}
EOS
	)
	echo "${JSON_CODE}"
}


# 権威DNSサーバへの伝搬待ち合わせ
function wait_for_change_request {
	WAIT_INSYNC_INITIAL_TIME=45
	WAIT_INSYNC_INTERVAL_TIME=5
	START_TIME=$(date +%s)
	sleep ${WAIT_INSYNC_INITIAL_TIME}

	# AWS CLIの実行コマンドライン設定
	AWS_CMDLINE="aws route53 get-change"
	AWS_CMDLINE+=" --id \"${UPSERT_ID}\""
	AWS_CMDLINE+=" --query ChangeInfo.Status"
	AWS_CMDLINE+=" --output text"
	if [[ -n ${RT53_CMDOPT} ]]
	then
		AWS_CMDLINE="${AWS_CMDLINE} ${RT53_CMDOPT}"
	fi
	echo "[CommandLine] ${AWS_CMDLINE}"

	while true
	do
		# AWS CLIの実行
		# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
		eval "$(eval "timeout -s SIGKILL ${AWSCLI_TO} ${AWS_CMDLINE}" \
					2> >(GETCHG_STATUS_RET_ERR=$(cat); declare -p GETCHG_STATUS_RET_ERR) \
					1> >(GETCHG_STATUS_RET=$(cat); declare -p GETCHG_STATUS_RET); \
					GETCHG_STATUS_EXIT_CODE=$?; declare -p GETCHG_STATUS_EXIT_CODE )"
		# AWS CLIタイムアウト
		if [[ ${GETCHG_STATUS_EXIT_CODE} -eq ${EXIT_CODE_WITH_TO} ]]
		then
			delete_jsonfile
			set_alert_message "${AWSDNS_ERR_AWSCLITO_MSG}"
			echo "the aws cli command timed out. (AWSCLI_TO:${AWSCLI_TO})" 1>&2
			echo "clpawsdns start script has failed." 1>&2
			exit ${AWSDNS_ERR_AWSCLITO_CODE}
		fi
		# AWS CLI異常終了
		if [[ ${GETCHG_STATUS_EXIT_CODE} -ne 0 ]]
		then
			delete_jsonfile
			awscli_err_handle "${GETCHG_STATUS_RET_ERR}"
			echo "failed to get change status." 1>&2
			echo "clpawsdns start script has failed." 1>&2
			exit ${AWSDNS_ERR_AWSCLI_CODE}
		elif [[ "${GETCHG_STATUS_RET}" = "INSYNC" ]]
		then
			echo "wait for change request succeeded."
			break
		fi
		CURRENT_TIME=$(date +%s)
		ELAPSED_TIME=$((CURRENT_TIME- START_TIME))
		if [[ ${ELAPSED_TIME} -gt ${WAIT_INSYNC_TO} ]]
		then
			echo "wait for change request timed out." 1>&2
			break
		fi
		sleep ${WAIT_INSYNC_INTERVAL_TIME}
	done
}


# レコードセットの登録/更新に使うjsonファイルの削除
function delete_jsonfile {
	rm -f "${UPSERT_JSON_FILE}"
}


##
# 処理開始
##

echo "clpawsdns start script start."


##
# clpcloudutilの呼び出し
##
SCRIPT_ROOT=$(dirname "$(realpath "$0")")
. "${SCRIPT_ROOT}/../common/clpcloudutil.sh"

# clpcloudutil出力フラグ
ENABLE_UTIL_LOG=0
# DISABLE_UTIL_LOG=1


##
# 返値
##
AWSDNS_SUCCESS_CODE=0					# 成功
AWSDNS_ERR_AWSCLI_CODE=50				# AWS CLI 失敗
AWSDNS_ERR_AWSCLITO_CODE=51				# AWS CLI タイムアウト
AWSDNS_ERR_SCHAWSCLI_CODE=52			# AWS CLI が存在しない
AWSDNS_ERR_GETOCFENV_CODE=53			# OCF環境変数の取得に失敗
AWSDNS_ERR_NOTEXIST_REC_CODE=54			# レコードセットが存在しない
AWSDNS_ERR_NOTMATCH_RECIP_CODE=55		# レコードセットのIPが不正
AWSDNS_ERR_INTERNAL_CODE=79				# 内部エラー


##
# エラーメッセージ
##
AWSDNS_ERR_AWSCLI_MSG="The AWS CLI command failed."
AWSDNS_ERR_AWSCLITO_MSG="Timeout occurred."
AWSDNS_ERR_SCHAWSCLI_MSG="The AWS CLI command is not found."
AWSDNS_ERR_GETOCFENV_MSG="Failed to obtain the setting value."
AWSDNS_ERR_NOTEXIST_REC_MSG="The resource record set in Amazon Route 53 does not exist."
AWSDNS_ERR_NOTMATCH_RECIP_MSG="IP address different from the setting is registered in the resource record set of Amazon Route 53."
AWSDNS_ERR_INTERNAL_MSG="Internal error occurred."


##
# シグナル指定した上でタイムアウト発生時の終了コード(128 + SIGNAL)
##
EXIT_CODE_WITH_TO=$((128 + 9))


##
# shmrmsetコマンド関連の初期化
##
if [[ -z ${CLP_RESOURCENAME} ]]
then
	set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
	echo "failed to obtain the resouce name." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_INTERNAL_CODE}
fi
if [[ -z ${CLP_GROUPNAME} ]]
then
	echo "failed to obtain the value required for the shmrmset command." 1>&2
	# 処理継続 (継続後の処理エラー時はアラート出力不可)
fi
GROUP_NAME="${CLP_GROUPNAME}"			# グループ名
RESOURCE_NAME="${CLP_RESOURCENAME}"		# リソース名
RESOURCE_TYPE="awsdns"					# リソースタイプ
MAX_SHMRM_SIZE=$((128 - 1))				# エラーメッセージの最大文字数


##
# AWS CLIの設定
##
export PATH=$PATH:/usr/local/bin


##
# AWS CLIの存在確認
##
which aws > /dev/null 2>&1
SCHAWSCLI_EXIT_CODE=$?
if [[ ${SCHAWSCLI_EXIT_CODE} -ne 0 ]]
then
	set_alert_message "${AWSDNS_ERR_SCHAWSCLI_MSG}"
	echo "failed to search aws cli path." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_SCHAWSCLI_CODE}
fi


##
# AWS CLIで使用する環境変数の設定
##
clpcloudutil_env_init ${ENABLE_UTIL_LOG} "${SCRIPT_ROOT}/clpaws_setting.conf"
ENV_INIT_EXIT_CODE=$?
if [[ ${ENV_INIT_EXIT_CODE} -ne 0 ]]
then
	echo "Failed to obtain the environment variables." 1>&2
fi


##
# AWS CLIコマンドラインオプションの取得
##
RT53_CMDOPT=$(clpcloudutil_awscli_cmdopt ${ENABLE_UTIL_LOG} route53)
GET_AWSCLI_CMDOPT_EXIT_CODE=$?
if [[ ${GET_AWSCLI_CMDOPT_EXIT_CODE} -ne 0 ]]
then
	echo "Failed to obtain the aws cli command line options." 1>&2
fi


##
# OCF環境変数の取得
##
PARAM_NUM=8
for i in $(seq 1 $PARAM_NUM); do
	CLP_OCF_PARAM_VAR="CLP_OCF_PARAM${i}"
	if [[ -z ${!CLP_OCF_PARAM_VAR} ]]
	then
		set_alert_message "${AWSDNS_ERR_GETOCFENV_MSG}"
		echo "failed to get the ocf environment variable." 1>&2
		echo "clpawsdns start script has failed." 1>&2
		exit ${AWSDNS_ERR_GETOCFENV_CODE}
	fi
done
HOSTZONE_ID="${CLP_OCF_PARAM1}"					# ホストゾーンID
RECORD_NAME="${CLP_OCF_PARAM2}"					# リソースレコードセット名
RECORD_IP="${CLP_OCF_PARAM3}"					# IPアドレス
RECORD_TTL="${CLP_OCF_PARAM4}"					# TTL(秒)
# RECORD_DELETE_FLAG="${CLP_OCF_PARAM5}"		# 非活性時にリソースレコードセットを削除する
AWSCLI_TO="${CLP_OCF_PARAM6}"					# AWS CLI タイムアウト(秒)
WAIT_INSYNC_FLAG="${CLP_OCF_PARAM7}"	# 権威DNSサーバへの伝搬を待機する
WAIT_INSYNC_TO="${CLP_OCF_PARAM8}"			# 権威DNSサーバ伝搬タイムアウト (秒)
# 型チェック
for i in "RECORD_TTL" "AWSCLI_TO" \
			"WAIT_INSYNC_FLAG" "WAIT_INSYNC_TO"
do
	if ! is_non_negative_number "${!i}"
	then
		set_alert_message "${AWSDNS_ERR_GETOCFENV_MSG}"
		echo "'$i' contains non-negative values. (${!i})" 1>&2
		echo "clpawsdns start script has failed." 1>&2
		exit ${AWSDNS_ERR_GETOCFENV_CODE}
	fi
done
echo "HOSTZONE_ID: ${HOSTZONE_ID}"
echo "RECORD_NAME: ${RECORD_NAME}"
echo "RECORD_IP: ${RECORD_IP}"
echo "RECORD_TTL: ${RECORD_TTL}"
echo "AWSCLI_TO: ${AWSCLI_TO}"
echo "WAIT_INSYNC_FLAG: ${WAIT_INSYNC_FLAG}"
echo "WAIT_INSYNC_TO: ${WAIT_INSYNC_TO}"


##
# <インストールパス>/work/awsdsnディレクトリを作成する
##
#パスの作成
WORK_AWSDNS_PATH="${SCRIPT_ROOT}/../../work/awsdns"
#<インストールパス>/work/awsdsnディレクトリが存在しない場合は作成
if [[ ! -d "${WORK_AWSDNS_PATH}" ]]
then
	umask 033
	mkdir "${WORK_AWSDNS_PATH}"
	MKDIR_EXIT_CODE=$?
	if [[ ${MKDIR_EXIT_CODE} -ne 0 ]]
	then
		set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
		echo "failed to make work directory." 1>&2
		echo "clpawsdns start script has failed." 1>&2
		exit ${AWSDNS_ERR_INTERNAL_CODE}
	fi
	#ディレクトリの作成を待ち合わせ
	DIR_RETRY=10
	DIR_INTERVAL=1
	DIR_CHK_FLAG=0
	for i in $(seq ${DIR_RETRY})
	do
		sleep ${DIR_INTERVAL}
		if [[ -d "${WORK_AWSDNS_PATH}" ]]
		then
			echo "make a work directory succeeded."
			DIR_CHK_FLAG=$(( DIR_CHK_FLAG + 1 ))
			break
		fi
	done
	#<インストールパス>/work/awsdsnディレクトリの作成に失敗
	if [[ ${DIR_CHK_FLAG} -eq 0 ]]
	then
		set_alert_message "${AWSDNS_ERR_INTERNAL_MSG}"
		echo "failed to make work directory." 1>&2
		echo "clpawsdns start script has failed." 1>&2
		exit ${AWSDNS_ERR_INTERNAL_CODE}
	fi
fi


##
# レコードセットの登録/更新
##
# レコードセットの登録/更新に使うjsonファイルの作成
UPSERT_JSON_FILE="${WORK_AWSDNS_PATH}/upsert_record_${RESOURCE_NAME}.json"
# リソースレコードセット名にエスケープコードを含む場合、エスケープコードをエスケープする
if [[ "${RECORD_NAME}" =~ "\\" ]]
then
	RECORD_NAME_ORG="${RECORD_NAME}"
	RECORD_NAME=${RECORD_NAME//\\/\\\\}
fi
UPSERT_JSON_CODE=$(create_jsoncode)
echo "${UPSERT_JSON_CODE}" > "${UPSERT_JSON_FILE}"
# リソースレコードセット名を元に戻す
if [[ "${RECORD_NAME}" =~ "\\" ]]
then
	RECORD_NAME="${RECORD_NAME_ORG}"
fi

# AWS CLIの実行コマンドライン設定
AWS_CMDLINE="aws route53 change-resource-record-sets"
AWS_CMDLINE+=" --hosted-zone-id \"${HOSTZONE_ID}\""
AWS_CMDLINE+=" --change-batch \"file://${UPSERT_JSON_FILE}\""
AWS_CMDLINE+=" --query ChangeInfo.Id"
AWS_CMDLINE+=" --output text"
if [[ -n ${RT53_CMDOPT} ]]
then
	AWS_CMDLINE="${AWS_CMDLINE} ${RT53_CMDOPT}"
fi
echo "[CommandLine] ${AWS_CMDLINE}"

# AWS CLIの実行
# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
eval "$(eval "timeout -s SIGKILL ${AWSCLI_TO} ${AWS_CMDLINE}" \
			2> >(UPSERT_RECORD_RET_ERR=$(cat); declare -p UPSERT_RECORD_RET_ERR) \
			1> >(UPSERT_RECORD_RET=$(cat); declare -p UPSERT_RECORD_RET); \
			UPSERT_RECORD_EXIT_CODE=$?; declare -p UPSERT_RECORD_EXIT_CODE )"
# AWS CLIタイムアウト
if [[ ${UPSERT_RECORD_EXIT_CODE} -eq ${EXIT_CODE_WITH_TO} ]]
then
	set_alert_message "${AWSDNS_ERR_AWSCLITO_MSG}"
	echo "the aws cli command timed out. (AWSCLI_TO:${AWSCLI_TO})" 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_AWSCLITO_CODE}
fi
# AWS CLI異常終了
if [[ ${UPSERT_RECORD_EXIT_CODE} -ne 0 ]]
then
	delete_jsonfile
	awscli_err_handle "${UPSERT_RECORD_RET_ERR}"
	echo "failed to upsert the resource record set." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_AWSCLI_CODE}
fi
UPSERT_ID=$UPSERT_RECORD_RET
echo "upsert the resource record set succeeded."


##
# リソースレコードセットの存在確認
##
# AWS CLIの実行コマンドライン設定
AWS_CMDLINE="aws route53 list-resource-record-sets"
AWS_CMDLINE+=" --hosted-zone-id \"${HOSTZONE_ID}\""
AWS_CMDLINE+=" --query \"ResourceRecordSets[?Name=='${RECORD_NAME}']\""
AWS_CMDLINE+=" --output text"
if [[ -n ${RT53_CMDOPT} ]]
then
	AWS_CMDLINE="${AWS_CMDLINE} ${RT53_CMDOPT}"
fi
echo "[CommandLine] ${AWS_CMDLINE}"

# AWS CLIの実行
# サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
eval "$(eval "timeout -s SIGKILL ${AWSCLI_TO} ${AWS_CMDLINE}" \
			2> >(LIST_RECORD_RET_ERR=$(cat); declare -p LIST_RECORD_RET_ERR) \
			1> >(LIST_RECORD_RET=$(cat); declare -p LIST_RECORD_RET); \
			LIST_RECORD_EXIT_CODE=$?; declare -p LIST_RECORD_EXIT_CODE )"
# AWS CLIタイムアウト
if [[ ${LIST_RECORD_EXIT_CODE} -eq ${EXIT_CODE_WITH_TO} ]]
then
	delete_jsonfile
	set_alert_message "${AWSDNS_ERR_AWSCLITO_MSG}"
	echo "the aws cli command timed out. (AWSCLI_TO:${AWSCLI_TO})" 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_AWSCLITO_CODE}
fi
# AWS CLI異常終了
if [[ ${LIST_RECORD_EXIT_CODE} -ne 0 ]]
then
	delete_jsonfile
	awscli_err_handle "${LIST_RECORD_RET_ERR}"
	echo "failed to check resource record set." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_AWSCLI_CODE}
fi

# レコードセットの存在確認
if [[ -z "${LIST_RECORD_RET}" ]]
then
	delete_jsonfile
	set_alert_message "${AWSDNS_ERR_NOTEXIST_REC_MSG}"
	echo "the resource record set ${RECORD_NAME} in Amazon Route 53 does not exist." 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_NOTEXIST_REC_CODE}
fi

# レコードセットのIPアドレス確認
LIST_RECORD_IP=$(echo ${LIST_RECORD_RET} | awk '{print $5}')
if [[ "${LIST_RECORD_IP}" != "${RECORD_IP}" ]]
then
	delete_jsonfile
	set_alert_message "${AWSDNS_ERR_NOTMATCH_RECIP_MSG}"
	echo "unexpected IP address of the resource record set ${RECORD_NAME} in Amazon Route 53. (registered: ${LIST_RECORD_IP})" 1>&2
	echo "clpawsdns start script has failed." 1>&2
	exit ${AWSDNS_ERR_NOTMATCH_RECIP_CODE}
fi

echo "check resource record set succeeded."


##
# 権威DNSサーバへの伝搬の待ち合わせ判定
##
if [[ ${WAIT_INSYNC_FLAG} -ne 0 ]]
then
	#待ち合わせ開始
	wait_for_change_request
fi


##
# 処理終了
##

# レコードセットの登録/更新に使うjsonファイルの削除
delete_jsonfile

echo "clpawsdns start script has succeeded."
exit ${AWSDNS_SUCCESS_CODE}
