#!/bin/bash
# shellcheck disable=SC1091,SC2030,SC2031

# ==========================================================
# clpawsaz-watch.sh
# -----------------
# [AWS Availability Zone 監視スクリプト]
# 自サーバが所属するAZの健全性を監視します
# ==========================================================

#
# デバッグ
#
DebugMode=false


#
# モジュール読み込み
#
ScriptRoot=$(dirname "$(realpath "$0")")
. "${ScriptRoot}/../common/clpcloudutil.sh"


#
# 環境変数
#
export AWS_CONFIG_FILE="/root/.aws/config"
export AWS_SHARED_CREDENTIALS_FILE="/root/.aws/credentials"
export AWS_DEFAULT_OUTPUT="text"

export PATH="${PATH}:/usr/local/bin"


#
# スクリプト終了コード (= アラートイベントID)
#
Success=0                                           # [正常](0)
ErrorAWSFailed=50                                   # [異常] AWS CLI失敗(50)
ErrorAWSTimeout=51                                  # [異常] AWS CLIタイムアウト(51)
ErrorAZStatus=52                                    # [異常] AZのステータス異常(52)
# ErrorInternal=79                                    # [異常] 内部エラー(79)
WarnOffset=100                                      # [警告オフセット値](100)
WarnAWSFailed=$((ErrorAWSFailed + WarnOffset))      # [警告] AWS CLIタイムアウト(150)
WarnAWSTimeout=$((ErrorAWSTimeout + WarnOffset))    # [警告] AWS CLI失敗(151)
WarnAZStatus=$((ErrorAZStatus + WarnOffset))        # [警告] AZのステータス異常(152)
WarnNotExistAWSCmd=153                              # [警告] AWS CLI未インストール(153)
WarnSetAWSEnvVariableFailed=154                     # [警告] AWS環境変数設定失敗(154)
WarnGetAWSAddingOptionsFailed=155                   # [警告] AWS CLI追加オプション取得失敗(155)
# WarnInternal=178                                    # [警告] 内部エラー(178)
WarnInitialize=179                                  # [警告] 初期化エラー(179)
# 以下は「[警告] 初期化エラー」に丸める
WarnNotSetClpVariable=$WarnInitialize               # [警告] 環境変数(CLP_*)未設定(179)
WarnInvalidClpVariable=$WarnInitialize              # [警告] 環境変数(CLP_*)不正(179)


#
# アラートログ出力メッセージ
#
# (*1) {0}には、AWS CLIメッセージからエラー原因を抽出して設定
#      抽出不可の場合は「The AWS CLI command was failed.」
#
declare -A AlertMessageTable=(
    [$Success]="Command succeeded."
    [$ErrorAWSFailed]="The AWS CLI command failed."
    [$ErrorAWSTimeout]="Timeout occurred."
    [$ErrorAZStatus]="Failed to monitor the availability zone."
    # [$ErrorInternal]="Internal error occurred."
    [$WarnAWSFailed]="The AWS CLI command failed."
    [$WarnAWSTimeout]="Timeout occurred."
    [$WarnAZStatus]="Failed to monitor the availability zone."
    [$WarnNotExistAWSCmd]="The AWS CLI command is not found."
    [$WarnSetAWSEnvVariableFailed]="Failed to obtain the environment variables."
    [$WarnGetAWSAddingOptionsFailed]="Failed to obtain the AWS CLI command line options."
    # [$WarnInternal]="Internal error occurred."
    [$WarnInitialize]="Initialize error occurred."
)
# 以下は「[警告] 初期化エラー」に丸める ※Key値は、[警告] 初期化エラーと同じ
# AlertMessage[$WarnNotSetClpVariable]="${AlertMessage[$WarnInitialize]}"
# AlertMessage[$WarnInvalidClpVariable]="${AlertMessage[$WarnInitialize]}"


#
# 上位で設定する環境変数: CLP_OCF_PARAM の数
#
ClpOcfParamCnt=9


#
# 外部コマンド正常終了
#
ExCmdSuccess=0


#
# アラートログ出力用の共通メモリ最大領域
#
ShmMaxSize=$((128 - 1))


#
# 共通モジュール出力フラグ
#
EnableUtilLogging=0
DisableUtilLogging=1


#
# AZステータス
#
Available="available"
Information="information"
Impaired="impaired"
Unavailable="unavailable"


#
# AZステータスのハンドリング
#
AZStatusHdlSuccess=0
AZStatusHdlWarn=1
AZStatusHdlError=2


#
# AWS CLI コマンド応答取得失敗時動作
#
NoRecoveryNoWarn=0
NoRecoveryWarn=1
Recovery=2


#
# アラートログ出力文字列(動的設定用)
#
UseAlertMessageTable=true
AlertMessage=""


# ----------------------------------------------------------
#
# 関数定義
#
# ----------------------------------------------------------

#
# [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
# ※未設定の場合は、clpaws_setting.conf を環境変数に設定
#
function SetAWSEnvironmentVariable {
    awsSettingConf="${ScriptRoot}/clpaws_setting.conf"
    clpcloudutil_env_init $EnableUtilLogging "$awsSettingConf"
}


#
# [クラウド: AWS CLI コマンドラインオプション] の取得
#
function GetAWSAddingOptions {
    awsService="ec2"
    clpcloudutil_awscli_cmdopt $DisableUtilLogging $awsService
}


#
# [AWS CLI コマンド応答取得失敗時動作] に応じた終了コードを取得する
#
function ShiftStatusOnAWSError {
    # (Note:)
    #    [回復動作を実行しない(警告を表示しない)] 場合は、
    #    上位モジュールへ「正常」で返却するため
    #    CLI実行権限不足などのケースを含めた、すべてのケースでアラート通知しません
    exitError=$1
    exitWarn=$2
    case $recoveryActOnAwsErr in
        "$NoRecoveryNoWarn") exitCode=$Success; status="Normal" ;;
        "$NoRecoveryWarn") exitCode=$exitWarn; status="Warn" ;;
        "$Recovery") exitCode=$exitError; status="Error" ;;
        *)
            WriteStdErr "Unknown recoveryActOnAwsErr: '$recoveryActOnAwsErr'"
            exitCode=$exitError
            status="Error" ;;
    esac
    WriteDebug "Shift status: $status"
    echo "$exitCode"
}


#
# アラートログ出力文字列を共有メモリへ設定する
#
function SetAlertMessage {
    alertMessage=$1
    monType="awsazw"
    cmdLine="clpshmrmset --descript --mon -t $monType -n \"$monName\" -m \"$alertMessage\""
    result=$(eval "$cmdLine" 2>&1)
    shmRmSetExitCode=$?
    if [[ $shmRmSetExitCode -ne $ExCmdSuccess ]]; then
        WriteStdOut "[CommandLine] $cmdLine"
        WriteStdErr "The 'clpshmrmset' command failed. ($shmRmSetExitCode)"
        WriteStdErr "$(echo "$result" | TrimWhiteSpace)"
        # 処理継続
    fi
}


#
# 標準出力に文字列を出力する
#
function WriteStdOut {
    local message=$1
    if ! $DebugMode; then
        # printf "%04d: %s\n" "${BASH_LINENO[0]}" "$message"
        echo "$message"
    else
        # printf "[STDOUT] %04d: %s\n" "${BASH_LINENO[0]}" "$message"
        echo "[STDOUT] $message"
    fi
}


#
# 標準エラー出力に文字列を出力する
#
function WriteStdErr {
    local message=$1
    if ! $DebugMode; then
        # printf "%04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
        echo "$message" >&2
    else
        # printf "[STDERR] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >&2
        echo "[STDERR] $message" >&2
    fi
}


#
# コンソールに文字列を出力する(デバッグ用)
#
function WriteDebug {
    local message=$1
    if ! $DebugMode; then
        :
    else
        # printf "[ DEBUG] %04d: %s\n" "${BASH_LINENO[0]}" "$message" >/dev/tty
        echo "[ DEBUG] $message" >/dev/tty
    fi
}


#
# 0以上の整数か判定する
#
function IsNonNegativeNumber {
    number=$1
    if [[ $number =~ ^[0-9]+$ && $number -ge 0 ]]; then
        return 0
    else
        return 1
    fi
}


#
# AWS AZモニタリソースを監視する
#
function DoMonitor {

    #
    # 上位モジュールで設定した環境変数: CLP_* の確認
    #
    if [[ -z $CLP_RESOURCENAME ]]; then
        WriteStdErr "The environment variable 'CLP_RESOURCENAME' has not been set."
        # 処理継続 (継続後の処理エラー時はアラート出力不可)
    fi
    monName=$CLP_RESOURCENAME
    WriteStdOut "monName: $monName"

    for i in $(seq 1 $ClpOcfParamCnt); do
        CLP_OCF_PARAM_VAR="CLP_OCF_PARAM${i}"
        if [[ -z ${!CLP_OCF_PARAM_VAR} ]]; then
            WriteStdErr "The environment variable '$CLP_OCF_PARAM_VAR' has not been set."
            return $WarnNotSetClpVariable
        fi
    done
    azName=$CLP_OCF_PARAM1
    recoveryActOnAwsErr=$CLP_OCF_PARAM2
    azStatusHdlAvailable=$CLP_OCF_PARAM3
    azStatusHdlInformation=$CLP_OCF_PARAM4
    azStatusHdlImpaired=$CLP_OCF_PARAM5
    azStatusHdlUnavailable=$CLP_OCF_PARAM6
    monTimeoutSec=$CLP_OCF_PARAM7
    monTimeoutMargin=$CLP_OCF_PARAM8
    execScriptMargin=$CLP_OCF_PARAM9
    # 型チェック
    for i in "recoveryActOnAwsErr" "azStatusHdlAvailable" \
             "azStatusHdlInformation" "azStatusHdlImpaired" \
             "azStatusHdlUnavailable" "monTimeoutSec" \
             "monTimeoutMargin" "execScriptMargin"; do
        if ! IsNonNegativeNumber "${!i}"; then
            WriteStdErr "'$i' contains non-negative values. (${!i})"
            return $WarnInvalidClpVariable
        fi
    done
    WriteStdOut "azName: $azName"
    WriteStdOut "recoveryActOnAwsErr: $recoveryActOnAwsErr"
    WriteDebug "azStatusHdlAvailable: $azStatusHdlAvailable"
    WriteDebug "azStatusHdlInformation: $azStatusHdlInformation"
    WriteDebug "azStatusHdlImpaired: $azStatusHdlImpaired"
    WriteDebug "azStatusHdlUnavailable: $azStatusHdlUnavailable"
    WriteStdOut "monTimeoutSec: $monTimeoutSec"
    WriteDebug "monTimeoutMargin: $monTimeoutMargin"
    WriteDebug "execScriptMargin: $execScriptMargin"


    #
    # AWS CLIインストール確認 (コマンドの存在確認)
    #
    if ! command -v aws >/dev/null; then
        WriteStdErr "The AWS CLI command was not found."
        return $WarnNotExistAWSCmd
    fi


    #
    # [クラウド: AWS関連機能実行時の環境変数] を環境変数に設定
    # ※未設定の場合は、clpaws_setting.conf を環境変数に設定
    #
    if ! SetAWSEnvironmentVariable; then
        WriteStdErr "An error occurred while setting AWS environment variables."
        return $WarnSetAWSEnvVariableFailed
    fi


    #
    # [クラウド: AWS CLI コマンドラインオプション] の取得
    #
    if ! awsAddingOptions=$(GetAWSAddingOptions); then
        WriteStdErr "An error occurred while retrieving AWS additional options."
        return $WarnGetAWSAddingOptionsFailed
    fi
    if [[ -n $awsAddingOptions ]]; then
        WriteStdOut "awsAddingOptions: $awsAddingOptions"
    fi


    #
    # タイムアウト倍率の取得
    #
    defaultToRatio=1
    toRatioResult=$(clptoratio -s 2>&1)
    toRatioExitCode=$?
    if [[ $toRatioExitCode -eq $ExCmdSuccess ]]; then
        # 出力結果(present toratio : <倍率>)から<倍率>のみを抽出
        toRatio=$(echo "$toRatioResult" | awk -F':' '{print $2}' | TrimWhiteSpace)
        if [[ ! $toRatio =~ ^[0-9]+$ ]]; then
            WriteStdErr "toRatio is not a number. ($toRatio)"
            WriteStdOut "Use the default timeout ratio. ($defaultToRatio)"
            toRatio=$defaultToRatio
        fi
    else
        WriteStdErr "The 'clptoratio' comand failed. ($toRatioExitCode)"
        WriteStdErr "$toRatioResult"
        WriteStdOut "Use the default timeout ratio. ($defaultToRatio)"
        toRatio=$defaultToRatio
    fi


    #
    # AWS CLIタイムアウトの取得
    #
    lowerLimitSec=3
    awsTimeoutSec=$((monTimeoutSec * toRatio - monTimeoutMargin - execScriptMargin))
    if [[ $awsTimeoutSec -lt $lowerLimitSec ]]; then
        awsTimeoutSec=$lowerLimitSec
    fi
    WriteStdOut "awsTimeoutSec: $awsTimeoutSec (ratio:$toRatio)"


    #
    # AWS CLIの実行コマンドライン設定
    #
    query="AvailabilityZones[].State"
    awsCmdLine="aws ec2 describe-availability-zones --zone-names \"$azName\" --query $query"
    if [[ -n $awsAddingOptions ]]; then
        awsCmdLine="$awsCmdLine $awsAddingOptions"
    fi
    if $DebugMode; then
        awsCmdLine="$awsCmdLine --debug"
    fi
    WriteStdOut "[CommandLine] $awsCmdLine"


    #
    # AWS CLIの実行
    #
    # ※シグナル指定した上でタイムアウト発生時の終了コードは、128 + SIGNAL
    exitCodeWithTimeout=$((128 + 9))
    # サブシェルで取得した標準出力・標準エラー出力・実行結果を取得
    eval "$(eval "timeout -s SIGKILL $awsTimeoutSec $awsCmdLine" \
                2> >(awsResultErr=$(cat); declare -p awsResultErr) \
                1> >(awsResult=$(cat); declare -p awsResult); \
                awsExitCode=$?; declare -p awsExitCode )"
    if [[ $awsExitCode -eq $exitCodeWithTimeout ]]; then
        # AWS CLIタイムアウト
        WriteStdErr "The AWS CLI command timed out. (awsTimeoutSec:$awsTimeoutSec)"
        return "$(ShiftStatusOnAWSError $ErrorAWSTimeout $WarnAWSTimeout)"
    fi
    if [[ $awsExitCode -ne $ExCmdSuccess ]]; then
        # AWS CLI異常終了
        WriteStdErr "The 'aws' command failed. ($awsExitCode)"
        awsResultErr=$(echo "$awsResultErr" | TrimWhiteSpace)
        WriteStdErr "$awsResultErr"
        if cause=$(ExtractAWSErrorCause "$awsResultErr" "$ShmMaxSize"); then
            AlertMessage="$cause"
            UseAlertMessageTable=false
        fi
        return "$(ShiftStatusOnAWSError $ErrorAWSFailed $WarnAWSFailed)"
    fi
    azStatus=$awsResult
    WriteStdOut "azStatus: $azStatus"


    #
    # AZステータスに応じてハンドリング
    #
    case $azStatus in
        "$Available") azStatusHdl=$azStatusHdlAvailable ;;
        "$Information") azStatusHdl=$azStatusHdlInformation ;;
        "$Impaired") azStatusHdl=$azStatusHdlImpaired ;;
        "$Unavailable") azStatusHdl=$azStatusHdlUnavailable ;;
        *)
            WriteStdErr "Unknown azStatus: '$azStatus'"
            azStatusHdl=$AZStatusHdlError ;;
    esac
    WriteStdOut "azStatusHdl: $azStatusHdl"
    case $azStatusHdl in
        "$AZStatusHdlSuccess") exitCode=$Success ;;
        "$AZStatusHdlWarn") exitCode=$WarnAZStatus ;;
        "$AZStatusHdlError") exitCode=$ErrorAZStatus ;;
        *)
            WriteStdErr "Unknown azStatusHdl: '$azStatusHdl'"
            exitCode=$ErrorAZStatus ;;
    esac
    # AZステータス異常
    if [[ $exitCode -ne $Success ]]; then
        WriteStdErr "AZ status health check has failed. ($azName)"
    fi

    # 監視処理終了
    return $exitCode
}


#
# Main (エントリーポイント)
#
function Main {
    WriteDebug "Start the monitoring process for AWS AZ monitoring resource."

    # AZモニタの監視開始
    DoMonitor
    MonExitCode=$?
    if [[ $MonExitCode -eq $Success ]]; then
        WriteStdOut "Succeeded in monitoring for AWS AZ monitoring resource."
    else
        if $UseAlertMessageTable; then
            message="${AlertMessageTable[$MonExitCode]}"
        else
            message="$AlertMessage"
        fi
        SetAlertMessage "$message"
        WriteStdErr "Failed in monitoring for AWS AZ monitoring resource. ($MonExitCode)"
    fi

    exit $MonExitCode
}


# ----------------------------------------------------------
#
# 処理開始
#
# ----------------------------------------------------------
if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then
    Main
fi
