#! /bin/sh 

#
#  usage : clpvcenterrestapiforcestop.sh <sub-command>
# 
#  sub-command :
#         stop
#         check-stop
#         monitor
#

#------- static variables -------#

readonly OPT_MODE_MONITOR="monitor"
readonly OPT_MODE_STOPSRV="stop"
readonly OPT_MODE_CHECKSTOP="check-stop"

readonly MSG_CMD_FAILED="Command failed."
readonly MSG_ERR_INVALID_OPTION="Invalid option."
readonly MSG_ERR_INVALID_PARAM="Invalid parameter."

readonly VM_STATE_POWER_ON="POWERED_ON"
readonly VM_STATE_POWER_OFF="POWERED_OFF"

readonly RET_CMD_SUCCESS=0
readonly RET_CMD_FAILED=1

#------- variables -------#

opt_cnt=0
opt_nam=""
api_session_id=""
datacenter_id=""
vm_id=""
vm_state=""

ret_execute=${RET_CMD_SUCCESS}

if [ "${CLP_CERT_IGNORE}" -eq 0 ]
then
	curl_command="curl"
else
	curl_command="curl -k"
fi

# goto MAIN_ROUTINE


####################################################################
#                                                                  #
# Common Functions                                                 #
#                                                                  #
####################################################################

#####################################################################
# name   : CREATE_SESSION
# arg#1  : (none)
# local  : ret_cmd, ret_restapi, error_type
# in     : (none)
# out    : api_session_id
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  Others : An error occurred.
# 
# description:
#  Create rest api session.
####################################################################
CREATE_SESSION()
{
	ret_cmd=0
	ret_restapi=""
	error_type=""

	#------- execute api -------
	ret_restapi=$(${curl_command} -X POST -u "${CLP_VCENTER_USER}":"${CLP_VCENTER_PASSWORD}" "https://${CLP_VCENTER_HOST}/api/session")

	ret_cmd=$?
	
	#------- check exit code -------
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "CREATE_SESSION:${MSG_CMD_FAILED} [${ret_restapi}] " 1>&2
		return ${ret_cmd}
	fi
	
	#------- check session -------
	if [ -z "$ret_restapi" ]
	then
		echo "CREATE_SESSION:${MSG_CMD_FAILED} [No session.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- get error type -------
	error_type=$(PARSE_API_RESULT "$ret_restapi" "error_type")
	
	#------- check error type -------
	if [ -n "$error_type" ]
	then
		echo "CREATE_SESSION:${MSG_CMD_FAILED} [$ret_restapi]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	api_session_id=$(echo "${ret_restapi}" | sed 's/\"//g')

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : DELETE_SESSION
# arg#1  : (none)
# local  : ret_cmd, ret_restapi, error_type
# in     : api_session_id
# out    : api_session_id
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  Others : An error occurred.
# 
# description:
#  Delete rest api session.
####################################################################
DELETE_SESSION()
{
	ret_cmd=0
	ret_restapi=""
	error_type=""

	#------- check session -------
	if [ -z "$api_session_id" ]
	then
		return ${RET_CMD_SUCCESS}
	fi

	#------- execute api -------
	ret_restapi=$(${curl_command} -X DELETE "https://${CLP_VCENTER_HOST}/api/session" -H "vmware-api-session-id: ${api_session_id}" 2>&1)

	ret_cmd=$?
	
	#------- check exit code -------
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "DELETE_SESSION:${MSG_CMD_FAILED} [${ret_restapi}]" 1>&2
		return ${ret_cmd}
	fi

	api_session_id=""
	
	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : GET_DATACENTER_ID
# arg#1  : (none)
# local  : ret_cmd, ret_restapi
# in     : api_session_id
# out    : datacenter_id
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  Others : An error occurred.
# 
# description:
#  Get datacenter id.
####################################################################
GET_DATACENTER_ID()
{
	ret_cmd=0
	ret_restapi=""
	replace=$(echo "$CLP_DATACENTER_NAME" | sed "s/&/%26/g")

	#------- check session -------
	if [ -z "${api_session_id}" ]
	then
		echo "GET_DATACENTER_ID:${MSG_CMD_FAILED} [No session.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- execute api -------
	ret_restapi=$(${curl_command} -X GET "https://${CLP_VCENTER_HOST}/api/vcenter/datacenter?names=${replace}" -H "vmware-api-session-id: ${api_session_id}")
	
	ret_cmd=$?
	
	#------- check exit code -------
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "GET_DATACENTER_ID:${MSG_CMD_FAILED} [${ret_restapi}]" 1>&2
		return ${ret_cmd}
	fi

	datacenter_id=$(PARSE_API_RESULT "$ret_restapi" "datacenter")

	#------- check datacenter id -------
	if [ -z "${datacenter_id}" ]
	then
		echo "GET_DATACENTER_ID:${MSG_CMD_FAILED} [No datacenter. ${ret_restapi}]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : GET_VM_STATE
# arg#1  : (none)
# local  : ret_cmd, ret_restapi
# in     : api_session_id, datacenter_id
# out    : vm_id, vm_state
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  Others : An error occurred.
# 
# description:
#  Get virtual machine state.
####################################################################
GET_VM_STATE()
{
	ret_cmd=0
	ret_restapi=""
	replace=$(echo "$CLP_VMNAME" | sed "s/&/%26/g")
	
	#------- check session -------
	if [ -z "$api_session_id" ]
	then
		echo "GET_VM_STATE:${MSG_CMD_FAILED} [No session.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- check session -------
	if [ -z "$datacenter_id" ]
	then
		echo "GET_VM_STATE:${MSG_CMD_FAILED} [No datacenter.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- execute api -------
	ret_restapi=$(${curl_command} -X GET "https://${CLP_VCENTER_HOST}/api/vcenter/vm?names=${replace}&datacenters=${datacenter_id}" -H "vmware-api-session-id: ${api_session_id}")
	
	ret_cmd=$?
	
	#------- check exit code -------
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "GET_VM_STATE:${MSG_CMD_FAILED} [${ret_restapi}]" 1>&2
		return ${ret_cmd}
	fi

	vm_id=$(PARSE_API_RESULT "$ret_restapi" "vm")
	vm_state=$(PARSE_API_RESULT "$ret_restapi" "power_state")

	#------- check vm id -------
	if [ -z "$vm_id" ]
	then
		echo "GET_VM_STATE:${MSG_CMD_FAILED} [No vm id. ${ret_restapi}]" 1>&2
		return ${RET_CMD_FAILED}
	fi
	#------- check vm state -------
	if [ -z "$vm_state" ]
	then
		echo "GET_VM_STATE:${MSG_CMD_FAILED} [No vm state. ${ret_restapi}]" 1>&2
		return ${RET_CMD_FAILED}
	fi
	
	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : EXEC_FORCE_STOP
# arg#1  : (none)
# local  : ret_cmd, ret_restapi
# in     : api_session_id, vm_id
# out    : (none)
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  Others : An error occurred.
# 
# description:
#  Execute force stop.
####################################################################
EXEC_FORCE_STOP()
{
	ret_cmd=0
	ret_restapi=""

	#------- check session -------
	if [ -z "${api_session_id}" ]
	then
		echo "EXEC_FORCE_STOP:${MSG_CMD_FAILED} [No session.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- check vm_id -------
	if [ -z "${vm_id}" ]
	then
		echo "EXEC_FORCE_STOP:${MSG_CMD_FAILED} [No vm id.]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- execute api -------
	ret_restapi=$(curl -k -X POST "https://${CLP_VCENTER_HOST}/api/vcenter/vm/${vm_id}/power?action=${CLP_VCENTER_ACTION}" -H "vmware-api-session-id: ${api_session_id}")

	ret_cmd=$?
	
	#------- check exit code -------
	if [ ${ret_cmd} -ne ${RET_CMD_SUCCESS} ]
	then
		echo "EXEC_FORCE_STOP:${MSG_CMD_FAILED} [${ret_restapi}]" 1>&2
		return ${ret_cmd}
	fi
	
	#------- check result -------
	if [ -n "${ret_restapi}" ]
	then
		echo "EXEC_FORCE_STOP:${MSG_CMD_FAILED} [${ret_restapi}]" 1>&2
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : PARSE_API_RESULT
# arg#1  : rest api result.
# arg#2  : parameter.
# local  : ret_restapi, param, obj, data
# in     : (none)
# out    : data
# return : (only 0)
# 
# description:
#  Parse api results. 
####################################################################
PARSE_API_RESULT()
{
	ret_restapi="$1"
	param="$2"

	obj=$(echo "${ret_restapi}" | sed 's/\[]//g' | sed 's/^\[{\(.*\)}]$/\1/')
	data=$(echo "${obj}" | tr ',' '\n' | grep -e "${param}" | sed 's/^.*://' | tr -d '"')
	echo "${data}"

	return 0
}

####################################################################
#                                                                  #
# Main Routine                                                     #
#                                                                  #
####################################################################
#:MAIN_ROUTINE

#------- option check -------#

opt_cnt=0

for i in "$@"
do
	case "$i" in
	"${OPT_MODE_MONITOR}" )
		opt_nam=${OPT_MODE_MONITOR}
		opt_cnt=$((opt_cnt + 1))
		;;
	"${OPT_MODE_STOPSRV}" )
		opt_nam=${OPT_MODE_STOPSRV}
		opt_cnt=$((opt_cnt + 1))
		;;
	"${OPT_MODE_CHECKSTOP}" )
		opt_nam=${OPT_MODE_CHECKSTOP}
		opt_cnt=$((opt_cnt + 1))
		;;
	* )
		echo "${MSG_ERR_INVALID_OPTION}" 1>&2
		exit ${RET_CMD_FAILED}
		;;
	esac
done

if [ ${opt_cnt} -ne 1 ]
then
	echo "${MSG_ERR_INVALID_OPTION}" 1>&2
	exit ${RET_CMD_FAILED}
fi

#------- environmental variables check -------#
if [ "${opt_nam}" = "${OPT_MODE_STOPSRV}" ]; then
	if [ -z "$CLP_VCENTER_ACTION" ]; then
		echo "${MSG_ERR_INVALID_PARAM} [CLP_VCENTER_ACTION]" 1>&2
		exit ${RET_CMD_FAILED}
	fi
fi
if [ -z "$CLP_VCENTER_USER" ]; then
 	echo "${CLP_VCENTER_HOST} [CLP_VCENTER_HOST]" 1>&2
	exit ${RET_CMD_FAILED}
elif [ -z "$CLP_VCENTER_USER" ]; then
 	echo "${MSG_ERR_INVALID_PARAM} [CLP_VCENTER_USER]" 1>&2
	exit ${RET_CMD_FAILED}
elif [ -z "$CLP_VCENTER_PASSWORD" ]; then
 	echo "${MSG_ERR_INVALID_PARAM} [CLP_VCENTER_PASSWORD]" 1>&2
	exit ${RET_CMD_FAILED}
elif [ -z "$CLP_VMNAME" ]; then
 	echo "${MSG_ERR_INVALID_PARAM} [CLP_VMNAME]" 1>&2
	exit ${RET_CMD_FAILED}
elif [ -z "$CLP_DATACENTER_NAME" ]; then
 	echo "${MSG_ERR_INVALID_PARAM} [CLP_DATACENTER_NAME]" 1>&2
	exit ${RET_CMD_FAILED}
fi

#------- execute -------#
CREATE_SESSION
ret_execute=$?
if [ "${ret_execute}" -ne "${RET_CMD_SUCCESS}" ]; then
	exit ${ret_execute}
fi

GET_DATACENTER_ID
ret_execute=$?
if [ "${ret_execute}" -ne "${RET_CMD_SUCCESS}" ]; then
	DELETE_SESSION
	exit ${ret_execute}
fi

GET_VM_STATE
ret_execute=$?
if [ "${ret_execute}" -ne "${RET_CMD_SUCCESS}" ]; then
	DELETE_SESSION
	exit ${ret_execute}
fi

case "${opt_nam}" in
"${OPT_MODE_MONITOR}" )
	# do nothing
	;;
"${OPT_MODE_STOPSRV}" )
	if [ "${vm_state}" != "${VM_STATE_POWER_OFF}" ]; then
		EXEC_FORCE_STOP
		ret_execute=$?
	fi
	;;
"${OPT_MODE_CHECKSTOP}" )
	if [ "${vm_state}" != "${VM_STATE_POWER_OFF}" ]; then
		echo "${OPT_MODE_CHECKSTOP}: [${VM_STATE_POWER_ON}]" 1>&2
		ret_execute=${RET_CMD_FAILED}
	fi 
	;;
* )
	echo "${MSG_ERR_INVALID_OPTION}" 1>&2
	ret_execute=${RET_CMD_FAILED}
	;;
esac

DELETE_SESSION

exit ${ret_execute}