#!/bin/bash
##
#	Validation Check
##

#------------------------------------------------
#	command exit code
#------------------------------------------------
# command return code
EXITCODE_CMD_SUCCESS=0
EXITCODE_CMD_INVOKED_CANNOT_EXECUTE=126
EXITCODE_CMD_NOT_FOUND=127

# clpstat
EXITCODE_STAT_RSC_ONLINE=1		# resource online
EXITCODE_STAT_MON_NORMAL=0		# monitor normal

#------------------------------------------------
#	return code
#------------------------------------------------
RET_SUCCESS=0	# success
RET_FAIL_OPT=1	# option error

#------------------------------------------------
#	log
#------------------------------------------------
LOG_FNAME="validation"
LOG_EXTENTION="log"
LOG_INFO="INFO "
LOG_ERR="ERROR"
LOG_CMD="command :"
LOG_EXITCODE="exit code :"
LOG_CMDRESULT="command result :"

#------------------------------------------------
#	functions
#------------------------------------------------
# function return code
FUNC_RET_SUCCESS=0
FUNC_RET_FAIL=1
FUNC_RET_NONTARGET=2

##
#	Read Message File
##
declare -a msg_table_id=()
declare -a msg_table_item=()
declare -a msg_table_msg1=()
declare -a msg_table_msg2=()
already_read=false

function read_msg_file ()
{
	local i=0
	local wk_id
	local wk_item
	local wk_msg1
	local wk_msg2
	local wk_err_num
	local wk_err_id_array
	local wk_err_id
	local wk_err_id_name

	item=""
	msg1=""
	msg2=""

	if ! "${already_read}";
	then
		while read line;
		do
			if [[ "${line}" =~ ^[\s\t]*ID[\s\t]*=[\s\t]*([0-9]+)[\s\t]*$ ]];
			then
				# set message for ID, ITEM, MSG1, MSG2, ErrID

				# ID
				msg_table_id=("${msg_table_id[@]}" "${line#ID=}")
				wk_id="${line#ID=}"

				# ITEM
				read wk_item
				msg_table_item=("${msg_table_item[@]}" "${wk_item#ITEM=}")

				# MSG1
				read wk_msg1
				msg_table_msg1=("${msg_table_msg1[@]}" "${wk_msg1#MSG1=}")

				# MSG2
				read wk_msg2
				msg_table_msg2=("${msg_table_msg2[@]}" "${wk_msg2#MSG2=}")

				# ERR_ID
				read wk_err_id_array
				wk_err_id_array="${wk_err_id_array#ERR_ID=}"

				wk_err_num=1
				#  区切り文字を","のみに設定
				IFS=','
				for wk_err_id in ${wk_err_id_array};
				do
					# variable name : err_id_<id>_<number> (e.g. $err_id_1000_1)
					# value : Error ID (e.g. 1001)
					wk_err_id_name="err_id_${wk_id}_${wk_err_num}"
					eval ${wk_err_id_name}=${wk_err_id}
					wk_err_num=$(( wk_err_num + 1 ))
				done
				IFS=$change_IFS
			fi
		done < ${msgfpath}

		if [ `echo ${#msg_table_id[@]}` -gt 0 ];
		then
			already_read=true
		fi
	fi

	for wk_id in ${msg_table_id[@]};
	do
		if [ "${id}" = "${wk_id}" ];
		then
			# get messages for ID from message table
			item=${msg_table_item[i]}
			msg1=${msg_table_msg1[i]}
			msg2=${msg_table_msg2[i]}
			break
		fi
		i=$(( i + 1 ))
	done
}

##
#	set error id
#	 arg1 : error number
##
function set_err_id ()
{
	local err_num=$1
	err_id_name="err_id_${id}_${err_num}"
	err_id=$(eval "echo \"\${${err_id_name}}\"")
}

##
#	clpcfget command execution
#	 arg1 : option(g:string, e:enum)
#	 arg2 : policy type
#	 arg3 : server name
##
CFGET_SUCCESS=0		# success
CFGET_ERR_XMLEXIST=11	# xmlpath not exist

function clpcfget_exe ()
{
	local cfget_opt=$1
	local policy_type=$2
	local server_name=$3

	cfget=""
	cfget_exitcode=""
	local cmdline="clpcfget -${cfget_opt} \"${xml_path}\" -i \"${ipath}\""
	if [ "${policy_type}" != "" ];
	then
		cmdline="${cmdline} -p ${policy_type}"
	fi
	if [ "${server_name}" != "" ];
	then
		cmdline="${cmdline} -s ${server_name}"
	fi
	cfget=`eval ${cmdline}`
	cfget_exitcode=$?

	logmsg=`echo $LOG_INFO "PATH[${xml_path}], PARAM[${cfget}]"`
	clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME
}


##
#	Set Message : Could Not Check
##
function set_msg_couldnotchk ()
{
	local rep_msg=$1
	result_summary="FALSE"
	result="FALSE"
	if [ "${rep_msg}" != "" ];
	then
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rep_msg}|"`
	fi
	result_msg2=${checkerr_msg}
	if [ -z "${err_id}" ];
	then
		# unexpected error
		err_id=${unexpected_err_id}
	fi
}

##
#	Set Message : Not Check Target
##
function set_msg_notchktarget ()
{
	local rep_msg=$1
	result="-"
	if [ "${rep_msg}" != "" ];
	then
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rep_msg}|"`
	fi
	result_msg2=${notchktarget_msg}
}


##
#	Output Message : result summary [ID,Server,,Msg1,Result]
##
function output_msg_result_summary ()
{
	local wk_result=""
	case ${result_summary} in
		"TRUE" ) wk_result="PASS" ;;
		"FALSE") wk_result="FAIL" ;;
		*      ) wk_result="-"    ;;
	esac
	echo "\"${id}\",\"${server}\",\"\",\"${msg1}\",\"${wk_result}\"" >> "${output}"
}

##
#	Output Message : datetime [ID,Server,,Msg1,Datetime]
##
function output_msg_datetime ()
{
	local datetime=`date '+%Y/%m/%d %H:%M:%S'`
	echo "\"${id}\",\"${server}\",\"\",\"${msg1}\",\"${datetime}\"" >> "${output}"
}

##
#	Output Message : normal [ID,Server,Item,Msg1,Result,Msg2,ErrID]
#	 arg1 : output msg1
#	 arg2 : output msg2(error msg)
##
function output_msg ()
{
	local wk_outputmsg1=$1
	local wk_outputmsg2=$2
	local wk_result=""
	local wk_err_id=""
	case ${result} in
		"TRUE" ) wk_result="PASS"
		         wk_outputmsg2=""
		         wk_err_id=""
		         logmsg=`echo $LOG_INFO "Result : PASS"`
		         clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME ;;
		"FALSE") wk_result="FAIL"
		         wk_outputmsg2="ID: ${err_id}${newline}${wk_outputmsg2}"
		         if [ "$err_id" -eq "$unexpected_err_id" ];
		         then
		         	# unexpected error
		         	wk_err_id=""
		         else
		         	wk_err_id=${err_id}
		         fi
		         logmsg=`echo $LOG_ERR "Result : FAIL(${wk_err_id})"`
		         clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME ;;
		*      ) wk_result="-"
		         logmsg=`echo $LOG_INFO "Result : Nontarget"`
		         clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME ;;
	esac

	# remove last $newline
	wk_outputmsg2="${wk_outputmsg2%${newline}}"

	echo "\"${id}\",\"${server}\",\"${item}\",\"${wk_outputmsg1}\",\"${wk_result}\",\"${wk_outputmsg2}\",\"${wk_err_id}\"" >> "${output}"
}


##
#	Check
##

## start dependency
##  arg1 : resource type
##  arg2 : resource name
function start_dependency_check ()
{
	local resource_type=$1
	local resource_name=$2
	local target_resource="${resource_type}@${resource_name}"

	# enum group
	xml_path="/root/group"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for group_name in ${cfget}
		do
			# enum (group resource)
			xml_path="/root/group@${group_name}/resource"
			clpcfget_exe "e"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				for resource in ${cfget}
				do
					if [ "${resource}" = "${target_resource}" ];
					then
						# enum start dependency
						xml_path="/root/group@${group_name}/policy"
						clpcfget_exe "e"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							# get value (start dependency)
							xml_path="/root/group@${group_name}/policy@${server}/order"
							clpcfget_exe "g"
							if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
							then
								return ${FUNC_RET_SUCCESS}
							fi
						elif [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];
						then
							# target : all server(not set start dependency[default])
							return ${FUNC_RET_SUCCESS}
						fi
					fi
				done
			fi
		done
	fi

	return ${FUNC_RET_FAIL}
}

##### Not Tested!! ###############
## monitoring server
##  arg1 : monitor type
##  arg2 : monitor name
function monitoring_server_check ()
{
	local monitor_type=$1
	local monitor_name=$2

	# enum servers id
	xml_path="/root/monitor/${monitor_type}@${monitor_name}/polling/servers"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for servers_id in ${cfget}
		do
			# get value (polling server name)
			xml_path="/root/monitor/${monitor_type}@${monitor_name}/polling/servers@${servers_id}/name"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" = "${server}" ];
				then
					return ${FUNC_RET_SUCCESS}
				fi
			fi
		done
	elif [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];
	then
		# target : all server(not set monitoring server[default])
		return ${FUNC_RET_SUCCESS}
	fi

	return ${FUNC_RET_FAIL}
}
##################################

## ping
##  arg1 : IP address
function ping_check ()
{
	local ipaddr=$1
	local ping_exitcode=""
	local ping6_exitcode=""

	# ping : ipv4
	ping ${ipaddr} -c 3 > /dev/null 2>&1
	ping_exitcode=$?
	if [ "${ping_exitcode}" -eq 0 ];
	then
		return ${FUNC_RET_SUCCESS}
	elif [ "${ping_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${ping_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${ping_exitcode}
	else
		# ping : ipv6
		ping6 ${ipaddr} -c 3 > /dev/null 2>&1
		ping6_exitcode=$?
		if [ "${ping6_exitcode}" -eq 0 ];
		then
			return ${FUNC_RET_SUCCESS}
		elif [ "${ping6_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${ping6_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			return ${ping6_exitcode}
		fi
	fi

	return ${FUNC_RET_FAIL}
}

## IP address already check(fip/vip)
##  arg1 : IP address
function ipaddr_already_used_check ()
{
	local ipaddr=$1
	local exitcode=""
	local rsc_type_list=("fip" "vip")
	local rsc_type=""
	local xml_path=""
	local cmd_result=""
	local target_server=""

	# online/offline
	clpstat -s --cl | grep -E "^Servers.*:.*Offline$" > /dev/null 2>&1
	exitcode=$?
	if [ "${exitcode}" -eq "${EXITCODE_CMD_SUCCESS}" ];
	then
		# offline
		return ${FUNC_RET_FAIL}
	else
		# online
		for rsc_type in ${rsc_type_list[@]};
		do
			# fip/vip
			xml_path="/root/resource/${rsc_type}"
			cmd_result=`clpcfget -e ${xml_path}`
			exitcode=$?
			if [ "${exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				for rsc_name in ${cmd_result};
				do
					# resource status
					cmd_result=`clpstat -s --rsc ${rsc_name}`
					exitcode=$?
					if [ "${exitcode}" -eq "${EXITCODE_STAT_RSC_ONLINE}" ];
					then
						# resource status : online
						#  $cmd_result -> Resource <resource name> : Online (current <server name>)
						target_server=`echo ${cmd_result} | sed -e "s/.*(current \(.*\))/\1/g"`
						xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/ip"
						cmd_result=`clpcfget -g ${xml_path} -s ${target_server}`
						exitcode=$?
						if [ "${exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							cmd_result=`echo ${cmd_result%%/*}`
							cmd_result=`echo ${cmd_result%%%*}`
							if [ "${cmd_result}" = "${ipaddr}" ];
							then
								# fip/vip set ip address
								return ${FUNC_RET_SUCCESS}
							fi
						fi
					fi
				done
			fi
		done
	fi

	return ${FUNC_RET_FAIL}
}

## port number
##  arg1 : port number
function port_check ()
{
	local port=$1
	local eport_min
	local eport_max
	local rports
	local trash
	local sysctl_exitcode=""

	# 不正なポート番号ならエラー
	if [ ! "${port}" -gt 0 ];
	then
		return ${FUNC_RET_FAIL}
	fi

	# エフェメラルポートの範囲取得
	sysctl net.ipv4.ip_local_port_range > /dev/null 2>&1
	sysctl_exitcode=$?
	if [ "${sysctl_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${sysctl_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${sysctl_exitcode}
	fi
	IFS=$default_IFS read trash trash eport_min eport_max <<< `sysctl net.ipv4.ip_local_port_range`

	if ! [ "${eport_min}" -ge 0 -a "${eport_max}" -ge 0 ];
	then
		# 取得失敗なら次に進む
		:
	elif ! [ "${port}" -ge "${eport_min}" -a "${port}" -le "${eport_max}" ];
	then
		# 対象ポートが範囲外なら正常終了
		return ${FUNC_RET_SUCCESS}
	fi

	# 予約ポートの取得
	sysctl net.ipv4.ip_local_reserved_ports > /dev/null 2>&1
	sysctl_exitcode=$?
	if [ "${sysctl_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${sysctl_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${sysctl_exitcode}
	fi
	IFS=$default_IFS read trash trash rports <<< `sysctl net.ipv4.ip_local_reserved_ports`

	# 予約ポートとの比較
	while read -d, rport;
	do
		# 対象ポートが予約済みなら正常終了
		echo ${rport} | grep "-" > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			# 単一ポート
			if [ "${port}" = "${rport}" ];
			then
				return ${FUNC_RET_SUCCESS}
			fi
		else
			# 範囲指定
			rport1=`echo ${rport} | awk -F- '{print $1}'`
			rport2=`echo ${rport} | awk -F- '{print $2}'`
			if [ "${rport1}" -le "${port}" -a "${port}" -le "${rport2}" ];
			then
				return ${FUNC_RET_SUCCESS}
			fi
		fi
	done <<< "${rports},"

	return ${FUNC_RET_FAIL}
}

## Port Number Usage Check 
##  arg1 : port number
function port_usage_check ()
{
	local port=$1
	local cmd_result=""

	# 不正なポート番号ならエラー
	if [ ! "${port}" -gt 0 ];
	then
		return ${FUNC_RET_FAIL}
	fi

	# ポート番号の使用有無チェック(未使用ならOK)
	cmd_result=`ss -tuln | grep :${port}`
	if [ -z "${cmd_result}" ]; then
		return ${FUNC_RET_SUCCESS}
	else
		return ${FUNC_RET_FAIL}
	fi
}

## resources are active Check 
##  arg1 : resource name
function resource_active_check ()
{
	local resource_name=$1
	local cmd_result=""
	local exitcode=""

	# 対象リソースの活性サーバ確認(自サーバで活性中ならOK)
	cmd_result=`clprsc -n ${resource_name}`
	exitcode=$?
	if [ "${exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		if [ "${cmd_result}" = "${server}" ]; then
			return ${FUNC_RET_SUCCESS}
		else
			return ${FUNC_RET_FAIL}
		fi
	fi
}

## device
#  block device
#   arg1 : device name
function block_device_check ()
{
	local dev=$1
	if [ ! -b "${dev}" ];
	then
		return ${FUNC_RET_FAIL}
	fi
	return ${FUNC_RET_SUCCESS}
}
#  character device
#   arg1 : device name
function character_device_check ()
{
	local dev=$1
	if [ ! -c "${dev}" ];
	then
		return ${FUNC_RET_FAIL}
	fi
	return ${FUNC_RET_SUCCESS}
}

## kernel
function kernel_check ()
{
	local kernel_ver=`uname -r`
	local drv_dir="/opt/nec/clusterpro/drivers"
	local ka_dir
	local khb_dir
	local md_dir

	# check ka driver
	ka_dir="${drv_dir}/ka/distribution/current"
	if [ -d "${ka_dir}" ];
	then
		if [ ! -f "${ka_dir}/clpka-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	# check khb driver
	khb_dir="${drv_dir}/khb/distribution/current"
	if [ -d "${khb_dir}" ];
	then
		if [ ! -f "${khb_dir}/clpkhb-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	# check liscal driver
	md_dir="${drv_dir}/md/distribution/current"
	if [ -d "${md_dir}" ];
	then
		if [ ! -f "${md_dir}/liscal-${kernel_ver}.ko" ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}

## file system
#  disk
#   arg1 : device name
#   arg2 : file system
function disk_file_system_check ()
{
	local dev=$1
	local fs=$2
	local parted_exitcode=""
	local lsblk_exitcode=""

	# check file system
	#  -m : Number:Start:End:Size:File system:Name:Flags
	 parted -ms ${dev} p > /dev/null 2>&1
	parted_exitcode=$?
	 parted -ms ${dev} p | grep -E "([0-9]+):(.*):(.*):(.*):(${fs}):(.*):(.*)" > /dev/null 2>&1
	if [ $? -ne 0 ];
	then
		lsblk -dnf ${dev} > /dev/null 2>&1
		lsblk_exitcode=$?
		if [ "${lsblk_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${lsblk_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ] && \
			[ "${parted_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${parted_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			return ${lsblk_exitcode}
		fi

		lsblk -dnf ${dev} | grep ${fs} > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}

## /etc/fstab
#  disk,md,hd,diskhb
#   arg1 : device name
function fstab_entry_check_dev ()
{
	local dev=$1
	local grep_result=""
	local options=""

	# remove last "/"
	dev="${dev%/}"
	grep_result=`grep -E "^${dev}/*(\s)+" /etc/fstab | grep -v "#"`
	if [ $? -eq 0 ];
	then
		# /etc/fstab entry
		# fstab : <device> <dir> <type> <options> <dump> <fsck>
		options=`echo ${grep_result} | awk '{print $4}'`
		echo ${options} | grep "noauto" > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			# option : not "noauto"
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}
#   arg1 : mount point
function fstab_entry_check_mnt ()
{
	local mnt=$1
	local grep_result=""
	local options=""

	# remove last "/"
	mnt="${mnt%/}"

	# replace " "(space) with "\040"
	echo ${mnt} | grep " " > /dev/null 2>&1
	if [ $? -eq 0 ];
	then
		mnt=${mnt// /\\\\040}
	fi

	grep_result=`grep -E "(\s)+${mnt}/*(\s)+" /etc/fstab | grep -v "#"`
	if [ $? -eq 0 ];
	then
		# /etc/fstab entry
		# fstab : <device> <dir> <type> <options> <dump> <fsck>
		options=`echo ${grep_result} | awk '{print $4}'`
		echo ${options} | grep "noauto" > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			# option : not "noauto"
			return ${FUNC_RET_FAIL}
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}

## device size check
#  md,hd
#   arg1 : device name
function device_size_check ()
{
	local dev=$1
	local path=""
	local size=(0)

	path=`realpath ${dev}`
	size=(`lsblk -nbo SIZE ${path}`)
	# 1GB : 1073741824 byte
	if [ "${size[0]}" -lt 1073741824 ];
	then
		return ${FUNC_RET_FAIL}
	fi

	return ${FUNC_RET_SUCCESS}
}

## command existence
#   arg1 : command name
function cmd_existence_check ()
{
	local cmd=$1

	which ${cmd} > /dev/null 2>&1
	exitcode=$?
	if [ "${exitcode}" -eq 0 ];
	then
		return ${FUNC_RET_SUCCESS}
	elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${exitcode}
	fi

	return ${FUNC_RET_FAIL}
}

## Secure Boot
function secure_boot_check ()
{
	local mokutil_exitcode=""
	local bootctl_exitcode=""
	local mokutil_result=""
	local bootctl_result=""

	mokutil_result=`mokutil --sb-state`
	mokutil_exitcode=$?
	if [ "${mokutil_exitcode}" -eq 0 ];
	then
		echo ${mokutil_result} | grep "disabled" > /dev/null 2>&1
		if [ $? -ne 0 ];
		then
			return ${FUNC_RET_FAIL}
		fi
	else
		bootctl status > /dev/null 2>&1
		bootctl_exitcode=$?
		if [ "${mokutil_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${mokutil_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ] && \
			[ "${bootctl_exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${bootctl_exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			return ${bootctl_exitcode}
		fi

		bootctl_result=`bootctl status | grep "Secure Boot"`
		if [ $? -ne 0 ];
		then
			# not check target
			return ${FUNC_RET_NONTARGET}
		else
			echo ${bootctl_result} | grep "disabled" > /dev/null 2>&1
			if [ $? -ne 0 ];
			then
				return ${FUNC_RET_FAIL}
			fi
		fi
	fi

	return ${FUNC_RET_SUCCESS}
}

## watchdog use check
function watchdog_use_check ()
{
	# check driver(lsmod)
	lsmod | grep -e hpwdt -e i8xx_tco -e iTCO_wdt > /dev/null 2>&1
	exitcode=$?
	if [ "${exitcode}" -eq 0 ];
	then
		return ${FUNC_RET_FAIL}
	elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		return ${exitcode}
	fi

	# check /dev/watchdog
	if [ -e /dev/watchdog ];
	then
		# check /proc/PID
		for pid in $(ls /proc | grep -E '^[0-9]+$');
		do
			if [ -e /proc/$pid/fd ];
			then
				for fd in /proc/$pid/fd/*;
				do
					if [ "$(readlink $fd 2>/dev/null)" = "/dev/watchdog" ];
					then
						# echo "PID: $pid, CMD: $(cat /proc/$pid/comm)"
						if [ "$(cat /proc/$pid/comm)" != "clpuserw" ]; then
							return ${FUNC_RET_FAIL}
						fi
					fi
				done
			fi
		done
	fi

	return ${FUNC_RET_SUCCESS}
}

#------------------------------------------------
# Main
#------------------------------------------------
export LANG=C
EDITION_SSS="SSS"

# Source function library.
if [ -f /etc/rc.d/init.d/functions ]
then
	. /etc/rc.d/init.d/functions
elif [ -f /etc/rc.status ]
then
	. /etc/rc.status
elif [ -f /lib/lsb/init-functions ]
then
        . /lib/lsb/init-functions
fi

. /opt/nec/clusterpro/bin/clpfunctions
targetos=`/bin/uname`
clp_filedel $LOG_EXTENTION $LOG_FNAME $targetos
clp_logwrite $LOG_EXTENTION "clpvalidation start" $LOG_FNAME

## get option
ipath=""
opath=""
while getopts ":i:o:" opt;
do
	case ${opt} in
		i ) ipath=${OPTARG} ;;		# clp.conf
		o ) opath=${OPTARG} ;;		# output dir
		\?) exit ${RET_FAIL_OPT} ;;
	esac
done

## server, server list
server_fullname=`uname -n`		# get hostname (use output file path)
server=`echo ${server_fullname%%.*}`	# delete domain name

# enum server
xml_path="/root/server"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	server_list=${cfget}
else
	server_list=""
fi

## output file path
output="${opath}/${server_fullname}.csv"

## message string
result_summary="TRUE"		# 総合結果("TRUE" or "FALSE")
newline="<br/>"			# 改行文字
checkerr_msg=""			# メッセージ:エラー
notchktarget_msg=""		# メッセージ:チェック対象外
msg_cp="CP"			# クラスタパーティション
msg_dp="DP"			# データパーティション
rep="!REP!"			# 置換文字

## message file path
msgfpath="/opt/nec/clusterpro/etc/validation/resultmsg.us"

# get value (charset)
xml_path="/root/all/charset"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ "${cfget}" = "EUC-JP" ];
	then
		msgfpath="/opt/nec/clusterpro/etc/validation/resultmsg.jp"
	elif [ "${cfget}" = "GB2312" ];
	then
		msgfpath="/opt/nec/clusterpro/etc/validation/resultmsg.cn"
	fi
fi

# get value (edition)
xml_path="/root/all/edition"
clpcfget_exe "g"
edition=${cfget}

## Internal Field Separator
default_IFS=$IFS
change_IFS=$'\n'
IFS=$change_IFS

## ID : 2060,2061,2062,4020
#   check disk,md,hd,diskhb /etc/fstab entry
fstab_exists_flg=true
if ! [ -f /etc/fstab ];
then
	# /etc/fstab が存在しない
	fstab_exists_flg=false
fi


#--------------------
# ID : 3
#  get message(check error)
#--------------------
id=3
read_msg_file
checkerr_msg=${msg2}
unexpected_err_id_name="err_id_${id}_1"
unexpected_err_id=$(eval "echo \"\${${unexpected_err_id_name}}\"")

#--------------------
# ID : 4
#  get message(not check target)
#--------------------
id=4
read_msg_file
notchktarget_msg=${msg2}

#--------------------
# ID : 1
#  start time
#--------------------
id=1
read_msg_file
output_msg_datetime

#--------------------
# ID : 1000
#  check NP resolution ip address(ping)
#--------------------
id=1000
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# enum pingnp
xml_path="/root/networkpartition/pingnp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	output_flg=true
	for pingnp_name in ${cfget};	# for : 1
	do
		# get value (device)
		xml_path="/root/networkpartition/pingnp@${pingnp_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			# get value (info)
			xml_path="/root/server@${server}/device@${cfget}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					# enum grp
					xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp"
					clpcfget_exe "e"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						for grp_id in ${cfget};	# for : 2
						do
							# enum list
							xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list"
							clpcfget_exe "e"
							if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
							then
								for list_id in ${cfget};	# for : 3
								do
									# get value (IP address)
									xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list@${list_id}/ip"
									clpcfget_exe "g"
									if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
									then
										ping_check "${cfget}"
										exitcode=$?
										if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
										then
											# failed command execute
											checkerr_flg=true
											set_err_id 1
											break 3		# -> for : 1
										elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
										then
											result_summary="FALSE"
											result="FALSE"
											result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
											set_err_id 2
										fi
									else
										# failed to get value
										checkerr_flg=true
										break 3		# -> for : 1
									fi
								done
							else
								# failed to enum pingnp/grp/list
								checkerr_flg=true
								break 2		# -> for : 1
							fi
						done
					else
						# failed to enum pingnp/grp
						checkerr_flg=true
						break 1		# -> for : 1
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
				break 1		# -> for : 1
			fi
		else
			# failed to get value
			checkerr_flg=true
			break 1		# -> for : 1
		fi
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum pingnp
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1001
#  check NP resolution ip address(cluster server address)
#--------------------
id=1001
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# enum pingnp
xml_path="/root/networkpartition/pingnp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	output_flg=true
	for pingnp_name in ${cfget};	# for : 1
	do
		# get value (device)
		xml_path="/root/networkpartition/pingnp@${pingnp_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			# get value (info)
			xml_path="/root/server@${server}/device@${cfget}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					# enum grp
					xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp"
					clpcfget_exe "e"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						for grp_id in ${cfget};	# for : 2
						do
							# enum list
							xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list"
							clpcfget_exe "e"
							if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
							then
								for list_id in ${cfget};	# for : 3
								do
									# get value (IP address)
									xml_path="/root/networkpartition/pingnp@${pingnp_name}/grp@${grp_id}/list@${list_id}/ip"
									clpcfget_exe "g"
									if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
									then
										# get NP ping IPaddress
										np_ping_address=$cfget

										# get cluster server IPaddress
										xml_path="/root/server"
										clpcfget_exe "e"
										if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
										then
											for server_name in ${cfget};	# for : 4
											do
												# get value (device)
												xml_path="/root/server@${server_name}/device"
												clpcfget_exe "e"
												if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
												then
													for device_name in ${cfget};	# for : 5
													do
														# get value (info)
														xml_path="/root/server@${server_name}/device@${device_name}/info"
														clpcfget_exe "g"
														if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
														then
															# check ip address 
															if [ "${np_ping_address}" = "${cfget}" ];
															then
																result_summary="FALSE"
																result="FALSE"
																result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
																set_err_id 1
																break 5		# -> for : 1
															fi
														fi
													done # for : 5
												fi
											done # for : 4

										fi
									else
										# failed to get value
										checkerr_flg=true
										break 3		# -> for : 1
									fi
								done # for : 3
							else
								# failed to enum pingnp/grp/list
								checkerr_flg=true
								break 2		# -> for : 1
							fi
						done # for : 2
					else
						# failed to enum pingnp/grp
						checkerr_flg=true
						break 1		# -> for : 1
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
				break 1		# -> for : 1
			fi
		else
			# failed to get value
			checkerr_flg=true
			break 1		# -> for : 1
		fi
	done # for : 1
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視
then
	# failed to enum pingnp
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1010
#  check port number
#--------------------
id=1010
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
xml_path_list=(
	"/root/cluster/api/port"		# 29001 (TCP)
	"/root/cluster/api/ibport"		# 29008 (TCP)
	"/root/cluster/trns/port"		# 29002 (TCP)
	"/root/webmgr/http/port"		# 29003 (TCP)
	"/root/cluster/rstd/http/port"		# 29009 (TCP)
	"/root/cluster/rstd/service/port"	# 29010 (TCP)
	"/root/cluster/heartbeat/port/recv"	# 29002 (UDP)
	"/root/cluster/heartbeat/khbport/recv"	# 29006 (UDP)
	"/root/webalert/daemon/udpport"		# 29003 (UDP)
)

## get message
read_msg_file

## check
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

for xml_path in ${xml_path_list[@]};
do
	# get value (port)
	clpcfget_exe "g"
	if [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない場合はポリシータイプを指定して再取得
	then
		# policy type
		pol=`echo ${xml_path#/root/}`
		pol=`echo ${pol%%/*}`
		# get value (port)
		clpcfget_exe "g" "${pol}"
	fi

	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		port_check "${cfget}"
		exitcode=$?
		if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			# failed command execute
			checkerr_flg=true
			set_err_id 1
			break
		elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			set_err_id 2
		fi
	else
		# failed to get value
		checkerr_flg=true
		break
	fi
done

if "${checkerr_flg}";
then
	# could not check
	set_msg_couldnotchk
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 1011
#  check port number(mirror)
#--------------------
if [ "${edition}" != "${EDITION_SSS}" ];
then
	id=1011
	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
	## get message
	read_msg_file

	## check
	checkerr_flg=false
	result="TRUE"
	result_msg1=${msg1}
	result_msg2=""
	err_id=""

	# get value (port)
	xml_path="/root/mdagent/port"
	clpcfget_exe "g"
	if [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない場合はポリシータイプを指定して再取得
	then
		# policy type
		pol=`echo ${xml_path#/root/}`
		pol=`echo ${pol%%/*}`
		# get value (port)
		clpcfget_exe "g" "${pol}"
	fi

	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		port_check "${cfget}"
		exitcode=$?
		if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			# failed command execute
			checkerr_flg=true
			set_err_id 1
		elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
			set_err_id 2
		fi
	else
		# failed to get value
		checkerr_flg=true
	fi

	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1012
#  check port number(log)
#--------------------
id=1012
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# get value (method)
xml_path="/root/cluster/event/method"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ "${cfget}" -eq 1 ];	# 0:UNIX DOMAIN, 1:UDP, 2:MSGQ
	then
		# method : UDP
		output_flg=true
		# get value (port)
		xml_path="/root/cluster/event/port"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			port_check "${cfget}"
			exitcode=$?
			if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
			then
				# failed command execute
				checkerr_flg=true
				set_err_id 1
			elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
				set_err_id 2
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi
	fi
else
	# failed to get value
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1100
#  Check if the specified path is installation path.
#--------------------
id=1100
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# check logarc use flag
xml_path="/root/cluster/logarc/use"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}"  ];
then
	if [ "${cfget}" -eq "1" ];
	then
		output_flg=true
		# get logarc path
		xml_path="/root/cluster/logarc/path"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			# check directory path is under INSTALLPATH
			if [[ "${cfget}" = /opt/nec/clusterpro* ]];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=${msg2}
				set_err_id 2
			fi
		elif [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];
		then
			# could not check
			checkerr_flg=true
			set_err_id 1
		else
			# clp cfget error
			checkerr_flg=true
		fi
	fi
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];
then
	# failed to get value
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1101
#  Check if the specified path exists.
#--------------------
id=1101
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# check logarc use flag
xml_path="/root/cluster/logarc/use"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}"  ];
then
	if [ "${cfget}" -eq "1" ];
	then
		output_flg=true
		# get logarc path
		xml_path="/root/cluster/logarc/path"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			# check exists of directory
			if [ ! -d "${cfget}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=${msg2}
				set_err_id 2
			fi
		elif [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];
		then
			# could not check
			checkerr_flg=true
			set_err_id 1
		else
			# clp cfget error
			checkerr_flg=true
		fi
	fi
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];
then
	# failed to get value
	output_flg=true
	checkerr_flg=true
fi

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 1200
#  /dev/watchdog usage check
#--------------------
id=1200
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

# Check if softdog is used(cluster haltp)
xml_path="/root/haltp/method"
clpcfget_exe "g"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	if [ "${cfget}" = "softdog" ];
	then
		output_flg=true
	fi
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	checkerr_flg=true
fi

# check /dev/watchdog
if "${output_flg}";
then
	watchdog_use_check
	exitcode=$?
	if [ "${exitcode}" -eq "${FUNC_RET_FAIL}" ];
	then
		result_summary="FALSE"
		result="FALSE"
		result_msg2=${msg2}
		set_err_id 1
	elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		checkerr_flg=true
	fi

	if ! "${checkerr_flg}";
	then
		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
fi

if "${checkerr_flg}";
then
	# failed to enum userw
	set_msg_couldnotchk "Shutdown Monitor"
	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2000
#  check fip ip address(ping)
#--------------------
id=2000
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum fip
xml_path="/root/resource/fip"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for fip_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${fip_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "fip" "${fip_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (IP address)
			xml_path="/root/resource/fip@${fip_name}/parameters/ip"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				cfget=`echo ${cfget%%/*}`
				cfget=`echo ${cfget%%%*}`
				ping_check "${cfget}"
				exitcode=$?
				if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
				then
					ipaddr_already_used_check "${cfget}"
					exitcode=$?
					if [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						set_err_id 2
					fi
				elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
				then
					# failed command execute
					checkerr_flg=true
					set_err_id 1
				fi
			elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
			then
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum fip
	set_msg_couldnotchk "Floating IP resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2001
#  check vip ip address(ping)
#--------------------
id=2001
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum vip
xml_path="/root/resource/vip"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for vip_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${vip_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "vip" "${vip_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (IP address)
			xml_path="/root/resource/vip@${vip_name}/parameters/ip"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				cfget=`echo ${cfget%%/*}`
				ping_check "${cfget}"
				exitcode=$?
				if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
				then
					ipaddr_already_used_check "${cfget}"
					exitcode=$?
					if [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						set_err_id 2
					fi
				elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
				then
					# failed command execute
					checkerr_flg=true
					set_err_id 1
				fi
			elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
			then
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum vip
	set_msg_couldnotchk "Virtual IP resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2010
#  check disk drive letter
#--------------------
id=2010
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

declare -a device_check_fail_list=()

## check
# enum disk
xml_path="/root/resource/disk"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for disk_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${disk_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "disk" "${disk_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (device)
			xml_path="/root/resource/disk@${disk_name}/parameters/device"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				block_device_check "${cfget}"
				if [ $? -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
					device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
					set_err_id 1
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2011
#  check md drive letter
#--------------------
id=2011
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum md
xml_path="/root/resource/md"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for md_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${md_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "md" "${md_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (dppath)
			xml_path="/root/resource/md@${md_name}/parameters/diskdev/dppath"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				block_device_check "${cfget}"
				if [ $? -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${msg_dp} ${cfget}|"`
					device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
					set_err_id 1
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if ! "${checkerr_flg}";
			then
				# get value (cppath)
				xml_path="/root/resource/md@${md_name}/parameters/diskdev/cppath"
				clpcfget_exe "g" "" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					block_device_check "${cfget}"
					if [ $? -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${msg_cp} ${cfget}|"`
						device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
						set_err_id 1
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Mirror disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2012
#  check hd drive letter
#--------------------
id=2012
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum hd
xml_path="/root/resource/hd"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for hd_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${hd_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "hd" "${hd_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (dppath)
			xml_path="/root/resource/hd@${hd_name}/parameters/diskdev/dppath"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				block_device_check "${cfget}"
				if [ $? -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${msg_dp} ${cfget}|"`
					device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
					set_err_id 1
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if ! "${checkerr_flg}";
			then
				# get value (cppath)
				xml_path="/root/resource/hd@${hd_name}/parameters/diskdev/cppath"
				clpcfget_exe "g" "" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					block_device_check "${cfget}"
					if [ $? -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${msg_cp} ${cfget}|"`
						device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
						set_err_id 1
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Hybrid disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2050 - 2051
#  check md/hd drive letter size (CP 1GB)
#--------------------
declare -a drive_check_data=()
# タブ区切り : <rsc_type>	<id>	<name>
drive_check_data=(
	"md	2050	Mirror disk resource"
	"hd	2051	Hybrid disk resource"
)

for data_list in "${drive_check_data[@]}";
do
	# 区切り文字をタブのみに設定
	IFS='	'

	data=(${data_list[@]})
	rsc_type=${data[0]}
	id=${data[1]}
	name=${data[2]}

	IFS=$change_IFS

	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME

	## get message
	read_msg_file

	## check
	# enum resource
	xml_path="/root/resource/${rsc_type}"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for rsc_name in ${cfget};
		do
			checkerr_flg=false
			notchktarget_flg=false
			result="TRUE"
			result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rsc_name}|"`
			result_msg2=""
			err_id=""
			

			start_dependency_check "${rsc_type}" "${rsc_name}"
			exitcode=$?
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				# get value (cppath)
				xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/diskdev/cppath"
				clpcfget_exe "g" "" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					# partition presence check
					for device_name in ${device_check_fail_list[@]};
					do
						if [ ${cfget} = ${device_name} ];
						then
							notchktarget_flg=true
							break
						fi
					done

					if ! "${notchktarget_flg}";
					then
						# check device size
						device_size_check "${cfget}"
						if [ $? -ne "${FUNC_RET_SUCCESS}" ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=`echo ${msg2} | sed -e "s|${rep}|${cfget}|"`
							set_err_id 1
						fi
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			else
				# not check target
				notchktarget_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			elif "${notchktarget_flg}";
			then
				# not check target
				set_msg_notchktarget
			fi

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		done
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource
		set_msg_couldnotchk "${name}"

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
done

#--------------------
# ID : 2060
#  check disk /etc/fstab entry
#--------------------
id=2060
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum disk
xml_path="/root/resource/disk"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for disk_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${disk_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "disk" "${disk_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			## デバイス名
			# get value (device)
			xml_path="/root/resource/disk@${disk_name}/parameters/device"
			clpcfget_exe "g" "" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if "${fstab_exists_flg}";
				then
					# /etc/fstab が存在する
					fstab_entry_check_dev "${cfget}"
					if [ $? -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						set_err_id 1
					fi
				else
					# /etc/fstab が存在しない
					checkerr_flg=true
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if ! "${checkerr_flg}";
			then
				## マウントポイント(ディスクタイプ:raw以外)
				# get value (disktype)
				xml_path="/root/resource/disk@${disk_name}/parameters/disktype"
				clpcfget_exe "g" "disk" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					if [ "${cfget}" != "raw" ];
					then
						# get value (mount point)
						xml_path="/root/resource/disk@${disk_name}/parameters/mount/point"
						clpcfget_exe "g" "" "${server}"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if "${fstab_exists_flg}";
							then
								# /etc/fstab が存在する
								fstab_entry_check_mnt "${cfget}"
								if [ $? -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
									set_err_id 1
								fi
							else
								# /etc/fstab が存在しない
								checkerr_flg=true
							fi
						else
							# failed to get value
							checkerr_flg=true
						fi
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2061 - 2062
#  check md,hd /etc/fstab entry
#--------------------
declare -a drive_check_data=()
# タブ区切り : <rsc_type>	<id>	<name>
drive_check_data=(
	"md	2061	Mirror disk resource"
	"hd	2062	Hybrid disk resource"
)

for data_list in "${drive_check_data[@]}";
do
	# 区切り文字をタブのみに設定
	IFS='	'

	data=(${data_list[@]})
	rsc_type=${data[0]}
	id=${data[1]}
	name=${data[2]}

	IFS=$change_IFS

	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME

	## get message
	read_msg_file

	## check
	# enum resource
	xml_path="/root/resource/${rsc_type}"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for rsc_name in ${cfget};
		do
			checkerr_flg=false
			result="TRUE"
			result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rsc_name}|"`
			result_msg2=""
			err_id=""

			start_dependency_check "${rsc_type}" "${rsc_name}"
			exitcode=$?
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				## マウントポイント(ファイルシステム:none以外)
				# get value (file system)
				xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/fs"
				clpcfget_exe "g" "${rsc_type}" "${server}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					if [ ${cfget} != "none" ];
					then
						# get value (mount point)
						xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/mount/point"
						clpcfget_exe "g" "" "${server}"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if "${fstab_exists_flg}";
							then
								# /etc/fstab が存在する
								fstab_entry_check_mnt "${cfget}"
								if [ $? -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
									set_err_id 1
								fi
							else
								# /etc/fstab が存在しない
								checkerr_flg=true
							fi
						else
							# failed to get value
							checkerr_flg=true
						fi
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi

				if ! "${checkerr_flg}";
				then
					## データパーティションデバイス名
					# get value (dppath)
					xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/diskdev/dppath"
					clpcfget_exe "g" "" "${server}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						if "${fstab_exists_flg}";
						then
							# /etc/fstab が存在する
							fstab_entry_check_dev "${cfget}"
							if [ $? -ne "${FUNC_RET_SUCCESS}" ];
							then
								result_summary="FALSE"
								result="FALSE"
								result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
								set_err_id 1
							fi
						else
							# /etc/fstab が存在しない
							checkerr_flg=true
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				fi

				if ! "${checkerr_flg}";
				then
					## クラスタパーティションデバイス名
					# get value (cppath)
					xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/diskdev/cppath"
					clpcfget_exe "g" "" "${server}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						if "${fstab_exists_flg}";
						then
							# /etc/fstab が存在する
							fstab_entry_check_dev "${cfget}"
							if [ $? -ne "${FUNC_RET_SUCCESS}" ];
							then
								result_summary="FALSE"
								result="FALSE"
								result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
								set_err_id 1
							fi
						else
							# /etc/fstab が存在しない
							checkerr_flg=true
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				fi

				if "${checkerr_flg}";
				then
					# could not check
					set_msg_couldnotchk
				fi
			else
				# not check target
				set_msg_notchktarget
			fi

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		done
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource
		set_msg_couldnotchk "${name}"

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
done

#--------------------
# ID : 2021 - 2022
#  check md,hd port(Mirror Data Port Number, Heartbeat Port Number, ACK2 Port Number)
#--------------------
# タブ区切り : <rsc_type>	<id>	<name>
port_num_check_data=(
	"md	2021	Mirror disk resource"
	"hd	2022	Hybrid disk resource"
)

for data_list in "${port_num_check_data[@]}";
do
	# 区切り文字をタブのみに設定
	IFS='	'

	data=(${data_list[@]})
	rsc_type=${data[0]}
	id=${data[1]}
	name=${data[2]}

	IFS=$change_IFS

	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME

	## get message
	read_msg_file

	## check
	# enum resource
	xml_path="/root/resource/${rsc_type}"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for rsc_name in ${cfget};
		do
			checkerr_flg=false
			result="TRUE"
			result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rsc_name}|"`
			result_msg2=""
			err_id=""

			start_dependency_check "${rsc_type}" "${rsc_name}"
			exitcode=$?
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				xml_path_list=(
					"/root/resource/${rsc_type}@${rsc_name}/parameters/mddriver/port"	# Mirror Data Port Number
					"/root/resource/${rsc_type}@${rsc_name}/parameters/mddriver/hbport"	# Heartbeat Port Number
					"/root/resource/${rsc_type}@${rsc_name}/parameters/mddriver/ack2port"	# ACK2 Port Number
				)

				for xml_path in ${xml_path_list[@]};
				do
					# get value (port, hbport, ack2port)
					clpcfget_exe "g"
					if [ "${cfget_exitcode}" -eq "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない場合はポリシータイプを指定して再取得
					then
						clpcfget_exe "g" "${rsc_type}"
					fi

					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						port_check "${cfget}"
						exitcode=$?
						if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
						then
							# failed command execute
							checkerr_flg=true
							set_err_id 1
							break 1
						elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
							set_err_id 2
						fi
					else
						# failed to get value
						checkerr_flg=true
						break 1
					fi
				done
			else
				# not check target
				set_msg_notchktarget
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		done
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource
		set_msg_couldnotchk "${name}"

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
done

#--------------------
# ID : 2020
#  check Azure probe port
#--------------------
id=2020
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum azurepp
xml_path="/root/resource/azurepp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for azurepp_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${azurepp_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "azurepp" "${azurepp_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (port)
			xml_path="/root/resource/azurepp@${azurepp_name}/parameters/probeport"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				port_check "${cfget}"
				exitcode=$?
				if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
				then
					# failed command execute
					checkerr_flg=true
					set_err_id 1
				elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
				then
					result_summary="FALSE"
					result="FALSE"
					result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
					set_err_id 2
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Azure probe port resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2023
#  Port Number Usage Check for lbpp
#--------------------
id=2023
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum lbpp
xml_path="/root/resource/lbpp"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for lbpp_name in ${cfget};
	do
		checkerr_flg=false
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${lbpp_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "lbpp" "${lbpp_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (port)
			xml_path="/root/resource/lbpp@${lbpp_name}/parameters/port"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				# port usage check
				port_usage_check "${cfget}"
				exitcode=$?
				if [ "${exitcode}" -eq "${FUNC_RET_FAIL}" ];
				then
					clp_logwrite $LOG_EXTENTION "INFO  ${cfget} port is in use" $LOG_FNAME
					# lbpp resource active check
					resource_active_check "${lbpp_name}"
					exitcode=$?
					if [ "${exitcode}" -eq "${FUNC_RET_FAIL}" ];
					then
						clp_logwrite $LOG_EXTENTION "INFO  lbpp is inactive on this server" $LOG_FNAME
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						set_err_id 1
					else
						clp_logwrite $LOG_EXTENTION "INFO  lbpp is active on this server" $LOG_FNAME
					fi
				else
					clp_logwrite $LOG_EXTENTION "INFO  ${cfget} port is not in use" $LOG_FNAME
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "LP probe port resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2030
#  file system check(disk)
#--------------------
id=2030
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum disk
xml_path="/root/resource/disk"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for disk_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${disk_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "disk" "${disk_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (disktype)
			xml_path="/root/resource/disk@${disk_name}/parameters/disktype"
			clpcfget_exe "g" "disk" "${server}"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				disk_type=${cfget}
				if [ "${disk_type}" = "disk" ];
				then
					notchktarget_flg=false
					# get value (file system)
					xml_path="/root/resource/disk@${disk_name}/parameters/fs"
					clpcfget_exe "g" "disk" "${server}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						file_system=${cfget}
						# get value (device)
						xml_path="/root/resource/disk@${disk_name}/parameters/device"
						clpcfget_exe "g" "" "${server}"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							disk_file_system_check "${cfget}" "${file_system}"
							exitcode=$?
							if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
							then
								# failed command execute
								checkerr_flg=true
								set_err_id 1
							elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
							then
								result_summary="FALSE"
								result="FALSE"
								result_msg2=${msg2}
								set_err_id 2
							fi
						else
							# failed to get value
							checkerr_flg=true
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				elif [ "${disk_type}" = "raw" ];
				then
					# OK(PASS)
					notchktarget_flg=false
				elif [ "${disk_type}" = "lvm" ];
				then
					# not check
					notchktarget_flg=true
				elif [ "${disk_type}" = "vxvm" ];
				then
					# not check
					notchktarget_flg=true
				else
					# could not check
					checkerr_flg=true
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			elif "${notchktarget_flg}";
			then
				# not check target
				set_msg_notchktarget
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Disk resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2031 - 2032
#  file system check(md,hd)
#--------------------
declare -a file_system_check_data=()
file_system_check_data=(
	"md	2031	Mirror disk resource"
	"hd	2032	Hybrid disk resource"
)

for data_list in "${file_system_check_data[@]}";
do
	IFS='	'

	data=(${data_list[@]})
	rsc_type=${data[0]}
	id=${data[1]}
	name=${data[2]}

	IFS=$change_IFS

	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME

	## get message
	read_msg_file

	## check
	# enum resource
	xml_path="/root/resource/${rsc_type}"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for rsc_name in ${cfget};
		do
			checkerr_flg=false
			notchktarget_flg=true
			result="TRUE"
			result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rsc_name}|"`
			result_msg2=""
			err_id=""

			start_dependency_check "${rsc_type}" "${rsc_name}"
			exitcode=$?
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				# get value (file system)
				xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/fs"
				clpcfget_exe "g" "${rsc_type}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					file_system=${cfget}
					if [ "${file_system}" = "none" ];
					then
						# not check
						notchktarget_flg=true
					else
						# get value (mkfs)
						xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/mkfs"
						clpcfget_exe "g" "${rsc_type}"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							mkfs=${cfget}
							if [ "${mkfs}" != "0" ];
							then
								# not check
								notchktarget_flg=true
							else
								notchktarget_flg=false
								# get value (dppath)
								xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/diskdev/dppath"
								clpcfget_exe "g" "" "${server}"
								if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
								then
									disk_file_system_check "${cfget}" "${file_system}"
									exitcode=$?
									if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
									then
										# failed command execute
										checkerr_flg=true
										set_err_id 1
									elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
									then
										result_summary="FALSE"
										result="FALSE"
										result_msg2=${msg2}
										set_err_id 2
									fi
								else
									# failed to get value
									checkerr_flg=true
								fi
							fi
						else
							# failed to get value
							checkerr_flg=true
						fi
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			else
				# not check target
				notchktarget_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			elif "${notchktarget_flg}";
			then
				# not check target
				set_msg_notchktarget
			fi

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		done
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource
		set_msg_couldnotchk "${name}"

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
done


#--------------------
# ID : 2071 - 2072
#  mount point check(md,hd)
#--------------------
declare -a mount_point_check_data=()
mount_point_check_data=(
	"md	2071	Mirror disk resource"
	"hd	2072	Hybrid disk resource"
)

for data_list in "${mount_point_check_data[@]}";
do
	IFS='	'

	data=(${data_list[@]})
	rsc_type=${data[0]}
	id=${data[1]}
	name=${data[2]}

	IFS=$change_IFS

	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME

	## get message
	read_msg_file

	## check
	# enum resource
	xml_path="/root/resource/${rsc_type}"
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		for rsc_name in ${cfget};
		do
			checkerr_flg=false
			notchktarget_flg=true
			result="TRUE"
			result_msg1=`echo ${msg1} | sed -e "s|${rep}|${rsc_name}|"`
			result_msg2=""
			err_id=""

			start_dependency_check "${rsc_type}" "${rsc_name}"
			exitcode=$?
			if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
			then
				# get value (file system)
				xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/fs"
				clpcfget_exe "g" "${rsc_type}"
				if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
				then
					file_system=${cfget}
					if [ "${file_system}" = "ext3" -o "${file_system}" = "ext4" -o "${file_system}" = "reiserfs" ];
					then
						notchktarget_flg=false
						# get value (mount option)
						xml_path="/root/resource/${rsc_type}@${rsc_name}/parameters/mount/option"
						clpcfget_exe "g" "${rsc_type}"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							mount_option=${cfget}
							echo "${mount_option}" | grep "data=writeback" > /dev/null 2>&1
							if [ $? -eq 0 ];
							then
								result_summary="FALSE"
								result="FALSE"
								result_msg2=${msg2}
								set_err_id 1
							fi
						else
							# failed to get value
							checkerr_flg=true
						fi
					else
						# not check
						notchktarget_flg=true
					fi
				else
					# failed to get value
					checkerr_flg=true
				fi
			else
				# not check target
				notchktarget_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			elif "${notchktarget_flg}";
			then
				# not check target
				set_msg_notchktarget
			fi

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		done
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource
		set_msg_couldnotchk "${name}"

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
done


#--------------------
# ID : 2040
#  VG presence check
#--------------------
id=2040
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum volmgr
xml_path="/root/resource/volmgr"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for volmgr_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${volmgr_name}|"`
		result_msg2=""
		err_id=""

		start_dependency_check "volmgr" "${volmgr_name}"
		exitcode=$?
		if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
		then
			# get value (volume manager)
			xml_path="/root/resource/volmgr@${volmgr_name}/parameters/type"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				volmgr_type=${cfget}
				if [ "${volmgr_type}" = "lvm" ];
				then
					notchktarget_flg=false
					# get value (target name)
					xml_path="/root/resource/volmgr@${volmgr_name}/parameters/devname"
					clpcfget_exe "g"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						 vgs ${cfget} > /dev/null 2>&1
						exitcode=$?
						if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
						then
							# failed command execute
							checkerr_flg=true
							set_err_id 1
						elif [ "${exitcode}" -ne 0 ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
							set_err_id 2
						fi
					else
						# failed to get value
						checkerr_flg=true
					fi
				elif [ "${volmgr_type}" = "vxvm" ];
				then
					# not check
					notchktarget_flg=true
				elif [ "${volmgr_type}" = "zfspool" ];
				then
					# not check
					notchktarget_flg=true
				else
					# could not check
					checkerr_flg=true
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi

			if "${checkerr_flg}";
			then
				# could not check
				set_msg_couldnotchk
			elif "${notchktarget_flg}";
			then
				# not check target
				set_msg_notchktarget
			fi
		else
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum resource
	set_msg_couldnotchk "Volume manager resource"

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 3000
#  /dev/watchdog usage check
#--------------------
id=3000
## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
service_name=""

# Check if softdog is used(userw)
xml_path="/root/monitor/userw"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for userw_name in ${cfget};
	do
		# get value (method)
		xml_path="/root/monitor/userw@${userw_name}/parameters/method"
		clpcfget_exe "g" "userw"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			if [ "${cfget}" = "softdog" ];
			then
				output_flg=true
				service_name=${userw_name}
			fi
		elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
		then
			checkerr_flg=true
		fi
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	checkerr_flg=true
fi

# check /dev/watchdog
if "${output_flg}";
then
	result="TRUE"
	result_msg1=`echo ${msg1} | sed -e "s|${rep}|${service_name}|"`
	result_msg2=""
	err_id=""

	watchdog_use_check
	exitcode=$?
	if [ "${exitcode}" -eq "${FUNC_RET_FAIL}" ];
	then
		result_summary="FALSE"
		result="FALSE"
		result_msg2=${msg2}
		set_err_id 1
	elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
	then
		checkerr_flg=true
	fi

	if ! "${checkerr_flg}";
	then
		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	fi
fi

if "${checkerr_flg}";
then
	# failed to enum userw
	set_msg_couldnotchk "User mode monitor"
	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi


#--------------------
# ID : 4000
#  check kernel mode LAN heartbeat resource ip address(ping)
#--------------------
id=4000
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum lankhb
xml_path="/root/heartbeat/lankhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for lankhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${lankhb_name}|"`
		result_msg2=""
		err_id=""

		# get value (device)
		xml_path="/root/heartbeat/lankhb@${lankhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					for server_name in ${server_list};
					do
						# get value (info)
						xml_path="/root/server@${server_name}/device@${device}/info"
						clpcfget_exe "g"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if [ "${cfget}" != "" ];
							then
								notchktarget_flg=false
								ping_check "${cfget}"
								exitcode=$?
								if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
								then
									# failed command execute
									checkerr_flg=true
									set_err_id 1
									break
								elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
									set_err_id 2
								fi
							fi
						else
							# failed to get value
							checkerr_flg=true
							break
						fi
					done
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/lankhb
	set_msg_couldnotchk "Kernel Mode"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4001
#  check LAN heartbeat resource ip address(ping)
#--------------------
id=4001
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum lanhb
xml_path="/root/heartbeat/lanhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for lanhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${lanhb_name}|"`
		result_msg2=""
		err_id=""

		# get value (device)
		xml_path="/root/heartbeat/lanhb@${lanhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					for server_name in ${server_list};
					do
						# get value (info)
						xml_path="/root/server@${server_name}/device@${device}/info"
						clpcfget_exe "g"
						if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
						then
							if [ "${cfget}" != "" ];
							then
								notchktarget_flg=false
								ping_check "${cfget}"
								exitcode=$?
								if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
								then
									# failed command execute
									checkerr_flg=true
									set_err_id 1
									break
								elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
								then
									result_summary="FALSE"
									result="FALSE"
									result_msg2=`echo ${result_msg2}${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
									set_err_id 2
								fi
							fi
						else
							# failed to get value
							checkerr_flg=true
							break
						fi
					done
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/lanhb
	set_msg_couldnotchk "User Mode"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4010
#  check DISK drive letter
#--------------------
id=4010
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum diskhb
xml_path="/root/heartbeat/diskhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for diskhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${diskhb_name}|"`
		result_msg2=""
		err_id=""

		# get value (device)
		xml_path="/root/heartbeat/diskhb@${diskhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					notchktarget_flg=false
					block_device_check "${cfget}"
					if [ $? -ne "${FUNC_RET_SUCCESS}" ];
					then
						result_summary="FALSE"
						result="FALSE"
						result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
						device_check_fail_list=("${device_check_fail_list[@]}" "${cfget}")
						set_err_id 1
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/diskhb
	set_msg_couldnotchk "DISK"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 4020
#  check diskhb /etc/fstab entry
#--------------------
id=4020
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum diskhb
xml_path="/root/heartbeat/diskhb"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for diskhb_name in ${cfget};
	do
		checkerr_flg=false
		notchktarget_flg=true
		result="TRUE"
		result_msg1=`echo ${msg1} | sed -e "s|${rep}|${diskhb_name}|"`
		result_msg2=""
		err_id=""

		# get value (device)
		xml_path="/root/heartbeat/diskhb@${diskhb_name}/device"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			device=${cfget}
			# get value (info)
			xml_path="/root/server@${server}/device@${device}/info"
			clpcfget_exe "g"
			if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
			then
				if [ "${cfget}" != "" ];
				then
					notchktarget_flg=false
					if "${fstab_exists_flg}";
					then
						# /etc/fstab が存在する
						fstab_entry_check_dev "${cfget}"
						if [ $? -ne "${FUNC_RET_SUCCESS}" ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=`echo ${msg2}${newline} | sed -e "s|${rep}|${cfget}|"`
							set_err_id 1
						fi
					else
						# /etc/fstab が存在しない
						checkerr_flg=true
					fi
				fi
			else
				# failed to get value
				checkerr_flg=true
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi

		if "${checkerr_flg}";
		then
			# could not check
			set_msg_couldnotchk
		elif "${notchktarget_flg}";
		then
			# not check target
			set_msg_notchktarget
		fi

		# output result
		output_msg "${result_msg1}" "${result_msg2}"
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum heartbeat/diskhb
	set_msg_couldnotchk "DISK"

	# output output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 5000
#  check AWSCLI
#--------------------
id=5000
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
xml_path_list=(
	"/root/resource/awsdns"
	"/root/resource/awseip"
	"/root/resource/awsvip"
	"/root/monitor/awsazw"
	"/root/monitor/awsdnsw"
	"/root/monitor/awseipw"
	"/root/monitor/awsvipw"
)

## get message
read_msg_file

## check
output_flg=false
checkerr_flg=false
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

for xml_path in ${xml_path_list[@]};
do
	# enum resource/monitor
	clpcfget_exe "e"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		output_flg=true
		export PATH=$PATH:/usr/local/bin

		# AWSCLIのバージョン確認のみなら環境変数の設定は不要なためコメントアウト
		# scriptdir="$(cd $(dirname $0); pwd)"
		# . "${scriptdir}/../cloud/common/clpcloudutil.sh"
		# clpcloudutil_env_init "${clpcloudutil_true}" "${scriptdir}/../cloud/aws/clpaws_setting.conf"

		# AWSCLIのバージョン確認のみならAWSCLIのコマンドラインオプションの設定は不要なためコメントアウト
		# AWSサービスがEC2だった場合の処理を記載。
		# scriptdir="$(cd $(dirname $0); pwd)"
		# . "${scriptdir}/../cloud/common/clpcloudutil.sh"
		# ec2_cmdopt="`clpcloudutil_awscli_cmdopt \"${clpcloudutil_true}\" ec2`"

		aws --version > /dev/null 2>&1
		exitcode=$?
		if [ "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
		then
			# command not found
			checkerr_flg=true
		elif [ "${exitcode}" -ne 0 ];
		then
			result_summary="FALSE"
			result="FALSE"
			result_msg2=${msg2}
			set_err_id 1
		fi
		break
	elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
	then
		# failed to enum resource/monitor
		output_flg=true
		checkerr_flg=true
		break
	fi
done

if "${output_flg}";
then
	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 5001
#  check OS Boot timeout
#--------------------
if [ "${edition}" != "${EDITION_SSS}" ];
then
	id=5001
	clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
	## get message
	read_msg_file

	## check
	checkerr_flg=false
	result="TRUE"
	result_msg1=${msg1}
	result_msg2=""
	heartbeat_timeout=""
	timeout_val=""
	startupdelay=""
	total_val=""
	multi_val=1
	err_id=""

	# get value (heartbeat timeout)
	xml_path="/root/cluster/heartbeat/timeout"
	clpcfget_exe "g"
	if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
	then
		heartbeat_timeout=${cfget}
		if [ -f /etc/default/grub ];
		then
			# grub2 (in seconds)
			multi_val=1000
			#  pattern : GRUB_TIMEOUT=X
			timeout_val=`grep GRUB_TIMEOUT /etc/default/grub | grep -v "#" | awk -F= '{print $2}'`
			if [ "${timeout_val}" = "" ];
			then 
				#  pattern : GRUB_TIMEOUT X
				timeout_val=`grep GRUB_TIMEOUT /etc/default/grub | grep -v "#" | awk '{print $2}'`
			fi
		elif [ -f /boot/grub/menu.lst ];
		then
			# grub (in seconds)
			multi_val=1000
			#  pattern : timeout=X
			timeout_val=`grep timeout /boot/grub/menu.lst | grep -v "#" | awk -F= '{print $2}'`
			if [ "${timeout_val}" = "" ];
			then 
				#  pattern : timeout X
				timeout_val=`grep timeout /boot/grub/menu.lst | grep -v "#" | awk '{print $2}'`
			fi
		elif [ -f /etc/lilo.conf ];
		then
			# lilo (in 1/10 seconds)
			multi_val=100
			#  pattern : timeout=X
			timeout_val=`grep timeout /etc/lilo.conf | grep -v "#" | awk -F= '{print $2}'`
			if [ "${timeout_val}" = "" ];
			then 
				#  pattern : timeout X
				timeout_val=`grep timeout /etc/lilo.conf | grep -v "#" | awk '{print $2}'`
				if [ "${timeout_val}" = "" ];
				then 
					#  pattern : delay=X
					timeout_val=`grep delay /etc/lilo.conf | grep -v "#" | awk -F= '{print $2}'`
					if [ "${timeout_val}" = "" ];
					then 
						#  pattern : delay X
						timeout_val=`grep delay /etc/lilo.conf | grep -v "#" | awk '{print $2}'`
					fi
				fi
			fi
		fi

		if [ "${timeout_val}" != "" ];
		then
			timeout_val=$(( timeout_val * multi_val ))
		else
			timeout_val=0
		fi

		# get value (startupdelay)
		xml_path="/root/cluster/startupdelay"
		clpcfget_exe "g"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			startupdelay=$(( cfget * 1000 ))
			total_val=$(( timeout_val + startupdelay ))
			if [ "${heartbeat_timeout}" -gt "${total_val}" ];
			then
				result_summary="FALSE"
				result="FALSE"
				result_msg2=${msg2}
				set_err_id 1
			fi
		else
			# failed to get value
			checkerr_flg=true
		fi
	else
		# failed to get value
		checkerr_flg=true
	fi

	if "${checkerr_flg}";
	then
		# could not check
		set_msg_couldnotchk
	fi

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 5002
#  SELinux disable
#--------------------
id=5002
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

selinux_mode=`getenforce`
exitcode=$?
logmsg=`echo $LOG_INFO $LOG_CMD "getenforce"`
clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME
logmsg=`echo $LOG_INFO $LOG_EXITCODE "${exitcode}"`
clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME
logmsg=`echo $LOG_INFO $LOG_CMDRESULT "${selinux_mode}"`
clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME

if [ "${exitcode}" -ne 0 ];
then
	# could not check
	set_msg_couldnotchk
	set_err_id 1
elif [ "${selinux_mode}" = "Enforcing" ];
then
	 clpselctrl.sh --check > /dev/null 2>&1
	exitcode=$?
	logmsg=`echo $LOG_INFO $LOG_CMD "clpselctrl.sh --check"`
	clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME
	logmsg=`echo $LOG_INFO $LOG_EXITCODE "${exitcode}"`
	clp_logwrite $LOG_EXTENTION "$logmsg" $LOG_FNAME
	
	if [ "${exitcode}" -eq 1 ];
	then
		result_summary="FALSE"
		result="FALSE"
		result_msg2=${msg2}
		set_err_id 2
	elif [ "${exitcode}" -ne 0 ];
	then
		# could not check
		set_msg_couldnotchk
		set_err_id 1 
	fi
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 5004
#  Secure Boot disable
#--------------------
id=5004
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

secure_boot_check
exitcode=$?
if [ "${exitcode}" -eq "${FUNC_RET_NONTARGET}" ];
then
	# not check target
	set_msg_notchktarget
elif [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
then
	# failed command execute
	# could not check
	set_msg_couldnotchk
	set_err_id 1
elif [ "${exitcode}" -eq "${FUNC_RET_FAIL}" ];
then
	result_summary="FALSE"
	result="FALSE"
	result_msg2=${msg2}
	set_err_id 2
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 5003
#  Kernel
#--------------------
id=5003
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
result="TRUE"
result_msg1=${msg1}
result_msg2=""
err_id=""

kernel_check
if [ $? -ne "${FUNC_RET_SUCCESS}" ];
then
	result_summary="FALSE"
	result="FALSE"
	result_msg2=${msg2}
	set_err_id 1
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#--------------------
# ID : 59XX
#  Command install check
#--------------------

#----------
#  tar
#----------
id=5900
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

cmd_name="tar"
## check
result="TRUE"
result_msg1=`echo ${msg1} | sed -e "s|${rep}|${cmd_name}|"`
result_msg2=""
err_id=""

cmd_existence_check "${cmd_name}"
exitcode=$?
if [ "${exitcode}" -eq "${EXITCODE_CMD_INVOKED_CANNOT_EXECUTE}" -o "${exitcode}" -eq "${EXITCODE_CMD_NOT_FOUND}" ];
then
	# failed command execute
	set_msg_couldnotchk
	set_err_id 1
elif [ "${exitcode}" -ne "${FUNC_RET_SUCCESS}" ];
then
	result_summary="FALSE"
	result="FALSE"
	result_msg2=`echo ${msg2} | sed -e "s|${rep}|${cmd_name}|"`
	set_err_id 2
fi

# output result
output_msg "${result_msg1}" "${result_msg2}"

#----------
#  zip
#----------

#--------------------
# ID : 6000
#  Recovery action check for deactivation failure
#--------------------
id=6000
clp_logwrite $LOG_EXTENTION "-----$id-----" $LOG_FNAME
## get message
read_msg_file

## check
# enum types
xml_path="/root/resource/types"
clpcfget_exe "e"
if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
then
	for types_name in ${cfget};
	do
		# enum resource
		xml_path="/root/resource/${types_name}"
		clpcfget_exe "e"
		if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
		then
			for resource_name in ${cfget};
			do
				result="TRUE"
				result_msg1=`echo ${msg1} | sed -e "s|${rep}|(${resource_name})|"`
				result_msg2=""

				start_dependency_check "${types_name}" "${resource_name}"
				exitcode=$?
				if [ "${exitcode}" -eq "${FUNC_RET_SUCCESS}" ];
				then
					# get value (deact action)
					xml_path="/root/resource/${types_name}@${resource_name}/deact/action"
					clpcfget_exe "g" "${types_name}"
					if [ "${cfget_exitcode}" -eq "${CFGET_SUCCESS}" ];
					then
						if [ "${cfget}" -eq 0 -o "${cfget}" -eq 1 ];
						then
							result_summary="FALSE"
							result="FALSE"
							result_msg2=${msg2}
							set_err_id 1
						fi
					else
						# failed to enum resource
						set_msg_couldnotchk
					fi
				else
					# not check target
					set_msg_notchktarget
				fi

				# output result
				output_msg "${result_msg1}" "${result_msg2}"
			done
		else
			# failed to enum resource
			set_msg_couldnotchk "${types_name}"

			# output result
			output_msg "${result_msg1}" "${result_msg2}"
		fi
	done
elif [ "${cfget_exitcode}" -ne "${CFGET_ERR_XMLEXIST}" ];	# xmlパスが存在しない旨のエラーは無視(エラーコードは要確認)
then
	# failed to enum types
	set_msg_couldnotchk
	result_msg1=`echo ${msg1} | sed -e "s|${rep}||"`

	# output result
	output_msg "${result_msg1}" "${result_msg2}"
fi

#--------------------
# ID : 2
#  end time
#--------------------
id=2
read_msg_file
output_msg_datetime

#--------------------
# ID : 0
#  result summary
#--------------------
id=0
read_msg_file
output_msg_result_summary

clp_logwrite $LOG_EXTENTION "clpvalidation end" $LOG_FNAME

# end
