#! /bin/sh

# --- Options --- #
readonly OPT_HLP1="--help"
readonly OPT_HLP2="-h"
readonly OPT_ADD1="--add"
readonly OPT_ADD2="-a"
readonly OPT_DEL1="--delete"
readonly OPT_DEL2="-d"
readonly OPT_DEL3="--del"
readonly OPT_LST1="--list"
readonly OPT_LST2="-l"
readonly OPT_CHK1="--check"
readonly OPT_CHK2="-c"
readonly OPT_CHKP="--check-pp"
readonly OPT_RMVP="--remove-pp"
readonly PAD_0=""
readonly PAD_1=" "
readonly PAD_2="  "
readonly PAD_3="   "
readonly PAD_4="    "
readonly PAD_5="     "
readonly PAD_6="      "

# --- Type of context --- #
readonly TYPE_CLP_OPT="usr_t"
readonly TYPE_CLP_BIN="bin_t"
readonly TYPE_CLP_LIB="lib_t"
readonly TYPE_CLP_DRIVER="modules_object_t"
readonly TYPE_CLP_SERVICE="usr_t"
readonly TYPE_INITRC_EXEC="initrc_exec_t"
readonly TYPE_SYSTEMD_SVC="systemd_unit_file_t"
readonly TYPE_SYSTEMD_SH="bin_t"

# --- Command path or name --- #
readonly SEMANAGE="semanage"
readonly RESTORECON="restorecon"
readonly AWK="awk"
readonly GREP="grep"

# --- This command name / Return codes / Messages --- #
readonly CMD_NAME="clpselctrl.sh"
readonly RET_CMD_SUCCESS=0
readonly MSG_CMD_SUCCESS="${CMD_NAME} : Command succeeded."
readonly RET_CMD_FAILED=1
readonly MSG_CMD_FAILED="${CMD_NAME} : Command failed."
readonly RET_ERR_INTERNAL=2
readonly MSG_ERR_INTERNAL="${CMD_NAME} : Internal error."
readonly RET_ERR_INVALID_OPTION=3
readonly MSG_ERR_INVALID_OPTION="${CMD_NAME} error : Invalid option."
readonly RET_ERR_GREP=4
readonly MSG_ERR_GREP="${CMD_NAME} error : ${GREP} command failed."
readonly RET_ERR_NOT_ROOT=5
readonly MSG_ERR_NOT_ROOT="${CMD_NAME} error : Log in as root."
readonly RET_ERR_CMD_NOT_FOUND=6
readonly MSG_ERR_CMD_NOT_FOUND="${CMD_NAME} error : '${SEMANAGE}' and '${RESTORECON}' commands are required."
readonly MSG_CHK_OK="OK"
readonly MSG_CHK_NG="NG"
readonly RET_CHK_OK=${RET_CMD_SUCCESS}
readonly RET_CHK_NG=${RET_CMD_FAILED}
readonly RET_NOT_EXIST=${RET_CMD_SUCCESS}
readonly RET_YES_EXIST=${RET_CMD_FAILED}
readonly MSG_OBS_CHK_1="${CMD_NAME} : Please execute 'semodule -r"
readonly MSG_OBS_CHK_2="' to remove these SELinux policy packages."
readonly MSG_OBS_RMV_1="${CMD_NAME} : SELinux policy packages ("
readonly MSG_OBS_RMV_2=") have been removed."

# --- Module names --- #
readonly CLPKA="clpka"
readonly CLPKHB="clpkhb"
readonly LISCAL="liscal"


####################################################################
# Print usage
####################################################################

PRINT_USAGE()
{
	echo ""
	echo "Usage:"
	echo "  ${CMD_NAME} ${OPT_ADD1} ${PAD_3}: Add SELinux rules for the cluster"
	echo "  ${CMD_NAME} ${OPT_DEL1} ${PAD_0}: Delete SELinux rules for the cluster"
	echo "  ${CMD_NAME} ${OPT_LST1} ${PAD_2}: List added SELinux rules for the cluster"
	echo "  ${CMD_NAME} ${OPT_CHK1} ${PAD_1}: Check if all of the SELinux rules for the cluster have been added or not"
	echo "  ${CMD_NAME} ${OPT_CHKP} ${PAD_0}: Check if SELinux policy packages for the cluster have been loaded or not"
	echo "  ${CMD_NAME} ${OPT_RMVP} ${PAD_0}: Remove SELinux policy packages for the cluster"
	echo ""
}


####################################################################
# Check if commands are executable or not.
####################################################################

CHECK_COMMAND()
{
	local ret_lack=0
	local ret_exec=0

	# --- Check if commands exist or not --- #
	which ${SEMANAGE} ${RESTORECON}  > /dev/null 2>&1
	ret_lack=$?
	if [ ${ret_lack} -ne 0 ]
	then
		echo  "${MSG_ERR_CMD_NOT_FOUND}"  1>&2
		return ${RET_ERR_CMD_NOT_FOUND}
	fi

	# --- Check if the command is executable or not --- #
	${SEMANAGE} fcontext --list  > /dev/null
	ret_exec=$?
	if [ ${ret_exec} -ne 0 ]
	then
		echo  "${MSG_ERR_NOT_ROOT}"  1>&2
		return ${RET_ERR_NOT_ROOT}
	fi

	return ${RET_CMD_SUCCESS}
}

####################################################################
# Add/Delete rules of context type for cluster services and drivers
# and apply context (user and type) to them.
# ----- directory -----
# OPT         =/opt/$LCORP/$LPRODUCT
# LOGOPT      =$OPT/log
# ETCOPT      =$OPT/etc
# CPSYSTEMDOPT=$ETCOPT/systemd
####################################################################

CHANGE_RULES()
{
	local opt_act="$1"
	local ret_grep=0
	local ret_rule=${RET_CMD_SUCCESS}
	local ret_cmd=0
	local ret_chk=0

	# --- Check if the command is executable or not --- #
	CHECK_COMMAND
	ret_chk=$?
	if [ ${ret_chk} -ne ${RET_CMD_SUCCESS} ]
	then
		return ${ret_chk}
	fi

	# --- Check if the SELinux PP modules are loaded or not --- #
	CHECK_PP_MODULES "${OPT_CHK1}"
	ret_chk=$?
	if [ ${ret_chk} -ne ${RET_CMD_SUCCESS} ]
	then
		return ${ret_chk}
	fi

	# --- Delete obsoleted rules --- #
	${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_SERVICE} '/opt/nec/clusterpro/etc/init.d(/.*)?'  > /dev/null 2>&1
	${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_INITRC_EXEC} '/opt/nec/clusterpro/etc/init.d/.*'     > /dev/null 2>&1
	${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_SERVICE} '/opt/nec/clusterpro/etc/systemd(/.*)?'   > /dev/null 2>&1
	${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_DRIVER}  '/opt/nec/clusterpro/drivers(/.*)?(/distribution)?(/.*)?(/.*\.ko)?'  > /dev/null 2>&1

	# ------ 1. first rule (./*) ------ #

	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_OPT} '/opt/nec/clusterpro(/.*)?'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_CLP_OPT} '/opt/nec/clusterpro(/.*)?'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_OPT} '/opt/nec/clusterpro(/.*)?'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro(/.*)?'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ 2. rule (etc/systemd/*.service) ------ #
	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_SYSTEMD_SVC} '/opt/nec/clusterpro/etc/systemd/.*\.service'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_SYSTEMD_SVC} '/opt/nec/clusterpro/etc/systemd/.*\.service'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_SYSTEMD_SVC} '/opt/nec/clusterpro/etc/systemd/.*\.service'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/etc/systemd/.*\.service'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ 3. rule (etc/systemd/*.sh) ------ #
	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_SYSTEMD_SH}  '/opt/nec/clusterpro/etc/systemd/.*\.sh'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_SYSTEMD_SH}  '/opt/nec/clusterpro/etc/systemd/.*\.sh'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_SYSTEMD_SH}  '/opt/nec/clusterpro/etc/systemd/.*\.sh'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/etc/systemd/.*\.sh'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ 4. rule (drivers) ------ #
	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_DRIVER}  '/opt/nec/clusterpro/drivers(/.*)?'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_CLP_DRIVER}  '/opt/nec/clusterpro/drivers(/.*)?'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_DRIVER}  '/opt/nec/clusterpro/drivers(/.*)?'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/drivers(/.*)?'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ 5. rule (bin/*, */bin/*) ------ #
	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_BIN} '/opt/nec/clusterpro/(.*/)?bin(/.*)?'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_CLP_BIN} '/opt/nec/clusterpro/(.*/)?bin(/.*)?'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_BIN} '/opt/nec/clusterpro/(.*/)?bin(/.*)?'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/(.*/)?bin(/.*)?'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ 6. rule (lib/*, */lib/*) ------ #
	if [ "${opt_act}" = "${OPT_ADD1}" ]
	then
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_LIB} '/opt/nec/clusterpro/(.*/)?lib(/.*)?'  > /dev/null 2>&1
		${SEMANAGE} fcontext ${OPT_ADD1} --ftype a --type ${TYPE_CLP_LIB} '/opt/nec/clusterpro/(.*/)?lib(/.*)?'
		ret_cmd=$?
		if [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	else
		${SEMANAGE} fcontext ${OPT_DEL1} --ftype a --type ${TYPE_CLP_LIB} '/opt/nec/clusterpro/(.*/)?lib(/.*)?'  > /dev/null 2>&1
		ret_cmd=$?
		if [ ${ret_cmd} -eq 1 ]
		then
			${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/(.*/)?lib(/.*)?'  > /dev/null
			ret_grep=$?
			if [ ${ret_grep} -ne 1 ]
			then
				ret_rule=${RET_CMD_FAILED}
			fi
		elif [ ${ret_cmd} -ne 0 ]
		then
			ret_rule=${RET_CMD_FAILED}
		fi
	fi

	# ------ Apply rules to /opt/nec/clusterpro  ------ #
	${RESTORECON} -R -F /opt/nec/clusterpro

	# --- Return --- #
	if [ ${ret_rule} -ne ${RET_CMD_SUCCESS} ]
	then
		echo  "${MSG_CMD_FAILED}"
		return ${RET_CMD_FAILED}
	fi
	echo  "${MSG_CMD_SUCCESS}"
	return ${RET_CMD_SUCCESS}
}


####################################################################
# List rules
####################################################################

LIST_RULES()
{
	local ret_chk=0

	# --- Check if the command is executable or not --- #
	CHECK_COMMAND
	ret_chk=$?
	if [ ${ret_chk} -ne ${RET_CMD_SUCCESS} ]
	then
		return ${ret_chk}
	fi

	# --- List of rules --- #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F '/opt/nec/clusterpro'
	return ${RET_CMD_SUCCESS}
}


####################################################################
# Check rules are added or not for the validation check
####################################################################

CHECK_RULES()
{
	local ret_grep=0
	local ret_rule=0
	local ret_chk=0

	# --- Check if the command is executable or not --- #
	CHECK_COMMAND
	ret_chk=$?
	if [ ${ret_chk} -ne ${RET_CMD_SUCCESS} ]
	then
		return ${ret_chk}
	fi

	# --- Check rules --- #
	ret_rule=${RET_CHK_OK}

	# ------ 1. rule (./*) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro(/.*)?'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# ------ 2. rule (etc/systemd/*.service) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/etc/systemd/.*\.service'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# ------ 3. rule (etc/systemd/*.sh) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/etc/systemd/.*\.sh'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# ------ 4. rule (drivers) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/drivers(/.*)?'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# ------ 5. rule (bin/*, */bin/*) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/(.*/)?bin(/.*)?'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# ------ 6. rule (lib/*, */lib/*) ------ #
	${SEMANAGE} fcontext --list | ${AWK} '{print $1}' | ${GREP} -F -x '/opt/nec/clusterpro/(.*/)?lib(/.*)?'  > /dev/null
	ret_grep=$?
	if [ ${ret_grep} -ne 0 ]
	then
		ret_rule=${RET_CHK_NG}
	fi

	# --- Print result --- #
	if [ ${ret_rule} -eq ${RET_CHK_NG} ]
	then
		echo  "${MSG_CHK_NG}"
		return ${RET_CHK_NG}
	fi
	echo  "${MSG_CHK_OK}"
	return ${RET_CHK_OK}
}


####################################################################
# Check if obsolete SELinux Policy Package modules exist or not.
# Remove them if remove is specified.
####################################################################

CHECK_PP_MODULES()
{
	local opt_act="$1"
	local ret_grep=0
	local ret_chk=0
	local ret_cmd=0
	local mod_exist=0
	local mod_list=""

	# --- Check if the command is executable or not --- #
	if [ "${opt_act}" != "${OPT_CHK1}" ]
	then
		CHECK_COMMAND
		ret_chk=$?
		if [ ${ret_chk} -ne ${RET_CMD_SUCCESS} ]
		then
			return ${ret_chk}
		fi
	fi

	# --- Check modules are loaded --- #
	mod_exist=${RET_NOT_EXIST}

	for module_name in ${CLPKA} ${CLPKHB} ${LISCAL}
	do
		${SEMANAGE} module --list | ${AWK} '{print $1}' | ${GREP} -F -x "${module_name}"  > /dev/null
		ret_grep=$?
		if [ ${ret_grep} -eq 0 ]
		then
			mod_list=`echo -n "${mod_list} ${module_name}"`
			mod_exist=${RET_YES_EXIST}
		fi
	done

	# --- Print result / Remove policy packages --- #
	if [ ${mod_exist} -eq ${RET_YES_EXIST} ]
	then
		if [ "${opt_act}" = "${OPT_RMVP}" ]
		then
			${SEMANAGE} module --remove ${mod_list}
			ret_cmd=$?
			if [ ${ret_cmd} -eq 0 ]
			then
				echo "${MSG_OBS_RMV_1}${mod_list} ${MSG_OBS_RMV_2}"
				return ${RET_CMD_SUCCESS}
			fi
			return ${ret_cmd}
		else
			echo "${MSG_OBS_CHK_1}${mod_list}${MSG_OBS_CHK_2}"
			return ${RET_CMD_SUCCESS}
		fi
	fi
	return ${RET_CMD_SUCCESS}
}


####################################################################
# Main Routine
####################################################################

if [ $# -ne 1 ]
then
	echo "${MSG_ERR_INVALID_OPTION}" 1>&2
	PRINT_USAGE
	exit ${RET_CMD_FAILED}
fi

for i in $@
do
	case "$i" in
	"${OPT_ADD1}" | "${OPT_ADD2}" )
		CHANGE_RULES "${OPT_ADD1}"
		ret=$?
		exit ${ret}
		;;
	"${OPT_DEL1}" | "${OPT_DEL2}" | "${OPT_DEL3}" )
		CHANGE_RULES "${OPT_DEL1}"
		ret=$?
		exit ${ret}
		;;
	"${OPT_LST1}" | "${OPT_LST2}" )
		LIST_RULES
		ret=$?
		exit ${ret}
		;;
	"${OPT_CHK1}" | "${OPT_CHK2}" )
		CHECK_RULES
		ret=$?
		exit ${ret}
		;;
	"${OPT_CHKP}" )
		CHECK_PP_MODULES "${OPT_CHKP}"
		ret=$?
		exit ${ret}
		;;
	"${OPT_RMVP}" )
		CHECK_PP_MODULES "${OPT_RMVP}"
		ret=$?
		exit ${ret}
		;;
	"${OPT_HLP1}" | "${OPT_HLP2}" )
		PRINT_USAGE
		exit ${RET_CMD_SUCCESS}
		;;
	* )
		echo "${MSG_ERR_INVALID_OPTION}" 1>&2
		PRINT_USAGE
		exit ${RET_CMD_FAILED}
		;;
	esac
done

echo "${MSG_ERR_INVALID_OPTION}" 1>&2
PRINT_USAGE
exit ${RET_CMD_FAILED}
