#! /bin/bash

#------- static variables -------#

readonly SAVED_INNER_VERSION="530101"
readonly XPATH_CFCONV_CONVERTED_FLAG="cfconv/converted/v${SAVED_INNER_VERSION}"

#------- command return value -------#
readonly RET_CMD_SUCCESS=0
readonly RET_CMD_FAILED=1

readonly RET_CMD_NOTFOUND=2

readonly RET_TRUE=1
readonly RET_FALSE=0

#------- message -------#
readonly MSG_CMD_SUCCESS="Command succeeded."
readonly MSG_CMD_FAILED="Command failed."

readonly MSG_INIT_PASSWORD_OPE="Password for Operation has been initialized."
readonly MSG_INIT_PASSWORD_REF="Password for Reference has been initialized."
readonly MSG_SET_PASSWORD="Please set the password again by using Cluster WebUI."

readonly MSG_ERR_DIR="Not available in this directory."
readonly MSG_ERR_NOT_ROOT="Log in as root."
readonly MSG_ERR_NOT_FOUNTD_OUTPATH="The specified output-path does not exist."
readonly MSG_ERR_NOT_FOUNTD_CONFIG="Could not opened the configuration file. Check if the configuration file exists on the specified path."
readonly MSG_ERR_CONF="Invalid configuration file."
readonly MSG_ERR_OLD_CONF="The version of this configuration data file is not supported. Convert it with Builder for offline use (internal version 3.3.5-1), then retry."
readonly MSG_ERR_CMD_FAILED=": Command failed. code:"
readonly MSG_ERR_FILE="Failed to save the configuration file."

#------- clpcfget ret code -------#
readonly RET_CFGET_DEFINED=0
readonly RET_CFGET_NOT_ROOT=2
readonly RET_CFGET_NOT_CONFIG=6
readonly RET_CFGET_NOT_FOUND=11
readonly RET_CFGET_ERROR=255

#------- clpcfset ret code -------#
readonly RET_CFSET_SUCCESS=0
readonly RET_CFSET_ERR_USER=2
readonly RET_CFSET_ERR_CONF=3
readonly RET_CFSET_ERR_FILE=7
readonly RET_CFSET_NOT_FOUND=9

#------- other  -------#
readonly EXCL_NAME_PFX=excl
readonly EXCL_TYPE_NORMAL=normal
readonly EXCL_TYPE_HIGH=high

#------- variables -------#

opt_inpath=""
opt_outpath=""
conf_path=""
conf_ver=""
curdir=""
backup_conf_flag=0
inpath_curdir_flag=0
conf_ver_no1=0
conf_ver_no2=0
conf_ver_no3=0
conf_ver_no6=0
conf_serveros=""
init_password_ope=0
init_password_ref=0
cfset_exec_flag=0

cfget_val=""
cfget_list=""
excl_normal_groups=""
excl_high_groups=""
excl_groups=""
excl_cnt=0

####################################################################
#                                                                  #
# Common Functions                                                 #
#                                                                  #
####################################################################

####################################################################
# name   : DEBUG TARCE
####################################################################
TRACE()
{
if [ "${CLP_DEBUG}" = "1" ]; then
	echo "[trace] $1"
fi
}

####################################################################
# name   : PRINT_USAGE
####################################################################
PRINT_USAGE()
{
	echo ""
	echo "Usage: clpcfconv.sh -i input-path [-o output-path]"
	echo ""
	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CFGET_GET
# arg#1  : xpath
# out    : cfget_val
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_NOTFOUND : Not found.
#  RET_CMD_FAILED : clpcfget failed. 
# 
# description:
#  execute clpcfget -g
#####################################################################
CFGET_GET()
{
	cfget_val=""

	local cfget_xpath=$1
	local ret_cfget=0

	local CMD_NAME=clpcfget
	local cmd_line="${CMD_NAME} -g ${cfget_xpath} -i ${conf_path}"
TRACE "CFGET_GET cmd_line=${cmd_line}"

	#------- execute clpcfget -------
	cfget_val=`${cmd_line} 2>&1`
	ret_cfget=$?
TRACE "CFGET_GET ret_cfget=${ret_cfget} cfget_val=${cfget_val}"

	#------- check exit code -------
	case ${ret_cfget} in
	${RET_CFGET_DEFINED} )
		return ${RET_CMD_SUCCESS}
		;;
	${RET_CFGET_NOT_FOUND} )
		return ${RET_CMD_NOTFOUND}
		;;
	${RET_CFGET_NOT_ROOT} )
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		;;
	${RET_CFGET_NOT_CONFIG} )
		echo "${MSG_ERR_NOT_FOUNTD_CONFIG}" 1>&2
		;;
	* )
		echo "${CMD_NAME} ${MSG_ERR_CMD_FAILED}${ret_cfget}" 1>&2
		;;
	esac

	return ${RET_CMD_FAILED}
}

#####################################################################
# name   : CFGET_ENUM
# arg#1  : xpath
# out    : cfget_list
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_NOTFOUND : Not found.
#  RET_CMD_FAILED : clpcfget failed. 
# 
# description:
#  execute clpcfget -e
#####################################################################
CFGET_ENUM()
{
	cfget_list=""

	local enum_xpath=$1
	local ret_cfget=0

	local CMD_NAME=clpcfget
	local cmd_line="${CMD_NAME} -e ${enum_xpath} -i ${conf_path}"
	local cmd_ret=""
TRACE "CFGET_ENUM cmd_line=${cmd_line}"

	#------- execute clpcfget -------
	cfget_list=`${cmd_line} 2>/dev/null`
	ret_cfget=$?
TRACE "CFGET_ENUM ret_cfget=${ret_cfget}"

	#------- check exit code -------
	case ${ret_cfget} in
	${RET_CFGET_DEFINED} )
		return ${RET_CMD_SUCCESS}
		;;
	${RET_CFGET_NOT_FOUND} )
		return ${RET_CMD_NOTFOUND}
		;;
	${RET_CFGET_NOT_ROOT} )
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		;;
	${RET_CFGET_NOT_CONFIG} )
		echo "${MSG_ERR_NOT_FOUNTD_CONFIG}" 1>&2
		;;
	* )
		echo "${CMD_NAME} ${MSG_ERR_CMD_FAILED}${ret_cfget}" 1>&2
		;;
	esac

	return ${RET_CMD_FAILED}
}

####################################################################
# name   : EXECUTE_CFSET
# arg#1  : add or del.
# arg#2  : Tag name.
# arg#3  : value. (for add)
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Execute the cfset command.
####################################################################
EXECUTE_CFSET()
{
TRACE "EXECUTE_CFSET $1 $2 $3"

	local cfset_func=$1
	local cfset_tag=$2
	local cfset_val=$3

	local ret_cfset=0

	local CMD_NAME=clpcfset
	local cmd_line="${CMD_NAME} --internal-use ${cfset_func} clsparam ${cfset_tag}"
	local cmd_ret=""

	#------- execute clpcfset.exe -------
	if [ "${cfset_func}" = "del" ]; then
TRACE "EXECUTE_CFSET cmd_line=${cmd_line}"
		cmd_ret=`${cmd_line} 2>&1`
	else
TRACE "EXECUTE_CFSET cmd_line=${cmd_line} ${cfset_val}"
		cmd_ret=`${cmd_line} "${cfset_val}" 2>&1`
	fi
	ret_cfset=$?
TRACE "EXECUTE_CFSET ret_cfset=${ret_cfset}"

	#------- check exit code -------
	case ${ret_cfset} in
	${RET_CFSET_SUCCESS} )
		cfset_exec_flag=1
		return ${RET_CMD_SUCCESS}
		;;
	${RET_CFSET_ERR_USER} )
		echo "${MSG_ERR_NOT_ROOT}" 1>&2
		;;
	${RET_CFSET_ERR_CONF} )
		echo "${MSG_ERR_CONF}" 1>&2
		;;
	${RET_CFSET_ERR_FILE} )
		echo "${MSG_ERR_FILE}" 1>&2
		;;
	${RET_CFSET_NOT_FOUND} )
		if [ "${cfset_func}" = "del" ]; then
			return ${RET_CMD_SUCCESS}
		fi
		echo "${CMD_NAME} ${MSG_ERR_CMD_FAILED}${ret_cfset}" 1>&2
		;;
	* )
		echo "${CMD_NAME} ${MSG_ERR_CMD_FAILED}${ret_cfset}" 1>&2
		;;
	esac

	return ${RET_CMD_FAILED}
}

####################################################################
# name   : CFGET_CFSET_VAL
# arg#1  : cfget tag name.
# arg#2  : cfset tag name.
# arg#3  : del or omit. When 'del' is specified, the tag name will be deleted.
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED  : Failed.
# 
# description:
#  Set the value getted with the cfget command.
####################################################################
CFGET_CFSET_VAL()
{
TRACE "CFGET_CFSET_VAL $1 $2 $3"
	local getset_cfget_xpath="/root/$1"
	local getset_cfset_tag=$2
	local getset_cfset_deltag=""
	if [ "$3" = "del" ]; then
		getset_cfset_deltag=$1
	fi
	local ret_func=0

	# ------- get value -------
	CFGET_GET "${getset_cfget_xpath}"
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	getset_val="${cfget_val}"

	# ------- set value -------
	EXECUTE_CFSET add "${getset_cfset_tag}" "${getset_val}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	if [ "${getset_cfset_deltag}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi
	# ------- del tag -------
	EXECUTE_CFSET del "${getset_cfset_deltag}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

####################################################################
# name   : CHK_OPT
# arg#1  : Command option
# return : (none)
####################################################################
CHK_OPT()
{
	if [ "${opt_opt}" = "-i" ]; then
		if [ "${opt_inpath}" != "" ]; then
			return ${RET_CMD_FAILED}
		fi
		opt_inpath="$1"
		opt_opt=""
		return ${RET_CMD_SUCCESS}
	fi

	if [ "${opt_opt}" = "-o" ]; then
		if [ "${opt_outpath}" != "" ]; then
			return ${RET_CMD_FAILED}
		fi
		opt_outpath="$1"
		opt_opt=""
		return ${RET_CMD_SUCCESS}
	fi

	if [ "$1" = "-i" ]; then
		opt_opt="$1"
		return ${RET_CMD_SUCCESS}
	fi

	if [ "$1" = "-o" ]; then
		opt_opt="$1"
		return ${RET_CMD_SUCCESS}
	fi

	return ${RET_CMD_FAILED}
}

#----------------------------------------------------------------------
#	CHK_CONF
#----------------------------------------------------------------------

#####################################################################
# name   : ISNUMERIC
# arg#1  : string
# return : 
#  RET_TRUE : string is numeric.
#  RET_FALSE : string is not numeric.
# description:
#  is numeric
#####################################################################
ISNUMERIC()
{
	local str="$1"

	expr "${str}" + 1 >/dev/null 2>&1
	if [ $? -ge 2 ]; then
		return ${RET_FALSE}
	fi

	return ${RET_TRUE}
}

#####################################################################
# name   : CHK_VER
# out    : conf_ver_no1, conf_ver_no2, conf_ver_no3, conf_ver_no6 (numeric)
# arg#1  : version
# return : RET_CMD_SUCCESS
# description:
#  check version
#####################################################################
CHK_VER()
{
	local argver="$1"

	#------- check length  -------
	ver_len=`expr length ${argver}`
TRACE "CHK_VER ver_len=${ver_len}"

	if [ ${ver_len} -lt 3 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- check numeric  -------
	ISNUMERIC ${argver}
	local ret_func=$?
	if [ ${ret_func} -eq ${RET_FALSE} ]; then
		return ${RET_CMD_SUCCESS}
	fi

	conf_ver_no1=${argver:0:1}
	conf_ver_no2=${argver:0:2}
	conf_ver_no3=${argver:0:3}
	if [ ${ver_len} -ge 6 ]; then
		conf_ver_no6=${argver:0:6}
	fi
TRACE "CHK_VER conf_ver_no1=${conf_ver_no1} conf_ver_no2=${conf_ver_no2} conf_ver_no3=${conf_ver_no3} conf_ver_no6=${conf_ver_no6}" 

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : COPY_CONF
# arg#1  : src file
# arg#2  : dest path
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : copy failed. 
# 
# description:
#  copy clp.conf. (overwrite)
#####################################################################
COPY_CONF()
{
	local CMD_NAME=cp
	\cp -p -f $1 $2 > /dev/null
	local ret_copy=$?
	if [ ${ret_copy} -ne ${RET_CMD_SUCCESS} ]; then
		echo "${CMD_NAME} ${MSG_ERR_CMD_FAILED}${ret_copy}" 1>&2
		return ${RET_CMD_FAILED}
	fi

TRACE "COPY_CONF copy $1 -> $2" 
	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : RESTORE_CONF
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Restore failed. 
# 
# description:
#  restore clp.conf from clp.conf.prev.
#####################################################################
RESTORE_CONF()
{
	if [ ! -e ./clp.conf.prev ]; then
		return ${RET_CMD_SUCCESS}
	fi

	COPY_CONF ./clp.conf.prev ./clp.conf
	local ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi
	rm -f ./clp.conf.prev > /dev/null 2>&1

TRACE "RESTORE_CONF clp.conf.prev -> clp.conf"
	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CHK_CONF
# in     : curdir
# out    : conf_path, conf_ver, conf_serveros, backup_conf_flag, inpath_curdir_flag
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : not found clp.conf or not support clp.conf.
# 
# description:
#  Check input clp.conf.
#####################################################################
CHK_CONF()
{
	local ret_func=0
	local fullchk=""

	#------- check opt_outpath -------
	if [ "${opt_outpath}" = "." ]; then
		opt_outpath=""
	fi
	if [ "${opt_outpath}" = "${curdir}" ]; then
		opt_outpath=""
	fi

	if [ "${opt_outpath}" != "" ]; then
		if [ ! -d "${opt_outpath}" ]; then
			echo ${MSG_ERR_NOT_FOUNTD_OUTPATH} 1>&2
			return ${RET_CMD_FAILED}
		fi
	fi

	#------- check input clp.conf -------
	if [ "${opt_inpath}" = "." ]; then
		opt_inpath="${curdir}"
	fi

	fullchk="${opt_inpath:0:1}"
	if [ "${fullchk}" = "/" ]; then
		conf_path="${opt_inpath}/clp.conf"
	else
		conf_path="${curdir}/${opt_inpath}/clp.conf"
	fi
	if [ ! -e "${conf_path}" ]; then
		echo "${MSG_ERR_NOT_FOUNTD_CONFIG}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- get version from clp.conf -------
	CFGET_GET /root/trekking/savedinnerversion
	ret_func=$?
	if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
		return ${RET_CMD_FAILED}
	fi
	conf_ver="${cfget_val}"

	#------- check version -------
	CHK_VER "${conf_ver}"
	if [ ${conf_ver_no3} -lt 335 ]; then
		echo "${MSG_ERR_OLD_CONF}" 1>&2
		return ${RET_CMD_FAILED}
	fi

	#------- get severos from clp.conf -------
	CFGET_GET /root/all/serveros
	ret_func=$?
	if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
		return ${RET_CMD_FAILED}
	fi
	conf_serveros="${cfget_val}"

	#------- backup clp.conf on current directory  -------
	if [ -e ./clp.conf ]; then
		COPY_CONF ./clp.conf ./clp.conf.prev
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
		backup_conf_flag=1

		#------- check input path  -------
		if [ "${opt_inpath}" = "${curdir}" ]; then
			inpath_curdir_flag=1
			conf_path="${curdir}/clp.conf.prev"
			return ${RET_CMD_SUCCESS}
		fi
	fi

	#------- copy input clp.conf to current directory  -------
	COPY_CONF "${conf_path}" .
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

####################################################################
#                                                                  #
# Conv Sub Routine                                                 #
#                                                                  #
####################################################################

#----------------------------------------------------------------------
#	CONV_X335_TO_X40
#----------------------------------------------------------------------

#####################################################################
# name   : CONV_X40_BMPSIZE
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert difference bitmap size
#####################################################################
CONV_X40_BMPSIZE()
{
TRACE "CONV_X40_BMPSIZE"
	local ret_func=0

	if [ "${conf_serveros}" != "windows" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- check exist md resource -------
	CFGET_ENUM /root/resource/md
	ret_func=$?
	if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- check exist hd resource -------
	if [ ${ret_func} -eq ${RET_CMD_NOTFOUND} ]; then
		CFGET_ENUM /root/resource/hd
		ret_func=$?
		if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
			return ${RET_CMD_FAILED}
		fi
		if [ ${ret_func} -eq ${RET_CMD_NOTFOUND} ]; then
			return ${RET_CMD_SUCCESS}
		fi
	fi

	#------- set difference bitmap size (for windows) -------
	EXECUTE_CFSET add diskfltr/cpbitmapsize 1024
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	EXECUTE_CFSET add diskfltr/cphistorysize 1024
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X40_ALERT
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert alert exec (for linux)
#####################################################################
CONV_X40_ALERT()
{
TRACE "CONV_X40_ALERT"
	local ret_func=0

	if [ "${conf_serveros}" != "linux" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- check exist alert exec -------
#	CFGET_GET /root/cluster/altexec
#	ret_func=$?
#	case ${ret_func} in
#	${RET_CMD_FAILED} )
#		return ${RET_CMD_FAILED}
#		;;
#	${RET_CMD_NOTFOUND} )
#		return ${RET_CMD_SUCCESS}
#		;;
#	esac

	#------- del alert exec -------
	EXECUTE_CFSET del cluster/altexec
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X40_EXCLUSIVE
#----------------------------------------------------------------------

#####################################################################
# name   : X40_EXCLUSIVE_GET_EXCLGROUP
# arg#1  : groupname
# out    : excl_normal_groups, excl_high_groups (delimited space)
#          example: failoverA failoverB
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  get exclusive group and delete exclusive value in group
#####################################################################
X40_EXCLUSIVE_GET_EXCLGROUP()
{
TRACE "X40_EXCLUSIVE_GET_EXCLGROUP $1"
	local group_name=$1
	local ret_func=0
	local excl_val=0
	
	#------- get exclusive type -------
	CFGET_GET "/root/group@${group_name}/exclusive"
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	excl_val="${cfget_val}"

	#------- set group name -------
	case ${excl_val} in
	1 )
		if [ "${excl_normal_groups}" = "" ]; then
			excl_normal_groups="${group_name}"
		else
			excl_normal_groups="${excl_normal_groups} ${group_name}"
		fi
		;;
	2 )
		if [ "${excl_high_groups}" = "" ]; then
			excl_high_groups="${group_name}"
		else
			excl_high_groups="${excl_high_groups} ${group_name}"
		fi
		;;
	esac

	#------- delete exclusive value -------
	EXECUTE_CFSET del "group@${group_name}/exclusive"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : X40_EXCLUSIVE_SETRULE
# arg#1  : exclusive type
# in     : excl_groups
# in/out : excl_cnt
# return :
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  set exclusive rule
#####################################################################
X40_EXCLUSIVE_SETRULE()
{
	local ret_func=0

	if [ "${excl_groups}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	excl_cnt=`expr ${excl_cnt} + 1`
	local excl_name="${EXCL_NAME_PFX}${excl_cnt}"
	local excl_type=$1

	#------- set type -------
	EXECUTE_CFSET add "xclrule@${excl_name}/type" "${excl_type}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- set comment -------
	EXECUTE_CFSET add "xclrule@${excl_name}/comment"
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- set order -------
	EXECUTE_CFSET add "xclrule@${excl_name}/order" 0
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- set group name -------
	for excl_group_name in ${excl_groups}
	do
		EXECUTE_CFSET add "xclrule@${excl_name}/group@${excl_group_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X40_EXCLUSIVE
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert exclusive group
#####################################################################
CONV_X40_EXCLUSIVE()
{
TRACE "CONV_X40_EXCLUSIVE"

	local ret_func=0
	local gourp_list=""

	#------- get group list -------
	CFGET_ENUM /root/group
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	gourp_list="${cfget_list}"

	#------- get exclusive group and delete exclusive value in group -------
	excl_normal_groups=""
	excl_high_groups=""

	for grpname in ${gourp_list}
	do
		X40_EXCLUSIVE_GET_EXCLGROUP "${grpname}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	excl_cnt=0
	#------- set normal exclusive -------
	excl_groups="${excl_normal_groups}"
	X40_EXCLUSIVE_SETRULE "${EXCL_TYPE_NORMAL}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- set high exclusive -------
	excl_groups="${excl_high_groups}"
	X40_EXCLUSIVE_SETRULE "${EXCL_TYPE_HIGH}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X335_TO_X40
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert conf from X3.3.5 to X4.0
#####################################################################
CONV_X335_TO_X40()
{
TRACE "CONV_X335_TO_X40"
	local ret_func=0

	if [ ${conf_ver_no3} -ne 335 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- difference bitmap size -------
	CONV_X40_BMPSIZE
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- alert exec -------
	CONV_X40_ALERT
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- convert exclusive group -------
	CONV_X40_EXCLUSIVE
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X40_FOCNT
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert failover count method
#####################################################################
CONV_X40_FOCNT()
{
TRACE "CONV_X40_FOCNT"
	local ret_func=0

	if [ ${conf_ver_no3} -ne 335 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	if [ "${conf_serveros}" != "windows" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- set focnt method -------
	EXECUTE_CFSET add cluster/focnt/method 1
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X50_INITPASS
#----------------------------------------------------------------------

#####################################################################
# name   : X50_INITPASS_CHKDEL
# arg#1  : xpath (after /root/)
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_NOTFOUND : not set password. 
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  check password for cluster password method and delete it.
#####################################################################
X50_INITPASS_CHKDEL()
{
TRACE "X50_INITPASS_CHKDEL $1"
	local ret_func=0
	local psw_tagname=$1

	#------- check password -------
	CFGET_GET "/root/${psw_tagname}"
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_NOTFOUND}
		;;
	esac

	#------- delete password -------
	EXECUTE_CFSET del "${psw_tagname}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_INITPASS
# out    : init_password_ope, init_password_ref
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  initialize password for cluster password method.
#####################################################################
CONV_X50_INITPASS()
{
TRACE "CONV_X50_INITPASS"
	local ret_func=0

	if [ ${conf_ver_no1} -gt 4 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- password for operation -------
	X50_INITPASS_CHKDEL webmgr/security/adminpwd
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_SUCCESS} )
		init_password_ope=1
		;;
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	esac

	#------- password for reference -------
	X50_INITPASS_CHKDEL webmgr/security/userpwd
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_SUCCESS} )
		init_password_ref=1
		;;
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	esac

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X50_HTTPWAUTH
#----------------------------------------------------------------------

#####################################################################
# name   : X50_HTTPWAUTH_SETAUTH
# arg#1  : httpw monitor name
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  set authmethod for each httpw
#####################################################################
X50_HTTPWAUTH_SETAUTH()
{
TRACE "X50_HTTPWAUTH_SETAUTH $1"
	local ret_func=0
	local httpw_name=$1
	local httpw_tag=""
	local httpw_authmethod=0

	if [ "${httpw_name}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- set tagname -------
	case "${conf_serveros}" in
	"windows" )
		httpw_tag=agentparam
		;;
	"linux" )
		httpw_tag=parameters
		;;
	* )
		return ${RET_CMD_SUCCESS}
		;;
	esac

	#------- check exist username -------
	CFGET_GET "/root/monitor/httpw@${httpw_name}/${httpw_tag}/username"
	ret_func=$?
	if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
		return ${RET_CMD_FAILED}
	fi

	if [ ${ret_func} -eq ${RET_CMD_SUCCESS} ]; then
		if [ "${cfget_val}" != "" ]; then
			httpw_authmethod=1
		fi
	fi

	#------- set authmethod -------
	EXECUTE_CFSET add "monitor/httpw@${httpw_name}/${httpw_tag}/authmethod" "${httpw_authmethod}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_HTTPWAUTH
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert httpw authmethod
#####################################################################
CONV_X50_HTTPWAUTH()
{
TRACE "CONV_X50_HTTPWAUTH"

	local ret_func=0
	local rsc_list=""

	if [ ${conf_ver_no2} -ne 42 -a ${conf_ver_no2} -ne 43 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- get httpw list -------
	CFGET_ENUM /root/monitor/httpw
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	rsc_list="${cfget_list}"

	#------- set authmethod for each httpw-------
	for rsc_name in ${rsc_list}
	do
		X50_HTTPWAUTH_SETAUTH "${rsc_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X50_AWSDNS
#----------------------------------------------------------------------

#####################################################################
# name   : X50_AWSDNS_SETWAITSYNC
# arg#1  : awsdns resource name
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  set waitinsync for each awsdns
#####################################################################
X50_AWSDNS_SETWAITSYNC()
{
TRACE "X50_AWSDNS_SETWAITSYNC $1"
	local ret_func=0
	local awsdns_name=$1

	if [ "${awsdns_name}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- set waitinsync -------
	EXECUTE_CFSET add "resource/awsdns@${awsdns_name}/parameters/waitinsync" 0
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_AWSDNS
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert awsdns waitinsync
#####################################################################
CONV_X50_AWSDNS()
{
TRACE "CONV_X50_AWSDNS"

	local ret_func=0
	local rsc_list=""

	if [ ${conf_ver_no1} -ne 4 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- get awsdns list -------
	CFGET_ENUM /root/resource/awsdns
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	rsc_list="${cfget_list}"

	#------- set waitinsync for each awsdns-------
	for rsc_name in ${rsc_list}
	do
		X50_AWSDNS_SETWAITSYNC "${rsc_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X50_SERVICEW
#----------------------------------------------------------------------

#####################################################################
# name   : X50_SERVICEW_SETNAME
# arg#1  : servicew resource name
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  set name for each servicew
#####################################################################
X50_SERVICEW_SETNAME()
{
TRACE "X50_SERVICEW_SETNAME $1"
	local ret_func=0
	local servicew_name=$1
	local conv_servicew=0
	local servicew_target=""
	local service_name=""

	if [ "${servicew_name}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- check servicew polling -------
	CFGET_GET "/root/monitor/servicew@${servicew_name}/polling/timing"
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_SUCCESS} )
		if [ "${cfget_val}" = "1" ]; then
			conv_servicew=1
		fi
		;;
	${RET_CMD_NOTFOUND} )
		conv_servicew=1
		;;
	* )
		return ${RET_CMD_FAILED}
		;;
	esac

	if [ ${conv_servicew} -eq 0 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- get servicew target -------
	CFGET_GET /root/monitor/servicew@${servicew_name}/target
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	servicew_target="${cfget_val}"

	#------- get service name -------
	CFGET_GET "/root/resource/service@${servicew_target}/parameters/name"
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	service_name="${cfget_val}"

	#------- set servicew name -------
	EXECUTE_CFSET add "monitor/servicew@${servicew_name}/parameters/name" "${service_name}"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_SERVICEW
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert servicew name
#####################################################################
CONV_X50_SERVICEW()
{
TRACE "CONV_X50_SERVICEW"

	local ret_func=0
	local rsc_list=""

	if [ ${conf_ver_no1} -gt 4 ]; then
		return ${RET_CMD_SUCCESS}
	fi
	if [ "${conf_serveros}" != "windows" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- get servicew list -------
	CFGET_ENUM /root/monitor/servicew
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	rsc_list="${cfget_list}"

	#------- set name for each servicew -------
	for rsc_name in ${rsc_list}
	do
		X50_SERVICEW_SETNAME "${rsc_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X50_DECLUTTERING
#----------------------------------------------------------------------

#----------------------------------------------------------------------
#	CONV_X50_JVMLB
#----------------------------------------------------------------------

#####################################################################
# name   : X50_JRAW_DELLB
# arg#1  : jraw resource name
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  delete load balancer linkage settings for jvm
#  (in enabledelayedexpansion)
#####################################################################
X50_JRAW_DELLB()
{
TRACE "X50_JRAW_DELLB $1"
	local ret_func=0
	local jraw_name=$1

	if [ "${jraw_name}" = "" ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- del lb -------
	EXECUTE_CFSET del "monitor/jraw@${jraw_name}/parameters/lb"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_JVMLB
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  delete load balancer settings
#####################################################################
CONV_X50_JVMLB()
{
TRACE "CONV_X50_JVMLB"
	local ret_func=0
	local rsc_list=""

	#------- del jra lb -------
	EXECUTE_CFSET del jra/lb
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- get jraw list -------
	CFGET_ENUM /root/monitor/jraw
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	rsc_list="${cfget_list}"

	#------- del lb for each jraw -------
	for rsc_name in ${rsc_list}
	do
		X50_JRAW_DELLB "${rsc_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X50_DECLUTTERING
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  X5.0 declutter functions.
#####################################################################
CONV_X50_DECLUTTERING()
{
TRACE "CONV_X50_DECLUTTERING"
	local ret_func=0

	if [ ${conf_ver_no1} -gt 4 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- #2259 jvm load balancer -------
	CONV_X50_JVMLB
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#----------------------------------------------------------------------
#	CONV_X53_REDFISH
#----------------------------------------------------------------------

#####################################################################
# name   : X53_BMC_SERVERS
# arg#1  : server name
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  set info for each bmc forcestop server
#  (in enabledelayedexpansion)
#####################################################################
X53_BMC_SERVERS()
{
TRACE "X53_BMC_SERVERS $1"
	local ret_func=0
	local server_name=$1

	#------- IP address -------
	CFGET_CFSET_VAL "forcestop/bmc/server@${server_name}/parameters/ip" "server@${server_name}/bmc/ip" del
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- user -------
	CFGET_CFSET_VAL "forcestop/bmc/server@${server_name}/parameters/user" "server@${server_name}/bmc/user" del
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- password -------
	CFGET_CFSET_VAL "forcestop/bmc/server@${server_name}/parameters/password" "server@${server_name}/bmc/password" del
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : CONV_X53_REDFISH
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert bmc forcestop
#####################################################################
CONV_X53_REDFISH()
{
TRACE "CONV_X53_REDFISH"

	local ret_func=0
	local server_list=""

	if [ ${conf_ver_no2} -ne 50 -a ${conf_ver_no2} -ne 51 -a ${conf_ver_no2} -ne 52 ]; then
		return ${RET_CMD_SUCCESS}
	fi

	#------- get bmc server list -------
	CFGET_ENUM /root/forcestop/bmc/server
	ret_func=$?
	case ${ret_func} in
	${RET_CMD_FAILED} )
		return ${RET_CMD_FAILED}
		;;
	${RET_CMD_NOTFOUND} )
		return ${RET_CMD_SUCCESS}
		;;
	esac
	server_list="${cfget_list}"

	#------- set bmc info for each bmc server -------
	for srv_name in ${server_list}
	do
		X53_BMC_SERVERS "${srv_name}"
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	done

	return ${RET_CMD_SUCCESS}
}

#####################################################################
# name   : SAVE_VERSION
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  save inner version
#####################################################################
SAVE_VERSION()
{
TRACE "SAVE_VERSION cfset_exec_flag=${cfset_exec_flag}"
	local ret_func=0
	local set_converted_flag=1

	if [ ${conf_ver_no6} -ge ${SAVED_INNER_VERSION} ];then
		return ${RET_CMD_SUCCESS}
	fi

	#------- check the value of converted flag -------
	CFGET_GET "/root/${XPATH_CFCONV_CONVERTED_FLAG}"
	ret_func=$?
	if [ ${ret_func} -eq ${RET_CMD_FAILED} ]; then
		return ${RET_CMD_FAILED}
	fi
	if [ ${ret_func} -eq ${RET_CMD_SUCCESS} ]; then
		if [ ${cfget_val} -eq 1 ]; then
			set_converted_flag=0
		fi
	fi

	#------- set converted flag -------
	if [ ${set_converted_flag} -eq 1 ]; then
		EXECUTE_CFSET add ${XPATH_CFCONV_CONVERTED_FLAG} ${cfset_exec_flag}
		ret_func=$?
		if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
			return ${RET_CMD_FAILED}
		fi
	fi

	EXECUTE_CFSET add trekking/savedinnerversion ${SAVED_INNER_VERSION}
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

TRACE "SAVE_VERSION save innerversion=${SAVED_INNER_VERSION}"
	return ${RET_CMD_SUCCESS}
}


#####################################################################
# name   : CONV_MAIN
# return : 
#  RET_CMD_SUCCESS : Succeeded.
#  RET_CMD_FAILED : Failed. 
# 
# description:
#  convert conf main
#####################################################################
CONV_MAIN()
{
TRACE "CONV_MAIN"
	local ret_conv=0

	#------- #2259 -------
	CONV_X335_TO_X40
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #710 -------
	CONV_X40_FOCNT
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #1956 -------
	CONV_X50_INITPASS
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #722 -------
	CONV_X50_HTTPWAUTH
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #828 -------
	CONV_X50_AWSDNS
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #520 -------
	CONV_X50_SERVICEW
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #519 -------
	CONV_X50_DECLUTTERING
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- #8391 -------
	CONV_X53_REDFISH
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

	#------- save innerversion -------
	SAVE_VERSION
	ret_conv=$?
	if [ ${ret_conv} -ne ${RET_CMD_SUCCESS} ]; then
		return ${RET_CMD_FAILED}
	fi

TRACE "CONV_MAIN return 0"
	return ${RET_CMD_SUCCESS}
}

####################################################################
#                                                                  #
# Main Routine                                                     #
#                                                                  #
####################################################################
#:MAIN_ROUTINE

#------- directory check -------#

curdir=`pwd`

etcpath="/opt/nec/clusterpro/etc"
if [ "${curdir}" = "${etcpath}" ]; then
	echo "${MSG_ERR_DIR}" 1>&2
	echo "${MSG_CMD_FAILED}" 1>&2
	exit ${RET_CMD_FAILED}
fi

#------- option check -------#
opt_opt=""

for i in $@
do
	CHK_OPT "$i"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		PRINT_USAGE
		exit ${RET_CMD_FAILED}
	fi
done

if [ "${opt_opt}" != "" -o "${opt_inpath}" = "" ]; then
	PRINT_USAGE
	exit ${RET_CMD_FAILED}
fi

#------- conf check -------#
CHK_CONF
ret_func=$?
if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
	echo "${MSG_CMD_FAILED}" 1>&2
	exit ${RET_CMD_FAILED}
fi

#------- convert conf -------
CONV_MAIN
ret_func=$?
if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
	if [ ${inpath_curdir_flag} -eq 1 ]; then
		RESTORE_CONF
	fi
	echo "${MSG_CMD_FAILED}" 1>&2
	exit ${RET_CMD_FAILED}
fi

#------- copy clp.conf to outputpath -------
if [ "${opt_outpath}" != "" ]; then
	COPY_CONF ./clp.conf "${opt_outpath}/clp.conf"
	ret_func=$?
	if [ ${ret_func} -ne ${RET_CMD_SUCCESS} ]; then
		echo "${MSG_CMD_FAILED}" 1>&2
		exit ${RET_CMD_FAILED}
	fi
	if [ ${backup_conf_flag} -eq 1 ]; then
		RESTORE_CONF
	else
		rm -f ./clp.conf > /dev/null 2>&1
	fi
fi

echo "${MSG_CMD_SUCCESS}"

set_password_msg=0
if [ ${init_password_ope} -eq 1 ]; then
	echo ""
	echo "${MSG_INIT_PASSWORD_OPE}"
	set_password_msg=1
fi
if [ ${init_password_ref} -eq 1 ]; then
	if [ ${set_password_msg} -eq 0 ]; then
		echo ""
	fi
	echo "${MSG_INIT_PASSWORD_REF}"
	set_password_msg=1
fi
if [ ${set_password_msg} -eq 1 ]; then
	echo "${MSG_SET_PASSWORD}"
fi

exit ${RET_CMD_SUCCESS}

